package de.duehl.basics.text.extract;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;
import de.duehl.basics.text.extract.tools.PrepareLines;
import de.duehl.basics.text.extract.tools.UnwantedWordsRemover;
import de.duehl.basics.text.sort.WordsSorter;

/**
 * Diese Klasse extrahiert brauchbare Worte aus einem Text.
 *
 * @version 1.01     2019-04-25
 * @author Christian Dühl
 */

public class WordExtractor {

    /** Der Text aus dem die Worte extrahiert werden sollen. */
    private final String text;

    /** Die Zeilen des Textes, ein Zwischenprodukt. */
    private List<String> lines;

    /** Die extrahierten Worte. */
    private List<String> words;

    /**
     * Konstruktor.
     *
     * @param text
     *            Der Text aus dem die Worte extrahiert werden sollen.
     */
    public WordExtractor(String text) {
        this.text = text;
        extractWords();
    }

    /** Extrahiert die brauchbaren Worte aus dem übergebenen Text. */
    private void extractWords() {
        prepareLines();
        splitLinesIntoWords();
        makeWordsDisjunkt();
        removeUnwantedWords();
        sortWords();
    }

    private void prepareLines() {
        PrepareLines prepare = new PrepareLines(text);
        lines = prepare.getLines();
    }

    private void splitLinesIntoWords() {
        words = new ArrayList<>();
        for (String line : lines) {
            List<String> wordOfLine = Text.splitByWhitespace(line.strip());
            words.addAll(wordOfLine);
        }
    }

    private void makeWordsDisjunkt() {
        words = CollectionsHelper.createDisjunctList(words);
    }

    private void removeUnwantedWords() {
        UnwantedWordsRemover remover = new UnwantedWordsRemover(words);
        words = remover.getCleanedWords();
    }

    private void sortWords() {
        WordsSorter sorter = new WordsSorter(words);
        sorter.sort();
    }

    /** Getter für die extrahierten Worte. */
    public List<String> getWords() {
        return words;
    }

}
