package de.duehl.swing.ui.components.elements;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.layout.VerticalLayout;

/**
 * Diese Klasse stellt eine Liste von Dingen mit einer Überschrift dar und bietet verschiedene
 * Methoden, deren Darstellung zu beeinflussen.
 *
 * @version 1.01     2017-03-29
 * @author Christian Dühl
 */

public class TitledList {

    private final JLabel titleLabel;
    private final List<String> valueList;
    private final List<JLabel> valueLabelList;
    private final JPanel panel;
    private final JPanel listPanel;

    public TitledList(String title) {
        this(title, new ArrayList<>());
    }

    public TitledList(String title, List<String> valueList) {
        this.valueList = valueList;

        titleLabel = new JLabel(title);
        valueLabelList = new ArrayList<>();
        fillValueLabelList();

        panel = new JPanel();
        listPanel = new JPanel();

        populatePanel();
    }

    private void fillValueLabelList() {
        for (String value : valueList) {
            JLabel label = new JLabel(value);
            valueLabelList.add(label);
        }
    }

    private void populatePanel() {
        panel.setLayout(new BorderLayout());

        panel.add(titleLabel, BorderLayout.NORTH);
        panel.add(createListPanel(), BorderLayout.CENTER);
    }

    private Component createListPanel() {
        listPanel.setLayout(new VerticalLayout(2, VerticalLayout.BOTH));
        GuiTools.createTitle(listPanel);

        for (JLabel valueLabel : valueLabelList) {
            listPanel.add(valueLabel);
        }

        return listPanel;
    }

    public void addValueToList(String value) {
        valueList.add(value);
        JLabel valueLabel = new JLabel(value);
        valueLabelList.add(valueLabel);
        listPanel.add(valueLabel);
        //listPanel.repaint();
        listPanel.validate();
        //listPanel.revalidate();
    }

    public void setPreferredTitleSize(Dimension preferredTitleSize) {
        titleLabel.setPreferredSize(preferredTitleSize);
    }

    public void setTitleColor(Color titleColor) {
        titleLabel.setForeground(titleColor);
    }

    public void useMonospacedFontForTheTitle() {
        GuiTools.setMonospacedFont(titleLabel);
    }

    public void addToTitleSize(int size) {
        GuiTools.biggerFont(titleLabel, size);
    }

    public void setPreferredListElementSize(Dimension preferredListElementSize) {
        for (JLabel valueLabel : valueLabelList) {
            valueLabel.setPreferredSize(preferredListElementSize);
        }
    }

    public void setListColor(Color listColor) {
        for (JLabel valueLabel : valueLabelList) {
            valueLabel.setForeground(listColor);
        }
    }

    public void useMonospacedFontForTheList() {
        for (JLabel valueLabel : valueLabelList) {
            GuiTools.setMonospacedFont(valueLabel);
        }
    }

    public Component getPanel() {
        return panel;
    }

}
