package de.duehl.swing.ui.elements.size;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Component;
import java.awt.Point;
import java.util.List;

import javax.swing.JButton;

/**
 * Diese Klasse testet den Dialog zur Auswahl einer Größe.
 *
 * @version 1.01     2021-03-26
 * @author Christian Dühl
 */

public class ChangeButtonWithSizeChooser {

    /** Position des Elements, über dem der SizeChooser angezeigt werden soll. */
    private final Point parentLocation;

    /** Beschreibungstext auf dem Button. */
    private final String sizeDescription;

    /** Anzahl Zeilen in der Anordnung der Zahlen. */
    private final int rows;

    /** Anzahl Spalten in der Anordnung der Zahlen. */
    private final int columns;

    /** Menge der erlaubten auswählbaren Werte. */
    private final List<Integer> possibleSizes;

    /** Gewählte Größe. */
    private int size;

    /** Der Button der den Dialog zum ändern der Größe aufruft. */
    private final JButton sizeChangeButton;

    /** Gibt an, ob auch andere Größen zulässig sind. */
    private boolean differentSizesAreOk;

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Elements, über dem der SizeChooser angezeigt werden soll.
     * @param sizeDescription
     *            Beschreibungstext auf dem Button.
     * @param rows
     *            Anzahl Zeilen in der Anordnung der Zahlen.
     * @param columns
     *            Anzahl Spalten in der Anordnung der Zahlen.
     * @param possibleSizes
     *            Menge der erlaubten auswählbaren Werte.
     */
    public ChangeButtonWithSizeChooser(Point parentLocation, String sizeDescription,
            int rows, int columns, List<Integer> possibleSizes) {
        this.parentLocation = parentLocation;
        this.sizeDescription = sizeDescription;
        this.rows = rows;
        this.columns = columns;
        this.possibleSizes = possibleSizes;

        size = possibleSizes.get(0);
        sizeChangeButton = new JButton();

        differentSizesAreOk = false;

        initSizeColorButton();
    }

    private void initSizeColorButton() {
        setSizeButtonText();
        sizeChangeButton.addActionListener(e -> selectSize());
    }

    private void selectSize() {
        String description = sizeDescription;
        SizeChooser chooser = new SizeChooser(size, description, rows, columns,
                possibleSizes, parentLocation, null);
        if (differentSizesAreOk) {
            chooser.diffenrentSizesAreOk();
        }
        chooser.setSize(size);
        chooser.setVisible(true);
        int newsize = chooser.getSize();
        if (newsize != SizeChooser.NO_VALUE_SELECTED) {
            size = newsize;
            setSizeButtonText();
        }
    }

    private void setSizeButtonText() {
        sizeChangeButton.setText(sizeDescription + " ändern, momentan " + size);
    }

    /** Erlaubt auch andere Größen im Texteingabefeld. */
    public void differentSizesAreOk() {
        differentSizesAreOk = true;
    }

    /** Getter für den Button der den Dialog zum ändern der Größe aufruft. */
    public Component getSizeChangeButton() {
        return sizeChangeButton;
    }

    /** Getter für die gewählte Größe. */
    public int getSize() {
        return size;
    }

    /** Setter für die gewählte Größe. */
    public void setSize(int size) {
        this.size = size;
        setSizeButtonText();
    }

}
