package de.duehl.swing.ui.text;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;

import javax.swing.JTextArea;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse zeigt Text an.
 *
 * Bei direkter Verwendung bitte prüfen, ob nicht der TextPanel / TextViewer / TextEditor die
 * bessere Wahl wäre.
 *
 * @version 1.01     2024-06-13
 * @author Christian Dühl
 */

public class TextComponent implements TextualComponent {

    /** Die TextArea, in der der Text angezeigt oder editiert wird. */
    private final JTextArea textComponent;

    /**
     * Konstruktor.
     *
     * Bei direkter Verwendung bitte prüfen, ob nicht der TextViewer / TextEditor die bessere Wahl
     * wäre.
     */
    public TextComponent() {
        textComponent = new JTextArea();

        initComponent();
    }

    private void initComponent() {
        textComponent.setWrapStyleWord(true);
        textComponent.setLineWrap(true);
        textComponent.setEditable(false);
    }

    /** Setzt den anzuzeigenden Text. */
    @Override
    public void setText(String text) {
        textComponent.setText(text);
    }

    /** Setzt die Position des Cursors. */
    @Override
    public void setCaretPosition(int position) {
        textComponent.setCaretPosition(position);
    }

    /** Gibt die Länge des Textes zurück. */
    @Override
    public int getTextLength() {
        return textComponent.getText().length();
    }

    /** Zeichnet den Editor neu. */
    @Override
    public void repaint() {
        textComponent.repaint();
        textComponent.validate();
    }

    /** Setzt die gewünschte Größe des Text-Elements. */
    @Override
    public void setPreferredSize(int width, int height) {
        textComponent.setPreferredSize(new Dimension(width, height));
    }

    /** Färbt die übergebene Komponente ein, falls ein Colorizer übergeben wurde. */
    @Override
    public void setColors(Colorizer colorizer) {
        if (colorizer != null) {
            colorizer.setColors(textComponent);
        }
    }

    /** Gibt die in die Oberfläche einzubauende Komponente zurück. */
    @Override
    public Component getComponent() {
        return textComponent;
    }

    /** Verwendet einen Font, der für Sourcecode und Daten geeignet ist. */
    public void useMonoscpacedText() {
        GuiTools.setMonospacedFont(textComponent, 14);
    }

    /** Legt die Größe der Schriftart fest. */
    public void setFontSize(int fontSize) {
        GuiTools.setFontSize(textComponent, fontSize);
    }

    /** Vergrößert (oder verkleinert bei negativer Größe) die Schriftgröße. */
    public void biggerFont(int addSize) {
        GuiTools.biggerFont(textComponent, addSize);
    }

    /** Legt fest, ob die Komponente bearbeitbar ist. Per Default ist sie dies nicht. */
    public void setEditable(boolean editable) {
        textComponent.setEditable(editable);
    }

    /** Schaltet das Umbrechen der Zeilen ab. */
    public void useNoWordWrap() {
        textComponent.setLineWrap(false);
    }

    /** Getter für den dargestellten Text. */
    public String getText() {
        return textComponent.getText();
    }

    /** Setzt die Hintergrundfarben für die Darstellung. */
    public void setBackground(Color backgroundColor) {
        textComponent.setOpaque(true);
        textComponent.setBackground(backgroundColor);
    }

    /**
     * Lässt die Tastenkombinationen Page-Up und Page-Down an die übergeordnete Komponente
     * weiterreichen.
     */
    public void ignorePageUpAndPageDown() {
        GuiTools.ignorePageUpAndPageDownInComponent(textComponent);
    }

    /**
     * Lässt die Tastenkombinationen Pfeiltaste nach oben und Pfeiltaste nach oben in dem Feld für
     * den Namen an die übergeordnete Komponente weiterreichen.
     */
    public void ignoreUpAndDown() {
        GuiTools.ignoreUpAndDownInComponent(textComponent);
    }

    /**
     * Lässt die Tastenkombinationen Ctrl-Pos1 und Ctrl-End in dem Feld für den Namen an die
     * übergeordnete Komponente weiterreichen.
     */
    public void ignoreCtrlPos1AndCtrlEnd() {
        GuiTools.ignoreCtrlPos1AndCtrlEndInComponent(textComponent);
    }

    /** Setzt die Anzahl Zeilen des Textfeldes. */
    public void setRows(int numberOfRows) {
        textComponent.setRows(numberOfRows);
    }

    /** Setzt die Anzahl Spalten des Textfeldes. */
    public void setColumns(int numberOfColumns) {
        textComponent.setColumns(numberOfColumns);
    }

}
