package de.duehl.swing.ui.text;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Component;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JButton;

import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse stellt den Dialog zum Bearbeiten eines Textes im Editor dar.
 *
 * Beim Beenden wird der bearbeitete Text weitergereicht.
 *
 * @version 1.01     2020-02-12
 * @author Christian Dühl
 */

public class TextEditor extends TextViewer {

    /** Objekt, das informiert wird, wenn der OK-Button gedrückt wird. */
    private final EditorOkReactor reactor;

    /** Der Button zum Beenden. */
    private JButton quitButton; // kann leider nicht final sein, weil er bei der Initialisierung
                                // vom TextViewer bereits gebraucht wird.

    /**
     * Konstruktor.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     */
    public TextEditor(String dialogTitle, EditorOkReactor reaktor) {
        this(dialogTitle, new Point(0, 0), null, reaktor);
    }

    /**
     * Konstruktor als Viewer.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param parentLocation
     *            Position des Rahmens des Aufrufers.
     */
    public TextEditor(String dialogTitle, Point parentLocation, EditorOkReactor reaktor) {
        this(dialogTitle, parentLocation, null, reaktor);
    }

    /**
     * Konstruktor als Editor.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param parentLocation
     *            Position des Rahmens des Aufrufers.
     * @param programImage
     *            Bild des Programms.
     * @param contents
     *            Inhalt der angezeigt werden soll.
     * @param reaktor
     *            Objekt, das informiert wird, wenn der OK-Button gedrückt wird.
     */
    public TextEditor(String dialogTitle, Point parentLocation, Image programImage,
            EditorOkReactor reaktor) {
        this(dialogTitle, parentLocation, null, programImage, reaktor);
    }

    /**
     * Konstruktor.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param parentLocation
     *            Position des Rahmens des Aufrufers.
     * @param colorizer
     *            Farbverwaltung für die Gui.
     * @param programImage
     *            Bild des Programms.
     * @param reaktor
     *            Objekt, das informiert wird, wenn der OK-Button gedrückt wird.
     */
    public TextEditor(String dialogTitle, Point parentLocation, Colorizer colorizer,
            Image programImage, EditorOkReactor reaktor) {
        super(dialogTitle, parentLocation, colorizer, programImage);
        this.reactor = reaktor;
    }

    @Override
    protected void initTextComponent() {
        super.initTextComponent();
        switchToEditable();
    }

    @Override
    protected void okPressed() {
        reactor.workOnText(getText());
        super.okPressed();
    }

    @Override
    protected Component createLeftButtonPart() {
        quitButton = new JButton("Abbruch");
        initQuitButton();
        return quitButton;
    }

    private void initQuitButton() {
        setColors(quitButton);
        quitButton.addActionListener(e -> closeDialog());
    }

    /** Setzt den angezeigten Text auf dem Button zum Abbruch oder Beenden. */
    public final void setQuitButtonText(String buttonText) {
        quitButton.setText(buttonText);
    }

    protected final Component getQuitButton() {
        return quitButton;
    }

    protected final EditorOkReactor getReactor() {
        return reactor;
    }
}
