package de.duehl.swing.ui.text;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.border.EmptyBorder;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse stellt den Dialog zum Anzeigen eines Textes im Editor dar.
 *
 * @version 1.01     2024-06-13
 * @author Christian Dühl
 */

public class TextPanel {

    /** Die Komponente mit dem Text, der angezeigt (oder im Editor editiert) wird. */
    private final TextComponent textComponent;

    /** Die Scrollbar des Anzeigefensters. */
    private final JScrollPane scroll;

    /** Das gescrollte Panel das hier aufgebaut wird. */
    private final JPanel panel;

    /** Konstruktor. */
    public TextPanel() {
        textComponent = new TextComponent();
        scroll = new JScrollPane(textComponent.getComponent());
        panel = new JPanel();
        initPanel();
    }

    /** Setzt die Farben, falls ein Colorizer übergeben wird. */
    public void setColors(Colorizer colorizer) {
        if (null != colorizer) {
            textComponent.setColors(colorizer);
            colorizer.setColors(scroll);
            colorizer.setColors(panel);
        }
    }

    protected void switchToEditable() {
        textComponent.setEditable(true);
    }

    /** Verwendet einen Font, der für Sourcecode und Daten geeignet ist. */
    public void useMonoscpacedText() {
        textComponent.useMonoscpacedText();
    }

    /** Legt die Größe der Schriftart fest. */
    public void setFontSize(int fontSize) {
        textComponent.setFontSize(fontSize);
    }

    /** Vergrößert (oder verkleinert bei negativer Größe) die Schriftgröße. */
    public void biggerFont(int addSize) {
        textComponent.biggerFont(addSize);
    }

    /** Schaltet das Umbrechen der Zeilen ab. */
    public void useNoWordWrap() {
        textComponent.useNoWordWrap();
    }

    /** Zeigt den übergebenen Text an und scrollt zum Ende. */
    public void setText(String text) {
        textComponent.setText(text);
        scrollScrollbarToMinimumLater();
    }

    /** Erzeugt den mittleren Bereich. */
    private void initPanel() {
        panel.setLayout(new BorderLayout());
        panel.setBorder(new EmptyBorder(10, 10, 10, 10));

        panel.add(scroll, BorderLayout.CENTER);
    }

    /** Setzt die gewünschte Größe des Text-Elements. */
    public void setPreferredSize(int width, int height) {
        scroll.setPreferredSize(new Dimension(width, height));
    }

    /** Legt fest, ob die Komponente bearbeitbar ist. Per Default ist sie dies nicht. */
    public void setEditable(boolean editable) {
        textComponent.setEditable(editable);
    }

    /** Getter für die einzufügende Komponente. */
    public Component getComponent() {
        return scroll;
    }

    /** Getter für den dargestellten Text. */
    public String getText() {
        return textComponent.getText();
    }

    /**
     * Setzt eine Scrollleiste auf den Minimalwert. Passiert im gleichen Thread.
     *
     * @param scroll
     *            Zu bearbeitende Scrollleiste.
     */
    public void scrollScrollbarToMinimumLater() {
        GuiTools.scrollScrollbarToMinimumLater(scroll);
    }

    /**
     * Setzt eine Scrollleiste auf den Maximalwert. Passiert im gleichen Thread.
     *
     * @param scroll
     *            Zu bearbeitende Scrollleiste.
     */
    public void scrollScrollbarToMaximumLater() {
        GuiTools.scrollScrollbarToMaximumLater(scroll);
    }

    /**
     * Zeigt in einer Scrollleiste den vorherigen Abschnitt an (etwa für PAGE-UP).
     *
     * Passiert im gleichen Thread.
     *
     * @param scroll
     *            Zu bearbeitende Scrollleiste.
     */
   public void scrollScrollbarToPreviousSectionLater() {
        GuiTools.scrollScrollbarToPreviousSectionLater(scroll);
    }

   /**
    * Zeigt in einer Scrollleiste den nächsten Abschnitt an (etwa für PAGE-DOWN).
    *
    * Passiert im gleichen Thread.
    *
    * @param scroll
    *            Zu bearbeitende Scrollleiste.
    */
    public void scrollScrollbarToNextSectionLater() {
        GuiTools.scrollScrollbarToNextSectionLater(scroll);
    }

    /** Setzt die Position des Cursors. */
    public void setCaretPosition(int position) {
        textComponent.setCaretPosition(position);
    }

    /** Zeichnet den Editor neu. */
    public void repaint() {
        textComponent.repaint();
        scroll.validate();
    }

    /** Setzt die Hintergrundfarben für die Darstellung. */
    public void setBackground(Color backgroundColor) {
        textComponent.setBackground(backgroundColor);
    }

    /**
     * Lässt die Tastenkombinationen Page-Up und Page-Down an die übergeordnete Komponente
     * weiterreichen.
     */
    public void ignorePageUpAndPageDown() {
        textComponent.ignorePageUpAndPageDown();
    }

    /**
     * Lässt die Tastenkombinationen Pfeiltaste nach oben und Pfeiltaste nach oben in dem Feld für
     * den Namen an die übergeordnete Komponente weiterreichen.
     */
    public void ignoreUpAndDown() {
        textComponent.ignoreUpAndDown();
    }

    /**
     * Lässt die Tastenkombinationen Ctrl-Pos1 und Ctrl-End in dem Feld für den Namen an die
     * übergeordnete Komponente weiterreichen.
     */
    public void ignoreCtrlPos1AndCtrlEnd() {
        textComponent.ignoreCtrlPos1AndCtrlEnd();
    }

    /** Setzt die Anzahl Zeilen des Textfeldes. */
    public void setRows(int numberOfRows) {
        textComponent.setRows(numberOfRows);
    }

    /** Setzt die Anzahl Spalten des Textfeldes. */
    public void setColumns(int numberOfColumns) {
        textComponent.setColumns(numberOfColumns);
    }

}
