package de.duehl.vocabulary.japanese.startup.logic;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.basics.datetime.time.watch.StopWatch;
import de.duehl.basics.io.FileHelper;
import de.duehl.basics.logic.ErrorHandler;
import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;
import de.duehl.swing.ui.text.TextViewer;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalVocableData;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.common.persistence.SessionManager;
import de.duehl.vocabulary.japanese.data.KanjiSet;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.io.AllKanjiSetsReader;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;
import de.duehl.vocabulary.japanese.logic.internal.RealInternalDataRequester;
import de.duehl.vocabulary.japanese.logic.ownlists.KeyToVocableMapCreator;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.logic.symbol.kana.internal.RealInternalKanaDataRequester;
import de.duehl.vocabulary.japanese.logic.symbol.kana.internal.io.InternalKanaDataReader;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.io.InternalKanjiDataReader;
import de.duehl.vocabulary.japanese.logic.translation.GermanToJapaneseTranslation;
import de.duehl.vocabulary.japanese.logic.translation.JapaneseToGermanTranslation;
import de.duehl.vocabulary.japanese.logic.wrongtested.WrongTestedVocables;
import de.duehl.vocabulary.japanese.startup.ui.MessageAppender;
import de.duehl.vocabulary.japanese.startup.ui.SplashScreen;
import de.duehl.vocabulary.japanese.tools.VocabularyTools;
import de.duehl.vocabulary.japanese.ui.VocabularyTrainerGui;

/**
 * Diese Klasse lädt beim Starten des Vokabeltrainers die Vokabulare ein, sucht die Sound-Files und
 * stellt die Verbindung zu den benutzerabhängigen, internen Daten jeder Variablen her.
 *
 * Außerdem werden Datenstrukturen aufgebaut, die vom Vokabel-Trainer benötigt werden.
 *
 * @version 1.01     2025-06-12
 * @author Christian Dühl
 */

public class StartupLoader implements MessageAppender {

    /**
     * Die Extension für den Dateien mit den benutzerabhängigen, internen Daten zu jeder Vokabel.
     */
    private static final String INTERNAL_VARIABLE_DATA_EXTENSION = ".ivd";

    public static final String INTERNAL_DATA_DIRECTORY = FileHelper.concatPathes(
            SessionManager.VOCABLE_TRAINER_DIRECTORY, "internal_vocabulary_data");


    /** Die grafische Oberfläche. */
    private final VocabularyTrainerGui gui;

    /** Die Programmoptionen. */
    private final Options options;

    /** Die Liste mit den bekannten Vokabularien. */
    private List<Vocabulary> vocabularies;

    /** Die Liste aller Schlüssel der eingelesenen internen Daten. */
    private List<String> allKeysFromReadInternalData;

    /**
     * Die Liste der Schlüssel von eingelesenen internen Daten, welche keiner Vokabel zugeordnet
     * werden konnten.
     */
    private List<String> unusedKeysFromReadInternalData;

    /** Die Liste der Schlüssel die aus einer Vokabel erzeugt wurden. */
    private List<String> seenKeysFromVocables;

    /** Die Liste der Schlüssel die identisch aus mehr als einer Vokabel erzeugt wurden. */
    private List<String> multipleCreatedKeysFromVocables;

    /** Das Verzeichnis der internen Daten zu einer Vokabel nach ihrem Schlüssel. */
    private Map<String, InternalAdditionalVocableData> key2InternalDataMap;

    /** Das Verzeichnis der internen Daten zu einer Vokabel nach der zugehörigen Variablen. */
    private Map<Vocable, InternalAdditionalVocableData> vocable2InternalDataMap;

    /** Das Objekt das zu Vokabeln die internen Daten abfragt. */
    private InternalDataRequester internalDataRequester;

    /**
     * Die Nachricht mit der Anzahl Vokabeln und Vokabularien, die nach dem Laden in der
     * Statuszeile der Oberfläche angezeigt werden soll.
     */
    private String loadUpMessage;

    /** Zählt, wie viele interne Daten neu angelegt wurden. */
    private int newInternalDataCreationCounter;

    /** Das Objekt das sich um die Übersetzung von Deutsch in Japanisch kümmert. */
    private GermanToJapaneseTranslation germanToJapaneseTranslation;

    /**
     * Das Objekt das sich um die Übersetzung von Japanisch in Deutsch kümmert (für Abfragen die
     * nur Kana, aber nicht Kanji anzeigen).
     */
    private JapaneseToGermanTranslation japaneseToGermanTranslation;

    /** Die Verwaltung der eigenen Vokabellisten. */
    private OwnLists ownLists;

    /**
     * Die Verwaltung der beiden automatisch gepflegten Listen mit falsch abgefragten Vokabeln aus
     * Gruppen und anderen Vokabularien.
     */
    private WrongTestedVocables wrongTestedVocables;

    /** Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen kann. */
    private InternalKanjiDataRequester internalKanjiDataRequester;

    /** Die Liste der benutzerdefinierten Kanji-Mengen. */
    private List<KanjiSet> kanjiSets;

    /** Das Objekt das zu einem Kana die internen, benutzerabhängigen Daten abrufen kann. */
    private RealInternalKanaDataRequester internalKanaDataRequester;

    /** Die grafische Oberfläche beim Start in der die Meldungen angezeigt werden. */
    private final SplashScreen splashScreen;

    /** Misst die Laufzeit. */
    private StopWatch watch;

    /**
     * Konstruktor.
     *
     * @param gui
     *            Die grafische Oberfläche.
     * @param options
     *            Die Programmoptionen.
     */
    public StartupLoader(VocabularyTrainerGui gui, Options options) {
        this.gui = gui;
        this.options = options;
        splashScreen = new SplashScreen(gui.getProgramImage());
    }

    /** Führt das Laden, Suchen und Verknüpfen durch. */
    public void load() {
        try {
            tryToLoad();
        }
        catch (Exception exception) {
            ErrorHandler errorHandler = gui.createErrorHandler();
            errorHandler.error("Fehler beim Startup: " + exception.getMessage(), exception);
            closeSplashScreen();
            gui.closeDialog();
        }
    }

    private void tryToLoad() {
        init();
        showSplashScreen();
        startup();
        closeSplashScreen();
    }

    private void init() {
        watch = new StopWatch();
    }

    private void showSplashScreen() {
        splashScreen.setVisible(true);
    }

    private void startup() {
        readVocabularies();
        readInternalVocableData();
        joinVocablesWithInternalData();
        createInternalDataRequester();
        storeDateOfOldestVocableInEachVocabulary();
        createJapaneseToGermanTranslation();
        createGermanToJapaneseTranslation();
        initOwnListsAndReadWrongTestedVocableLists();
        checkDoubleKaniji();
        loadInternalKanjiData();
        loadKanjiSets();
        loadInternalKanaData();
        checkDoubleTranslationAtSameVocabel();
        checkEOrUInVocableFieldAusspracheIfWanted();
    }

    private void readVocabularies() {
        appendMessage("Lese Vokabularien ...");
        VocabularyAndSoundFileFinder finder = new VocabularyAndSoundFileFinder(options,
                (MessageAppender) this);
        finder.find();
        vocabularies = finder.getVocabularies();
        loadUpMessage = finder.getLoadUpMessage();
    }

    /** Fügt eine Nachricht hinzu. */
    @Override
    public void appendMessage(String message) {
        String realMessage;
        if (options.isShowTimestampsInStartup()) {
            if (message.isBlank()) {
                realMessage = message;
            }
            else {
                String timeIntro = watch.getTime() + " - ";

                if (message.startsWith(" ")) {
                    int index = Text.findIndexOfFirstNonSpace(message);
                    if (index == -1) {
                        throw new RuntimeException("Kann nicht sein, die Nachricht beginnt mit "
                                + "einem Leerzeichen, ist nicht blank, aber ich finde nicht den "
                                + "Index des ersten nicht Leerzeichens.");
                    }
                    String spaces = message.substring(0, index);
                    String rest = message.substring(index);
                    realMessage = spaces + timeIntro + rest;
                }
                else {
                    realMessage = timeIntro + message;
                }
            }
        }
        else {
            realMessage = message;
        }
        splashScreen.appendMessage(realMessage);
    }

    /** Lädt alle bereits bekannten internen Datensätze zu Vokabeln. */
    private void readInternalVocableData() {
        appendMessage("Lese interne benutzerabhängige Daten zu allen Vokabeln ...");
        FileHelper.createDirectoryIfNotExists(INTERNAL_DATA_DIRECTORY);

        List<String> internalFiles = FileHelper.findFilesInMainDirectoryNio2WithExtensions(
                INTERNAL_DATA_DIRECTORY, INTERNAL_VARIABLE_DATA_EXTENSION);
        appendMessage(NumberString.taupu(internalFiles.size()) + " interne Datensätze gefunden.");
        appendMessage("Lese interne Datensätze ein ...");

        List<InternalAdditionalVocableData> internalAdditionalVocableDatas = new ArrayList<>();
        for (String filename : internalFiles) {
            InternalAdditionalVocableData data = InternalAdditionalVocableData.load(filename);
            internalAdditionalVocableDatas.add(data);
        }
        appendMessage(NumberString.taupu(internalAdditionalVocableDatas.size())
                + " interne Datensätze eingelesen.");

        key2InternalDataMap = new HashMap<>();
        allKeysFromReadInternalData = new ArrayList<>();
        for (InternalAdditionalVocableData data : internalAdditionalVocableDatas) {
            String key = data.getKey();
            if (key2InternalDataMap.containsKey(key) || allKeysFromReadInternalData.contains(key)) {
                throw new RuntimeException("Der Schlüssel '" + key + "' kommt mehrfach vor!");
            }

            key2InternalDataMap.put(key, data);
            allKeysFromReadInternalData.add(key);
        }
        appendMessage("Verzeichnis mit " + NumberString.taupu(key2InternalDataMap.size())
                + " Verweisen von Schlüssel auf interne Datensätze aufgebaut.");
    }

    /**
     * Verbindet die Vokabeln mit den internen Daten.
     *
     * Dabei wird darauf auf die folgenden Umstände geachtet:
     *
     *     1) Ob dabei Schlüssel aus den eingelesenen internen Daten übrig bleiben, was darauf
     *        hindeutet, dass man an den Vokabeln etwas geändert hat und diese internen Daten
     *        entfernt werden sollten, ggf., nachdem man Daten daraus in die neu erzeugten
     *        internen Daten übernommen hat.
     *        Die erfolgt über die Liste
     *            unusedKeysFromReadInternalData.
     *     2) Es wird geprüft, dass keine zwei verschiedenen Vokabeln zum gleichen Schlüssel
     *        führen.
     *        Durch die Prüfung in
     *            VocabularyAndSoundFileFinder#
     *                checkIfVocablesAreUniqueWithKeyFromKanaAndFirstTranslations()
     *        wissen wir zwar, dass die Kombination von Kana und der ersten Übersetzung eindeutig
     *        ist. Auch der von
     *            VocabularyTools#createVocableKey()
     *        erzeugte Schlüssel basiert auf diesen beiden Daten, allerdings werden am Schlüssel
     *        einige Ersetzungen vorgenommen, so dass man nicht mehr sicher sein kann, dass der
     *        Schlüssel wirklich eindeutig ist.
     *        Also wird geprüft, dass es nicht vorkommt, dass die Schlüssel zweier
     *        unterschiedlicher Vokabeln dennoch gleich sind.
     *        Die erfolgt über die Listen
     *            seenKeysFromVocables und
     *            multipleCreatedKeysFromVocables.
     */
    private void joinVocablesWithInternalData() {
        appendMessage("Verbinde die Vokabeln mit den internen benutzerabhängigen Daten ...");
        vocable2InternalDataMap = new HashMap<>();
        newInternalDataCreationCounter = 0;

        unusedKeysFromReadInternalData = new ArrayList<>();
        unusedKeysFromReadInternalData.addAll(allKeysFromReadInternalData);

        seenKeysFromVocables = new ArrayList<>();
        multipleCreatedKeysFromVocables = new ArrayList<>();

        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable vocable : vocabulary.getVocables()) {
                joinVocableInternalData(vocable);
            }
        }

        appendMessage("Verzeichnis mit " + NumberString.taupu(vocable2InternalDataMap.size())
                + " Verweisen von Vokabeln auf interne Datensätze aufgebaut.");
        appendMessage(NumberString.taupu(newInternalDataCreationCounter)
                + " neue interne Daten angelegt.");

        if (unusedKeysFromReadInternalData.isEmpty()) {
            appendMessage("Alle interne Daten wurden einer Vokabel zugeordnet.");
        }
        else {
            appendMessage("Es gibt " + NumberString.taupu(unusedKeysFromReadInternalData.size())
                    + " interne Daten, welcher keiner Vokabel zugeordnet wurden:");
            for (String unusedKey : unusedKeysFromReadInternalData) {
                appendMessage("    " + unusedKey);
            }
        }

        if (multipleCreatedKeysFromVocables.isEmpty()) {
            appendMessage("Alle Vokabeln erzeugen unterschiedliche Schlüssel.");
        }
        else {
            appendMessage("Es gibt " + NumberString.taupu(multipleCreatedKeysFromVocables.size())
                    + " Schlüssel, welche zu mehr als einer Vokabel gehören");
            for (String multipleCreatedKey : multipleCreatedKeysFromVocables) {
                appendMessage("    " + multipleCreatedKey);
            }

            // Kann ja nur bei mir auftreten, daher:
            System.err.println("Daher wird der Start abgebrochen.");
            System.exit(1);
        }
    }

    /** Verbindet die übergebene Vokabel über den Schlüssel mit ihren internen Daten. */
    private void joinVocableInternalData(Vocable vocable) {
        String key = VocabularyTools.createVocableKey(vocable);
        unusedKeysFromReadInternalData.remove(key); // dieser interne Schlüssel wird verwendet!

        if (seenKeysFromVocables.contains(key)) {
            if (!multipleCreatedKeysFromVocables.contains(key)) {
                multipleCreatedKeysFromVocables.add(key);
            }
        }
        else {
            seenKeysFromVocables.add(key);
        }

        InternalAdditionalVocableData data;
        if (key2InternalDataMap.containsKey(key)) {
            data = key2InternalDataMap.get(key);
        }
        else {
            data = new InternalAdditionalVocableData();
            data.setKey(key);
            String internalDataFilename = FileHelper.concatPathes(INTERNAL_DATA_DIRECTORY,
                    key + INTERNAL_VARIABLE_DATA_EXTENSION);
            data.setFilename(internalDataFilename);
            data.setFirstSeenDate(new ImmutualDate()); // heute

            appendMessage("neu: " + internalDataFilename);
            ++newInternalDataCreationCounter;
            data.save();
        }

        vocable2InternalDataMap.put(vocable, data);
    }

    private void createInternalDataRequester() {
        internalDataRequester = new RealInternalDataRequester(vocable2InternalDataMap);
    }

    private void storeDateOfOldestVocableInEachVocabulary() {
        appendMessage("Trage Datum der ältesten Vokabel in allen Vokabularien ein ...");
        for (Vocabulary vocabulary : vocabularies) {
            storeDateOfOldestVocableInVocabulary(vocabulary);
        }
    }

    private void storeDateOfOldestVocableInVocabulary(Vocabulary vocabulary) {
        ImmutualDate oldestDate = new ImmutualDate();

        for (Vocable vocable : vocabulary.getVocables()) {
            InternalAdditionalVocableData data = vocable2InternalDataMap.get(vocable);
            ImmutualDate vocableFirstSeenDate = data.getFirstSeenDate();
            if (!vocableFirstSeenDate.equals(InternalAdditionalVocableData.NOT_SEEN_DATE)
                    && vocableFirstSeenDate.before(oldestDate)) {
                oldestDate = vocableFirstSeenDate;
            }
        }

        vocabulary.setFirstSeenDate(oldestDate);
    }

    private void createJapaneseToGermanTranslation() {
        appendMessage("Erzeuge Datenstrukturen für die Zuordnung von Kana und Kanji zu Vokabeln "
                + "sowie mehrdeutige Zuordnungen von Kana ohne Kanji zu Vokabeln für die "
                + "Übersetzung von Japanisch in Deutsch ...");
        japaneseToGermanTranslation = new JapaneseToGermanTranslation(vocabularies);
    }

    private void createGermanToJapaneseTranslation() {
        if (options.isCreateGermanJapaneseTranslationAtStartup()) {
            appendMessage("Erzeuge Datenstrukturen für mehrdeutige Übersetzungen von Deutsch in "
                    + "Japanisch ...");
            germanToJapaneseTranslation = new GermanToJapaneseTranslation(vocabularies);
        }
        else {
            appendMessage("Die Datenstrukturen für mehrdeutige Übersetzungen von Deutsch in "
                    + "Japanisch werden noch nicht erzeugt, sondern erst bei der ersten "
                    + "Verwendung.");
        }
    }

    private void initOwnListsAndReadWrongTestedVocableLists() {
        appendMessage("Erzeuge Verzeichnis Schlüssel zu Vokabel ...");
        KeyToVocableMapCreator creator = new KeyToVocableMapCreator(vocabularies,
                internalDataRequester);
        creator.create();
        Map<String, Vocable> keyToVocable = creator.getKeyToVocable();

        appendMessage("Erzeuge Verwaltung der eigene Listen ...");
        ownLists = new OwnLists(vocabularies, keyToVocable, internalDataRequester, options, gui);

        appendMessage("Erzeuge Verwaltung der automatisch gepflegten Listen der falsch übersetzten "
                + "Vokabeln ...");
        wrongTestedVocables = new WrongTestedVocables(vocabularies, keyToVocable,
                internalDataRequester);
    }

    /**
     * Überprüft, ob es bei den Kanji jedes Kanji-Zeichen nur einmal gibt.
     *
     * Es war einmal vorgekommen, dass ich ein Kanji unter zwei unterschiedlichen Namen zweimal in
     * der Kanji-Enum-Klasse hatte (HIMMEL und HIMMEL_PARADIES).
     */
    private void checkDoubleKaniji() {
        List<String> seenKanjiCharacters = new ArrayList<>();

        for (Kanji kanji : Kanji.getAllKanjiAsList()) {
            String character = kanji.getCharacter();
            if (seenKanjiCharacters.contains(character)) {
                reportAboutFatalErrorsAndExit("Zum Kanji '" + character + "' sind mehr als ein "
                        + "Enum-Element in der Klasse Kanji hinterlegt.");
            }
            seenKanjiCharacters.add(character);
        }
    }

    private void loadInternalKanjiData() {
        InternalKanjiDataReader reader = new InternalKanjiDataReader((MessageAppender) this);
        reader.read();
        internalKanjiDataRequester = reader.getRequester();
    }

    private void loadKanjiSets() {
        appendMessage("Lade die benutzerdefinierten Kanji-Mengen ...");
        kanjiSets = AllKanjiSetsReader.read();
    }

    private void loadInternalKanaData() {
        InternalKanaDataReader reader = new InternalKanaDataReader((MessageAppender) this);
        reader.read();
        internalKanaDataRequester = reader.getRequester();
    }

    /** Überprüft jede Vokabel, ob es bei ihren Übersetzungen doppelte gibt. */
    private void checkDoubleTranslationAtSameVocabel() {
        List<String> errors = new ArrayList<>();

        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable vocable : vocabulary.getVocables()) {
                String error = checkDoubleTranslationAtSameVocabel(vocable);
                if (!error.isBlank()) {
                    errors.add(error + " (im Vokabular '" + vocabulary.getDescription() + "').");
                }
            }
        }

        if (!errors.isEmpty()) {
            reportAboutFatalErrorsAndExit(Text.joinWithLineBreak(errors));
        }
    }

    /** Überprüft die übergebene Vokabel, ob es bei ihren Übersetzungen doppelte gibt. */
    private String checkDoubleTranslationAtSameVocabel(Vocable vocable) {
        List<String> notDisjunctTexts = CollectionsHelper.getNotDisjunktTexts(
                vocable.getTranslations());
        if (notDisjunctTexts.isEmpty()) {
            return "";
        }
        else {
            return "Zur Vokabel '" + vocable.getKanjiKanaRomajiWithJapaneseBraces()
                    + "' gibt es mehrfach vorkommende Übersetzungen: '"
                    + Text.join("', '", notDisjunctTexts) + "'";
        }
    }

    /**
     * Überprüft jede Vokabel, ob es in ihrer Aussprache ein "e" oder ein "u" gibt, falls
     * gewünscht.
     */
    private void checkEOrUInVocableFieldAusspracheIfWanted() {
        if (options.isInformAboutEOrUInVocableFieldAusspracheAtStartup()) {
            checkEOrUInVocableFieldAussprache();
        }
    }

    /** Überprüft jede Vokabel, ob es in ihrer Aussprache ein "e" oder ein "u" gibt. */
    private void checkEOrUInVocableFieldAussprache() {
        List<String> errors = new ArrayList<>();

        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable vocable : vocabulary.getVocables()) {
                String aussprache = vocable.getPronunciation();
                if (aussprache.contains("e") || aussprache.contains("u")) {
                    String error = "Die Vokabel '" + vocable.getKanjiKanaRomajiWithJapaneseBraces()
                        + "' (im Vokabular '" + vocabulary.getDescription() + "' enthält 'e' "
                                + "oder 'u' in ihrer Aussprache: '" + aussprache + "'.";
                    errors.add(error);
                }
            }
        }

        if (!errors.isEmpty()) {
            reportAboutFatalErrorsAndExit(Text.joinWithLineBreak(errors));
        }
    }

    private void closeSplashScreen() {
        splashScreen.closeDialog();
    }

    private void reportAboutFatalErrorsAndExit(String errorMessage) {
        String message = "Es ist ein Fehler aufgetreten, der den Start verhindert:\n"
                + errorMessage;
        splashScreen.appendMessage(message);
        closeSplashScreen();
        System.err.println(message);
        showFatalErrorMessagesAsDialog(message);
        System.exit(1);
    }

    private void showFatalErrorMessagesAsDialog(String message) {
        TextViewer viewer = new TextViewer("Fatale Fehlermedlung");
        viewer.setText(message);
        viewer.setVisible(true);
    }

    /**
     * Getter für die Nachricht mit der Anzahl Vokabeln und Vokabularien, die nach dem Laden in der
     * Statuszeile der Oberfläche angezeigt werden soll.
     */
    public String getLoadUpMessage() {
        return loadUpMessage;
    }

    /** Getter für die Liste mit den bekannten Vokabularien. */
    public List<Vocabulary> getVocabularies() {
        return vocabularies;
    }

    /** Getter für das Objekt, das zu Vokabeln die internen Daten abfragt. */
    public InternalDataRequester getInternalDataRequester() {
        return internalDataRequester;
    }

    /**
     * Getter für das Objekt das sich um die Übersetzung von Japanisch in Deutsch kümmert (für
     * Abfragen die nur Kana, aber nicht Kanji anzeigen).
     */
    public JapaneseToGermanTranslation getJapaneseToGermanTranslation() {
        return japaneseToGermanTranslation;
    }

    /** Getter für das Objekt das sich um die Übersetzung von Deutsch in Japanisch kümmert. */
    public GermanToJapaneseTranslation getGermanToJapaneseTranslation() {
        return germanToJapaneseTranslation;
    }

    /** Getter für die Verwaltung der eigenen Vokabellisten. */
    public OwnLists getOwnLists() {
        return ownLists;
    }

    /**
     * Getter für die Verwaltung der beiden automatisch gepflegten Listen mit falsch abgefragten
     * Vokabeln aus Gruppen und anderen Vokabularien.
     */
    public WrongTestedVocables getWrongTestedVocables() {
        return wrongTestedVocables;
    }

    /**
     * Getter für das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     * kann.
     */
    public InternalKanjiDataRequester getInternalKanjiDataRequester() {
        return internalKanjiDataRequester;
    }

    /** Getter für die Liste der benutzerdefinierten Kanji-Mengen. */
    public List<KanjiSet> getKanjiSets() {
        return kanjiSets;
    }

    /**
     * Getter für das Objekt das zu einem Kana die internen, benutzerabhängigen Daten abrufen kann.
     */
    public RealInternalKanaDataRequester getInternalKanaDataRequester() {
        return internalKanaDataRequester;
    }

    /** Getter für den Text, der beim Starten des Vokabeltrainers angezeigt wurde. */
    public String getStartUpLog() {
        return splashScreen.getSplashText();
    }

}
