package de.duehl.vocabulary.japanese.ui.components.bars;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Point;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.ui.resources.IconDefinitions;
import de.duehl.vocabulary.japanese.data.KanjiSet;
import de.duehl.vocabulary.japanese.logic.VocabularyTrainerLogic;
import de.duehl.vocabulary.japanese.ui.data.MessageSetter;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.KanjiTableDialog;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjiset.KanjiSetBarsRefresher;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjiset.KanjiSetEditor;
import de.duehl.vocabulary.japanese.ui.tools.VocabularyTrainerUiTools;

/**
 * Diese Klasse stellt eine Kanji-Mengen im Dialog zur Verwaltung der Kanji-Mengen
 * (KanjiSetManagementDialog) dar.
 *
 * Man kann die Kanji-Menge bearbeiten oder löschen.
 *
 * Sortiert wird hier nicht!
 * So viele Kanji-Mengen braucht man ja gar nicht, dass das nötig wäre.
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

public class KanjiSetBar {

    /** Die benutzerdefinierte Kanji-Menge. */
    private final KanjiSet kanjiSet;

    /** Kann die Bars mit den Kanji-Mengen neu zeichnen. */
    private final KanjiSetBarsRefresher kanjiSetBarsRefresher;

    /** Die Logik des Vokabeltrainers. */
    private final VocabularyTrainerLogic logic;

    /** Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen kann. */
    private final LongTimeProcessInformer informer;

    /** Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann. */
    private final MessageSetter messageSetter;

    /** Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird. */
    private final Point parentLocation;

    /** Das anzuzeigende ProgrammIcon. */
    private final Image programImage;

    /** Die Liste mit den nicht erlaubten Namen für das Umbenennen dieser Kanji-Menge. */
    private final List<String> notAllowedNames;

    /** Das Panel auf dem die Bar mit der Kanji-Menge dargestellt wird. */
    private final JPanel panel;

    /**
     * Konstruktor.
     *
     * @param kanjiSet
     *            Die benutzerdefinierte Kanji-Menge.
     * @param allNames
     *            Die Liste mit den Namen aller Kanji-Mengen.
     * @param kanjiSetBarsRefresher
     *            Kann die Bars mit den Kanji-Mengen neu zeichnen.
     * @param logic
     *            Die Logik des Vokabeltrainers.
     * @param informer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     * @param messageSetter
     *            Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das anzuzeigende ProgrammIcon.
     */
    public KanjiSetBar(KanjiSet kanjiSet, List<String> allNames,
            KanjiSetBarsRefresher kanjiSetBarsRefresher, VocabularyTrainerLogic logic,
            LongTimeProcessInformer informer, MessageSetter messageSetter, Point parentLocation,
            Image programImage) {
        this.kanjiSet = kanjiSet;
        this.kanjiSetBarsRefresher = kanjiSetBarsRefresher;
        this.logic = logic;
        this.informer = informer;
        this.messageSetter = messageSetter;
        this.parentLocation = parentLocation;
        this.programImage = programImage;

        notAllowedNames = new ArrayList<>();
        for (String name : allNames) {
            if (!name.equals(kanjiSet.getName())) {
                notAllowedNames.add(name);
            }
        }

        panel = new JPanel();

        createBarUi();
    }

    private void createBarUi() {
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        panel.add(createNameLabel(), BorderLayout.CENTER);
        panel.add(createButtonsPart(), BorderLayout.EAST);
    }

    private Component createNameLabel() {
        JLabel label = new JLabel(kanjiSet.getName());
        GuiTools.biggerFont(label, 10);
        return label;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(1, 0, 2, 2));

        panel.add(createViewAsListButon());
        panel.add(createEditButon());
        panel.add(createDeleteButon());

        return panel;
    }

    private Component createViewAsListButon() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.SHOW_VOCABULARY_LIST, e -> viewKanjiSetAsKanjiList(),
                "Vokabeln anzeigen");
        if (kanjiSet.getSet().isEmpty()) {
            button.setEnabled(false);
        }
        return button;
    }

    private void viewKanjiSetAsKanjiList() {
        String title = "Alle Kanji aus der Menge '" + kanjiSet.getName() + "'";
        KanjiTableDialog dialog = new KanjiTableDialog(kanjiSet.getSet(), logic.getOptions(),
                logic.getInternalDataRequester(), logic.getVocabularies(),
                logic.getInternalKanjiDataRequester(), informer, logic.getOwnLists(), title,
                messageSetter, parentLocation, programImage);
        dialog.setVisible(true);
    }

    private Component createEditButon() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.EDIT_OWN_LIST,
                e -> edit(),
                "Kanji-Menge bearbeiten");
        return button;
    }

    private void edit() {
        KanjiSetEditor editor = new KanjiSetEditor(kanjiSet, notAllowedNames, logic,
                panel.getLocation(), programImage);
        editor.setVisible(true);
        kanjiSetBarsRefresher.refreshKanjiSetBars();
    }

    private Component createDeleteButon() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.DELETE_OWN_LIST,
                e -> delete(),
                "Kanji-Menge löschen");
        button.setForeground(Color.RED);
        return button;
    }

    private void delete() {
        String name = kanjiSet.getName();
        String title = "Wirklich löschen?";
        String message = "Soll die Kanji-Menge '" + name + "' wirklich gelöscht werden?";
        boolean delete = GuiTools.askUser(panel, title, message);
        if (delete) {
            kanjiSetBarsRefresher.deleteKanjiSetFromListAndFileFromDisk(kanjiSet);
            kanjiSetBarsRefresher.refreshKanjiSetBars();
        }
    }

    /** Getter für das Panel auf dem die Bar mit der Kanji-Menge dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

}
