package de.duehl.vocabulary.japanese.ui.dialog.detail.related;

import java.awt.Image;
import java.awt.Point;
import java.util.List;

import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.grammar.VerbSearchWords;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.ui.data.MessageSetter;
import de.duehl.vocabulary.japanese.ui.dialog.VocabularyListerDialog;

/**
 * Diese Klasse findet zu einer Vokabel zugehörige Verb- und Adjektivformen zu finden, wenn die
 * Vokabel ein Verb bzw. ein Adjektiv ist.
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

class VerbsAndAdjectivesFinder {

    /** Die Programmoptionen. */
    private final Options options;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalDataRequester requester;

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalKanjiDataRequester kanjiRequester;

    /** Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann. */
    private final MessageSetter messageSetter;

    /**
     * Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese Klasse
     * verwendet.
     */
    private final Point location;

    /** Das Icon für das Programm. */
    private final Image programImage;

    /** Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen kann. */
    private final LongTimeProcessInformer longTimeProcessInformer;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param ownLists
     *            Die Verwaltung der eigenen Vokabellisten.
     * @param kanjiRequester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param messageSetter
     *            Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann.
     * @param location
     *            Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese
     *            Klasse verwendet.
     * @param programImage
     *            Das anzuzeigende ProgrammIcon.
     * @param longTimeProcessInformer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     */
    public VerbsAndAdjectivesFinder(Options options, InternalDataRequester requester,
            OwnLists ownLists, InternalKanjiDataRequester kanjiRequester,
            MessageSetter messageSetter, Point location, Image programImage,
            LongTimeProcessInformer longTimeProcessInformer) {
        this.options = options;
        this.requester = requester;
        this.ownLists = ownLists;
        this.kanjiRequester = kanjiRequester;
        this.messageSetter = messageSetter;
        this.location = location;
        this.programImage = programImage;
        this.longTimeProcessInformer = longTimeProcessInformer;
    }

    /**
     * Findet zur übergebenen Vokabel, welche ein Verb sein muss, alle Vokabeln mit den bekannten
     * Verbformen zu diesem Verb.
     *
     * @param vocable
     *            Die Verb-Vokabel.
     */
    public void findRelatedVerbs(Vocable vocable) {
        RelatedVocablesFinder finder = new RelatedVocablesFinder(ownLists.getVocabularies(),
                vocable, v -> v.isVerb());
        finder.find();
        List<Vocable> relatedVerbs = finder.getRelatedVocables();
        String description = "Verbformen zum Verb" + " "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces();

        VerbSearchWords.sortVocablesByVerbForms(relatedVerbs);

        listVocables(relatedVerbs, description);
    }

    /**
     * Findet zur übergebenen Vokabel, welche ein Adjektiv sein muss, alle Vokabeln mit den
     * bekannten Adjektivformen zu diesem Adjektiv.
     *
     * @param vocable
     *            Die Adjektiv-Vokabel.
     */
    public void findRelatedAdjectives(Vocable vocable) {
        RelatedVocablesFinder finder = new RelatedVocablesFinder(ownLists.getVocabularies(),
                vocable, v -> v.isAdjective());
        finder.find();
        List<Vocable> relatedVerbs = finder.getRelatedVocables();
        String description = "Adjektivformen zum Adjektiv" + " "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces();

        VerbSearchWords.sortVocablesByVerbForms(relatedVerbs);

        listVocables(relatedVerbs, description);
    }

    /**
     * Zeigt die übergebenen Vokabeln als Liste an.
     *
     * @param vocables
     *            Die anzuzeigenden Vokabeln.
     * @param description
     *            Die Beschreibung der anzuzeigenden Vokabeln.
     */
    private void listVocables(List<Vocable> vocables, String description) {
        VocabularyListerDialog dialog = new VocabularyListerDialog(options, requester, vocables,
                description, kanjiRequester, longTimeProcessInformer, ownLists, messageSetter,
                location, programImage);
        dialog.setVisible(true);
    }

}
