package de.duehl.vocabulary.japanese.ui.listcommander.list.element;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.JLabel;

import de.duehl.basics.text.Text;
import de.duehl.basics.text.html.HtmlTool;
import de.duehl.swing.ui.GuiTools;
import de.duehl.twosidecommander.ui.list.data.ListElementMoveReactor;
import de.duehl.twosidecommander.ui.list.element.ListElementDisplayer;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.ui.listcommander.OwnListCommander;

import static de.duehl.twosidecommander.ui.color.ListCommanderColors.*;

/**
 * Diese Klasse stellt die grafische Oberfläche eines Elements einer eigenen Liste des
 * Listen-Commanders dar, ergo eine Vokabel der eigenen Liste, mit dem sich generisch zwischen zwei
 * Listen von Dingen Listenelemente kopieren, verschieben oder löschen lassen.
 *
 * @version 1.01     2025-07-08
 * @author Christian Dühl
 */

public class OwnListElementDisplayer extends ListElementDisplayer {

    /** Der List-Commander. */
    private final OwnListCommander commander;

    /** Die dargestellte Vokabel (das Listenelement). */
    private final Vocable vocable;

    /** Das Label mit der Vokabel. */
    private JLabel label;

    /**
     * Konstruktor.
     *
     * @param listElementMoveReactor
     *            Reagiert auf die Buttons zum vertikalen Verschieben des Listenelements.
     * @param commander
     *            Der List-Commander.
     * @param vocable
     *            Die dargestellte Vokabel (das Listenelement).
     */
    public OwnListElementDisplayer(ListElementMoveReactor listElementMoveReactor,
            OwnListCommander commander, Vocable vocable) {
        super(listElementMoveReactor);
        this.commander = commander;
        this.vocable = vocable;
        setListElementShowDetailsName("Details der Vokabel anzeigen ...");
        initLabel();
        createGui();
    }

    private static final int MAX_TEXT_LENGTH = 80;

    private void initLabel() {
        String text = " "
                + vocable.getTranslations().get(0)
                + " - "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces()
                + " ";

        if (text.length() > MAX_TEXT_LENGTH) {
            text = text.substring(0, MAX_TEXT_LENGTH) + " ...";
        }

        text = Text.addLineBreaks(text, 60, 10);
        text = HtmlTool.htmlify(text);

        label = new JLabel(text);
        GuiTools.biggerFont(label, 3);
        label.setOpaque(true);
        label.setBackground(Color.WHITE);
        label.setBorder(BorderFactory.createEmptyBorder(0, 2, 0, 2)); // top left bottom right
        // Ich will den Innenbereich vergrößern, aber das klappt irgendwie nicht.
        label.setPreferredSize(new Dimension(1, 40));
    }

    /** Im Rechtsklick-Menü des Listenelements wurde die Ansicht der Details ausgewählt. */
    @Override
    protected final void reactOnShowDetailsClickInRightClickMenu() {
        commander.showVocableDetails(vocable);
    }

    /** Erstellt die Ui-Komponente mit der Ansicht der Vokabel in der Liste. */
    @Override
    protected Component createElementDisplay() {
        return label;
    }

    /** Stellt das Element ausgewählt und hervorgehoben dar. */
    @Override
    public void showSelectedAndHighlighted() {
        label.setForeground(SELECTED_LABEL_FOREGROUND_COLOR);
        label.setBackground(HIGHLIGHTED_LABEL_BACKGROUND_COLOR);
    }

    /** Stellt das Element ausgewählt dar. */
    @Override
    public final void showSelected() {
        label.setForeground(SELECTED_LABEL_FOREGROUND_COLOR);
        label.setBackground(NORMAL_LABEL_BACKGROUND_COLOR);
    }

    /** Stellt das Element hervorgehoben dar. */
    @Override
    public final void showHighlighted() {
        label.setForeground(NORMAL_LABEL_FOREGROUND_COLOR);
        label.setBackground(HIGHLIGHTED_LABEL_BACKGROUND_COLOR);
    }

    /** Stellt das Element normal dar. */
    @Override
    public final void showNormal() {
        label.setForeground(NORMAL_LABEL_FOREGROUND_COLOR);
        label.setBackground(NORMAL_LABEL_BACKGROUND_COLOR);
    }

    @Override
    public String toString() {
        return "OwnListElementDisplayer [" + vocable.getKanjiKanaRomajiWithJapaneseBraces() + "]";
    }

    /** Gibt eine Beschreibung des Elements zur Anzeige in Dialogen zurück. */
    @Override
    public String getElementDescription() {
        return vocable.getTranslations().get(0) + " - "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces();
    }

    /** Getter für die dargestellte Vokabel (das Listenelement). */
    public Vocable getVocable() {
        return vocable;
    }

    /** Erzeugt aus einem ListElementDisplayer einen OwnListElementDisplayer. */
    public final static OwnListElementDisplayer castListElementDisplayer(
            ListElementDisplayer listElementDisplayer) {
        return (OwnListElementDisplayer) listElementDisplayer;
    }

}
