package de.duehl.swing.ui.components.selections;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.JCheckBox;
import javax.swing.JPanel;

import de.duehl.swing.ui.components.selections.data.SearchWordSelectionInputs;

/**
 * Diese Klasse stellt die Auswahl eines Textes (z.B. in einem Suchdialog) dar, der CheckBoxen
 * enthält, mit denen man angeben kann, ob man ganze Worte sucht und ob die Groß-/Kleinschreibung
 * beachtet werden soll.
 *
 * @version 1.01     2024-11-22
 * @author Christian Dühl
 */

public class SearchWordStringSelection extends StringSelection {

    /** Die Auswahl, ob der Suchbegriff als ganzes Wort gesucht wird. */
    private final JCheckBox wholeWordCheckBox;

    /** Die Auswahl, ob die Groß-/Kleinschreibung beachtet wird. */
    private final JCheckBox caseSensitivCheckBox;

    /**
     * Konstruktor.
     *
     * @param title
     *            Überschrift für die Wahl dieser Datei oder dieses Verzeichnisses.
     */
    public SearchWordStringSelection(String title) {
        super(title);

        wholeWordCheckBox = new JCheckBox("ganzes Wort"); // ganzes Wort suchen
        caseSensitivCheckBox = new JCheckBox("Groß-/Kleinschreibung"); // Groß-/Kleinschreibung beachten

        mainPanel = createNewMainPanel();
    }

    /** Erstellt den Panel mit der Überschrift und dem Eingabefeld in vertikaler Anordnung. */
    private JPanel createNewMainPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(super.createPanel(), BorderLayout.CENTER);
        panel.add(createSearchOptionsPart(), BorderLayout.SOUTH);

        return panel;
    }

    private Component createSearchOptionsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        //panel.setLayout(new FlowLayout(FlowLayout.LEFT, 2, 0));
        //panel.setLayout(new GridLayout(0, 2, 1, 1));

        panel.add(wholeWordCheckBox, BorderLayout.WEST);
        panel.add(caseSensitivCheckBox, BorderLayout.EAST);

        return panel;
    }

    /** Ändert die Anordnung von Überschrift und Textfeld in horizontal, Überschrift links. */
    @Override
    public void makeHorizontal() {
        throw new RuntimeException("not implemented");
    }

    /** Gibt an, ob die Suche nach ganzen Worten erfolgen soll. */
    public boolean isWholeWordSearch() {
        return wholeWordCheckBox.isSelected();
    }

    /** Legt fest, ob die Suche nach ganzen Worten erfolgen soll. */
    public void setWholeWordSearch(boolean wholeWordSearch) {
        wholeWordCheckBox.setSelected(wholeWordSearch);
    }

    /** Gibt an, ob die Suche Groß-/Kleinschreibung berücksichtigen soll. */
    public boolean isCaseSensitiveSearch() {
        return caseSensitivCheckBox.isSelected();
    }

    /** Legt fest, ob die Suche Groß-/Kleinschreibung berücksichtigen soll. */
    public void setCaseSensitiveSearch(boolean caseSensitiveSearch) {
        caseSensitivCheckBox.setSelected(caseSensitiveSearch);
    }

    /** Lädt die Suchparameter aus dem übergebenen Speicherobjekt. */
    public void loadInputs(SearchWordSelectionInputs inputs) {
        setText(inputs.getText());
        setWholeWordSearch(inputs.isWholeWordSearch());
        setCaseSensitiveSearch(inputs.isCaseSensitiveSearch());
    }

    /** Speichert die Suchparameter im übergebenen Speicherobjekt. */
    public void storeInputs(SearchWordSelectionInputs inputs) {
        inputs.setText(getTrimmedText());
        inputs.setWholeWordSearch(isWholeWordSearch());
        inputs.setCaseSensitiveSearch(isCaseSensitiveSearch());
    }

    /** Gibt die Suchparameter zurück. */
    public SearchWordSelectionInputs getInputs() {
        SearchWordSelectionInputs inputs = new SearchWordSelectionInputs();
        storeInputs(inputs);
        return inputs;
    }

    /** Legt fest, ob die beiden Checkboxen fokussierbar sind. */
    public void setCheckboxesFocusable(boolean focusable) {
        wholeWordCheckBox.setFocusable(focusable);
        caseSensitivCheckBox.setFocusable(focusable);
    }

}

