package de.duehl.swing.ui.elements;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import javax.swing.JLabel;
import javax.swing.SwingUtilities;

import de.duehl.basics.text.NumberString;
import de.duehl.swing.ui.GuiTools;

/**
 * Diese Klasse stellt ein Label darf, das die auf ihm angezeigte Zählung pro Aufruf aktualisiert.
 *
 * @version 1.01     2020-11-12
 * @author Christian Dühl
 */

public class CountLabel extends JLabel {

    private static final long serialVersionUID = 1L;

    /** Text vor dem Zähler. */
    private volatile String startText;

    /** Text nach dem Zähler. */
    private volatile String endText;

    /** Zähler. */
    private int count;

    /** Konstruktor ohne Text vor und nach dem Zähler. */
    public CountLabel() {
        this("", 0, "");
    }

    /**
     * Konstruktor mit Text vor dem Zähler.
     *
     * @param startText
     *            Text vor dem Zähler.
     */
    public CountLabel(String startText) {
        this(startText, 0, "");
    }

    /**
     * Konstruktor mit Text vor und nach dem Zähler.
     *
     * @param startText
     *            Text vor dem Zähler.
     * @param endText
     *            Text nach dem Zähler.
     */
    public CountLabel(String startText, String endText) {
        this(startText, 0, endText);
    }

    /**
     * Konstruktor mit Text vor und nach dem Zähler sowie eigenem Zählerstand, der übergeben wird.
     *
     * @param startText
     *            Text vor dem Zähler.
     * @param count
     *            Zählerstand zu Beginn.
     * @param endText
     *            Text nach dem Zähler.
     */
    public CountLabel(String startText, int count, String endText) {
        super();
        this.startText = startText;
        this.count = count;
        this.endText = endText;

        GuiTools.setMonospacedFont(this); // ?
        setCounterAsText();
    }

    private void setCounterAsText() {
        SwingUtilities.invokeLater(() -> setCounterAsTextInEdt());
    }

    private void setCounterAsTextInEdt() {
        setText(startText + NumberString.taupu(count) + endText);
    }

    /** Setter für den Text vor dem Zähler. */
    public void setStartText(String startText) {
        this.startText = startText;
        setCounterAsText();
    }

    /** Setter für den Text nach dem Zähler. */
    public void setEndText(String endText) {
        this.endText = endText;
        setCounterAsText();
    }

    /** Legt den Zählerstand fest. */
    public void setCount(int count) {
        this.count = count;
        setCounterAsText();
    }

    /** Erhöht den Zählerstand. */
    public void increment() {
        ++count;
        setCounterAsText();
    }

    /** Stringrepräsentation. */
    @Override
    public String toString() {
        return "CountLabel [startText=" + startText + ", endText=" + endText + ", count=" + count
                + "]";
    }

}
