package de.duehl.swing.ui.move;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridLayout;

import javax.swing.JPanel;

import de.duehl.swing.ui.buttons.painted.DownButton;
import de.duehl.swing.ui.buttons.painted.FirstButton;
import de.duehl.swing.ui.buttons.painted.LastButton;
import de.duehl.swing.ui.buttons.painted.UpButton;
import de.duehl.swing.ui.colors.ColorTool;
import de.duehl.swing.ui.colors.NamedColorListFabric;
import de.duehl.swing.ui.move.data.VerticalMoveButtonsUser;

/**
 * Diese Klasse stellt einen Panel mit vier Buttons dar, um Dinge nach oben, unten, ganz nach oben
 * oder ganz nach unten zu bewegen.
 *
 * @version 1.01     2025-07-23
 * @author Christian Dühl
 */

public class VerticalMoveButtons {

    public static final String DEFAULT_MOVE_BUTTON_COLOR = NamedColorListFabric.DARKGREEN;
    public static final String DEFAULT_DISABLED_COLOR = NamedColorListFabric.DARKGRAY;
    public static final Dimension DEFAULT_BUTTON_DIMENSION = new Dimension(35, 35);


    /** Das Objekt, das die Buttons zum verschieben anzeigt und verwendet. */
    private final VerticalMoveButtonsUser moveButtonsUser;

    /** Der Panel mit Buttons zum Verschieben nach oben und unten. */
    private final JPanel panel;

    /** Der Schalter um das Element mit diesen Buttons an die oberste Stelle zu verschieben. */
    private final FirstButton firstButton;

    /** Der Schalter um das Element mit diesen Buttons nach oben zu verschieben. */
    private final UpButton upButton;

    /** Der Schalter um das Element mit diesen Buttons nach unten zu verschieben. */
    private final DownButton downButton;

    /** Der Schalter um das Element mit diesen Buttons an die unterste Stelle zu verschieben. */
    private final LastButton lastButton;

    /** Die Farbe eines Buttons, der benutzbar ist. */
    private Color moveButtonColor;

    /** Die Farbe eines Buttons, der nicht benutzbar ist. */
    private Color disabledColor;

    /** Die Größe der Move-Buttons. */
    private Dimension buttonDimension;

    /**
     * Konstruktor.
     *
     * @param moveButtonsUser
     *            Das Objekt, das die Buttons zum verschieben anzeigt und verwendet.
     */
    public VerticalMoveButtons(VerticalMoveButtonsUser moveButtonsUser) {
        this.moveButtonsUser = moveButtonsUser;

        panel = new JPanel();

        moveButtonColor = ColorTool.getColorByName(DEFAULT_MOVE_BUTTON_COLOR);
        disabledColor = ColorTool.getColorByName(DEFAULT_DISABLED_COLOR);
        buttonDimension = DEFAULT_BUTTON_DIMENSION;

        firstButton = new FirstButton(disabledColor);
        upButton = new UpButton(disabledColor);
        downButton = new DownButton(disabledColor);
        lastButton = new LastButton(disabledColor);

        init();
    }

    private void init() {
        initMoveUpAndDownButtonsPanel();
        initMoveButtons();
    }

    private void initMoveUpAndDownButtonsPanel() {
        panel.setLayout(new GridLayout(1, 0, 2, 2));

        panel.add(firstButton);
        panel.add(upButton);
        panel.add(downButton);
        panel.add(lastButton);
    }

    private void initMoveButtons() {
        firstButton.setHorizontal(false);
        firstButton.forceSquare();
        firstButton.addActionListener(e -> moveButtonsUser.moveMoveButtonsUserToFirst());

        upButton.forceSquare();
        upButton.addActionListener(e -> moveButtonsUser.moveMoveButtonsUserUp());

        downButton.forceSquare();
        downButton.addActionListener(e -> moveButtonsUser.moveMoveButtonsUserDown());

        lastButton.setHorizontal(false);
        lastButton.forceSquare();
        lastButton.addActionListener(e -> moveButtonsUser.moveMoveButtonsUserToLast());

        initMoveButtonsPreferredSize();
    }

    private void initMoveButtonsPreferredSize() {
        firstButton.setPreferredSize(buttonDimension);
        upButton.setPreferredSize(buttonDimension);
        downButton.setPreferredSize(buttonDimension);
        lastButton.setPreferredSize(buttonDimension);
    }

    /** Setter für die Farbe eines Buttons, der benutzbar ist. */
    public void setMoveButtonColor(Color moveButtonColor) {
        this.moveButtonColor = moveButtonColor;
        setUpAndDownButtonColorsAndEnabled();
    }

    /** Setter für die Farbe eines Buttons, der nicht benutzbar ist. */
    public void setDisabledColor(Color disabledColor) {
        this.disabledColor = disabledColor;
        setUpAndDownButtonColorsAndEnabled();
    }

    /**
     * Setter für die Farbe eines Buttons, der benutzbar ist über einen Farbnamen aus der Klasse
     * NamedColorListFabric.
     */
    public void setMoveButtonColor(String moveButtonColor) {
        this.moveButtonColor = ColorTool.getColorByName(moveButtonColor);
        setUpAndDownButtonColorsAndEnabled();
    }

    /**
     * Setter für die Farbe eines Buttons, der nicht benutzbar ist über einen Farbnamen aus der
     * Klasse NamedColorListFabric.
     */
    public void setDisabledColor(String disabledColor) {
        this.disabledColor = ColorTool.getColorByName(disabledColor);
        setUpAndDownButtonColorsAndEnabled();
    }

    /** Setter für die Größe der Move-Buttons. */
    public void setButtonDimension(Dimension buttonDimension) {
        this.buttonDimension = buttonDimension;
        initMoveButtonsPreferredSize();
    }

    /**
     * Setzt die Farben und Darstellung der Buttons abhängig davon, ob sie verschoben werden
     * können.
     */
    public void setUpAndDownButtonColorsAndEnabled() {
        boolean canMoveUp = moveButtonsUser.canMoveButtonsUserMoveUp();
        Color upButtonColor = canMoveUp ? moveButtonColor : disabledColor;

        upButton.setSymbolColor(upButtonColor);
        upButton.setEnabled(canMoveUp);
        upButton.repaint();

        firstButton.setSymbolColor(upButtonColor);
        firstButton.setEnabled(canMoveUp);
        firstButton.repaint();

        boolean canMoveDown = moveButtonsUser.canMoveButtonsUserMoveDown();
        Color downButtonColor = canMoveDown ? moveButtonColor : disabledColor;

        downButton.setSymbolColor(downButtonColor);
        downButton.setEnabled(canMoveDown);
        downButton.repaint();

        lastButton.setSymbolColor(downButtonColor);
        lastButton.setEnabled(canMoveDown);
        lastButton.repaint();
    }

    /** Getter für den Panel mit Buttons zum Verschieben nach oben und unten. */
    public JPanel getPanel() {
        return panel;
    }

}
