package de.duehl.swing.ui.text.html;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Component;
import java.net.URL;
import java.awt.Image;
import java.awt.Point;

import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog um einen Text in HTML in einem Textfeld anzuzeigen.
 *
 * @version 1.01     2022-08-04
 * @author Christian Dühl
 */

public final class HtmlDialog extends ModalDialogBase {

    /** Der eingebettete Panel mit HTML, Quellcode und Buttons. */
    private final HtmlAndSourceCodePanel htmlPanel;

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     */
    public HtmlDialog(String title) {
        this(title, new Point(100, 100));
    }

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     * @param parentLocation
     *            Lokation des Aufrufenden Dialoges.
     */
    public HtmlDialog(String title, Point parentLocation) {
        this(title, null, parentLocation);
    }

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     * @param programImage
     *            Image vom aufrufenden Programm.
     * @param parentLocation
     *            Lokation des Aufrufenden Dialoges.
     */
    public HtmlDialog(String title, Image programImage, Point parentLocation) {
        super(parentLocation, programImage, title);
        addEscapeBehaviour();

        htmlPanel = HtmlAndSourceCodePanel.createWithOkButtonAndToggleButton(() -> closeDialog());
        htmlPanel.setKeyBindings(getRootPane());

        fillDialog();
    }

    @Override
    protected void populateDialog() {
        add(htmlPanel.getComponent(), BorderLayout.CENTER);
    }

    /** Fügt einen weiteren Button rechts unten zum Dialog hinzu. */
    public void addButtonOnRight(Component button) {
        htmlPanel.addButtonOnRight(button);
    }

    /** Fügt einen weiteren Button links unten zum Dialog hinzu. */
    public void addButtonOnLeft(Component button) {
        htmlPanel.addButtonOnLeft(button);
    }

    /**
     * Schreibt den Text der Seite hinter der URL in den Hauptteil des Dialogs und setzt den
     * Textcursor auf den Anfang des Dokuments.
     *
     * @param url
     *            URL, aus der der Inhalt gelesen wird.
     */
    public void showHtml(URL url) {
        htmlPanel.showHtml(url);
    }

    /**
     * Schreibt das HTML in den Anzeigebereich und setzt den Textcursor auf den Anfang des
     * Dokuments.
     *
     * @param html
     *            Darzustellender Text.
     */
    public void setText(String html) {
        htmlPanel.setText(html);
    }

    /** Scrollt zum Anfang. */
    public void scrollScrollbarToMinimumLater() {
        htmlPanel.scrollScrollbarToMinimumLater();
    }

    /** Scrollt zum Ende. */
    public void scrollScrollbarToMaximumLater() {
        htmlPanel.scrollScrollbarToMaximumLater();
    }

    /** Scrollt weiter nach oben. */
    public void scrollScrollbarToPreviousSectionLater() {
        htmlPanel.scrollScrollbarToPreviousSectionLater();
    }

    /** Scrollt weiter nach unten. */
    public void scrollScrollbarToNextSectionLater() {
        htmlPanel.scrollScrollbarToNextSectionLater();
    }

    /** Setzt die gewünschte Größe. */
    @Override
    public void setPreferredSize(Dimension dimension) {
        htmlPanel.setPreferredSize(dimension);
        //super.setPreferredSize(dimension);
    }

}
