package de.duehl.vocabulary.japanese.data;

import java.util.List;

import de.duehl.basics.datetime.date.ImmutualDate;

/**
 * Diese Klasse stellt ein Vokabular dar.
 *
 * @version 1.01     2024-03-19
 * @author Christian Dühl
 */

public class Vocabulary {

    /** Die Beschreibung des Vokabulars. */
    private String description;

    /** Die Liste der Vokabeln der Vokabulars. */
    private final List<Vocable> vocables;

    /** Die Kategorie zu der das Vokabular gehört. */
    private String category;

    /** Die Unterkategorie zu der das Vokabular gehört. */
    private String subCategory;

    /** Das früheste Datum an dem eine Vokabel aus dem Vokabular das erste Mal eingelesen wurde. */
    private ImmutualDate firstSeenDate;

    /**
     * Konstruktor.
     *
     * @param description
     *            Die Beschreibung des Vokabulars.
     * @param vocables
     *            Die Liste der Vokabeln der Vokabulars
     * @param category
     *            Die Kategorie zu der das Vokabular gehört.
     * @param subCategory
     *            Die Unterkategorie zu der das Vokabular gehört.
     */
    public Vocabulary(String description, List<Vocable> vocables, String category,
            String subCategory) {
        this.description = description;
        this.vocables = vocables;
        this.category = category;
        this.subCategory = subCategory;
        firstSeenDate = new ImmutualDate();
    }

    /** Getter für die Beschreibung des Vokabulars. */
    public String getDescription() {
        return description;
    }

    /** Setter für die Beschreibung des Vokabulars. */
    void setDescription(String description) {
        this.description = description;
    }

    /** Getter für die Liste der eingelesenen Vokabeln. */
    public List<Vocable> getVocables() {
        return vocables;
    }

    /** Getter für die Kategorie zu der das Vokabular gehört. */
    public String getCategory() {
        return category;
    }

    /** Setter für die Kategorie zu der das Vokabular gehört. */
    public void setCategory(String category) { // Nur für eigene Listen
        this.category = category;
    }

    /** Getter für die Unterkategorie zu der das Vokabular gehört. */
    public String getSubCategory() {
        return subCategory;
    }

    /** Setter für die Unterkategorie zu der das Vokabular gehört. */
    public void setSubCategory(String subCategory) { // Nur für eigene Listen
        this.subCategory = subCategory;
    }

    /**
     * Getter für das früheste Datum an dem eine Vokabel aus dem Vokabular das erste Mal eingelesen
     * wurde.
     */
    public ImmutualDate getFirstSeenDate() {
        return firstSeenDate;
    }

    /**
     * Setter für das früheste Datum an dem eine Vokabel aus dem Vokabular das erste Mal eingelesen
     * wurde.
     */
    public void setFirstSeenDate(ImmutualDate firstSeenDate) {
        this.firstSeenDate = firstSeenDate;
    }

    @Override
    public String toString() {
        return "Vocabulary [description=" + description + ", vocables=" + vocables + ", category="
                + category + ", subCategory=" + subCategory + ", firstSeenDate=" + firstSeenDate
                + "]";
    }

    /** Erzeugt eine leserliche, schöne Ausgabe. */
    public String toNiceString() {
        StringBuilder builder = new StringBuilder();

        builder.append("Vocabulary:\n");
        builder.append("    Beschreibung                     : " + description + "\n");
        builder.append("    Kategorie                        : " + category + "\n");
        builder.append("    Unterkategorie                   : " + subCategory + "\n");
        builder.append("    Ältestes Datum einer der Vokabeln: " + firstSeenDate + "\n");
        builder.append("    Vocables:\n");

        for (Vocable vocable : vocables) {
            builder.append(vocable.toNiceString(8));
            builder.append("\n");
        }

        return builder.toString();
    }

}
