package de.duehl.vocabulary.japanese.data.symbol;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse stellt die Zeichen der Schrift Kanji (in UTF-Zeichen) dar, die ich bislang gelernt
 * habe. Da es davon über 50.000 Stück gibt, wird diese Klasse niemals vollständig sein.
 *
 * Ich hoffe aber, sie enthält irgendwann die 2.136 常用漢字 (Jōyō-Kanji), vgl.
 * https://de.wikipedia.org/wiki/J%C5%8Dy%C5%8D-Kanji.
 *
 * Die Klasse beinhaltet auch Methoden rund darum, um zu fragen, ob eine Zeichenkette
 * ausschließlich aus diesen Zeichen besteht oder wo das erste bzw. letzte dieser Zeichen zu finden
 * ist.
 *
 * Möchte man an die Übersetzungen, so kann man nach dem Wort über normalKanjiHiraganaWord oder
 * normalHiraganaWord suchen, das muss man nicht hier auch noch vorhalten.
 *
 * https://www-kanshudo-com.translate.goog/kanji/%E6%B0%97?_x_tr_sl=en&_x_tr_tl=de&_x_tr_hl=de&_x_tr_pto=rq
 *
 * @version 1.01     2025-10-29
 * @author Christian Dühl
 */

public enum Kanji {

    FISCH("魚", "Fisch", "魚", "さかな", 290, 112,
            CollectionsHelper.buildListFrom("GYO"),
            CollectionsHelper.buildListFrom("sakana", "uo")
            ),
    FLEISCH("肉", "Fleisch", "肉", "にく", 223, 102,
            CollectionsHelper.buildListFrom("NIKU"),
            CollectionsHelper.buildListFrom()
            ),
    EI("卵", "Ei", "卵", "たまご", 1103, 228,
            CollectionsHelper.buildListFrom("RAN"),
            CollectionsHelper.buildListFrom("tamago")
            ),
    WASSER("水", "Wasser", "水", "みず", 21, 73,
            CollectionsHelper.buildListFrom("SUI"),
            CollectionsHelper.buildListFrom("mizu")
            ),
    ESSEN("食", "essen", "食べる", "たべる", 322, 116,
            CollectionsHelper.buildListFrom("SHOKU", "[JIKI]"),
            CollectionsHelper.buildListFrom("ta(beru)", "ku(u)", "ku(rau)")
            ),
    TRINKEN("飲", "trinken", "飲む", "のむ", 323, 117,
            CollectionsHelper.buildListFrom("IN"),
            CollectionsHelper.buildListFrom("no(mu)")
            ),
    GROSS("大", "groß", "大きい", "ちいさい", 26, 74,
            CollectionsHelper.buildListFrom("DAI", "TAI"),
            CollectionsHelper.buildListFrom("oo(kii)", "oo-", "oo(ini)")
            ),
    KLEIN("小", "klein", "小さい", "ちいさい", 27, 74,
            CollectionsHelper.buildListFrom("SHOU"),
            CollectionsHelper.buildListFrom("chii(sai)", "ko-", "o-")
            ),
    NEU("新", "neu", "新しい", "あたらしい", 174, 95,
            CollectionsHelper.buildListFrom("SHIN"),
            CollectionsHelper.buildListFrom("atara(shii)", "ara(ta)", "nii-")
            ),
    ALT("古", "alt", "古い", "ふるい", 172, 95,
            CollectionsHelper.buildListFrom("KO"),
            CollectionsHelper.buildListFrom("furu(i)", "furu(su)")
            ),
    TAG_SONNE("日", "Tag, Sonne", "日", "にち", 5, 71,
            CollectionsHelper.buildListFrom("NICHI", "JITSU"),
            CollectionsHelper.buildListFrom("hi", "ka")
            ),
    BUCH_URSPRUNG("本", "Buch, Ursprung, Haupt-", "本", "ほん", 25, 74,
            CollectionsHelper.buildListFrom("HON"),
            CollectionsHelper.buildListFrom("moto")
            ),
    PERSON_MENSCH("人", "Person, Mensch", "人", "ひと", 1, 71,
            CollectionsHelper.buildListFrom("JIN", "NIN"),
            CollectionsHelper.buildListFrom("hito")
            ),
    GEHEN_FAHREN("行", "gehen, fahren, fliegen", "行く", "いく", 68, 80,
            CollectionsHelper.buildListFrom("KOU", "[AN]", "GYOU"),
            CollectionsHelper.buildListFrom("i(ku)", "yu(ku)", "okona(u)")
            ),
    ELEKTRIZITAET("電", "Elektrizität", "電車", "でんしゃ", 108, 86,
            CollectionsHelper.buildListFrom("DEN"),
            CollectionsHelper.buildListFrom()
            ),
    AUTO_WAGEN_RAD("車", "Auto, Wagen, Rad", "車", "くるま", 133, 89,
            CollectionsHelper.buildListFrom("SHA"),
            CollectionsHelper.buildListFrom("kuruma")
            ),
    FRAU("女", "Frau", "女", "おんな", 102, 85,
            CollectionsHelper.buildListFrom("JO", "NYO", "[NYOU]"),
            CollectionsHelper.buildListFrom("onna", "me")
            ),
    MANN("男", "Mann", "男", "おとこ", 101, 85,
            CollectionsHelper.buildListFrom("DAN", "NAN"),
            CollectionsHelper.buildListFrom("otoko")
            ),
    SPRECHEN("話", "sprechen, reden", "話す", "はなす", 238, 104,
            CollectionsHelper.buildListFrom("WA"),
            CollectionsHelper.buildListFrom("hanashi", "hana(su)")
            ),
    SICH_TREFFEN("会", "sich treffen, sich versammeln", "会う", "あう", 158, 93,
            CollectionsHelper.buildListFrom("KAI", "E"),
            CollectionsHelper.buildListFrom("a(u)")
            ),
    SICH_AUSRUHEN("休", "sich ausruhen", "休む", "休", 60, 79,
            CollectionsHelper.buildListFrom("KYUU"),
            CollectionsHelper.buildListFrom("yasu(mu)", "yasu(meru)", "yasu(maru)")
            ),
    SEHEN_ANSEHEN("見", "sehen, ansehen", "見る", "みる", 63, 79,
            CollectionsHelper.buildListFrom("KEN"),
            CollectionsHelper.buildListFrom("mi(ru)", "mi(eru)", "mi(seru)")
            ),
    HIMMEL_PARADIES("天", "Himmel, Paradies", "天気", "てんき", 141, 91,
            CollectionsHelper.buildListFrom("TEN"),
            CollectionsHelper.buildListFrom("ame", "[ama]")
            ),
    GEIST_SEELE_ENERGIE("気", "Geist, Seele, Energie", "元気", "げんき", 134, 90,
            CollectionsHelper.buildListFrom("KI", "KE"),
            CollectionsHelper.buildListFrom()
            ),
    STADT_ORTSCHAFT("町", "Stadt, Ortschaft", "町", "まち", 182, 96,
            CollectionsHelper.buildListFrom("CHOU"),
            CollectionsHelper.buildListFrom("machi")
            ),
    LERNEN("学", "lernen", "大学", "だいがく", 109, 86,
            CollectionsHelper.buildListFrom("GAKU"),
            CollectionsHelper.buildListFrom("mana(bu)")
            ),
    BLUME("花", "Blume", "花", "はな", 255, 107,
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("hana")
            ),
    JETZT("今", "jetzt", "今", "いま", 51, 78,
            CollectionsHelper.buildListFrom("KON", "KIN"),
            CollectionsHelper.buildListFrom("ima")
            ),
    ZEIT_STUNDE("時", "Zeit, Stunde, ... Uhr", "時", "じ", 42, 76,
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("toki")
            ),
    TEILEN_VERSTEHEN_TEIL_MINUTE("分", "teilen, Teil, Minute, verstehen", "分かる", "わかる", 38, 76,
            CollectionsHelper.buildListFrom("BUN", "BU", "FUN", "PUN"),
            CollectionsHelper.buildListFrom("wa(keru)", "wa(katsu)", "wa(kareru)", "wa(karu)")
            ),
    SPITZE_ZUVOR("先", "Spitze, zuvor", "先", "さき", 50, 78,
            CollectionsHelper.buildListFrom("SEN"),
            CollectionsHelper.buildListFrom("saki")
            ),
    GEBAEREN_LEBEN("生", "gebären, leben", "生む", "うむ", 44, 77,
            CollectionsHelper.buildListFrom("SEI", "SHOU"),
            CollectionsHelper.buildListFrom("i(kiru)", "i(keru)", "i(kasu)", "u(mu)", "ha(yasu)",
                    "ha(eru)", "o(u)")
            ),
    SPRACHE("語", "Sprache", "語", "ご", 67, 80,
            CollectionsHelper.buildListFrom("GO"),
            CollectionsHelper.buildListFrom("kata(ru)", "kata(rau)")
            ),
    REGEN("雨", "Regen", "雨", "あめ", 30, 75,
            CollectionsHelper.buildListFrom("U"),
            CollectionsHelper.buildListFrom("ame", "[ama]")
            ),
    KOMMEN("来", "kommen", "来る", "くる", 69, 80,
            CollectionsHelper.buildListFrom("RAI"),
            CollectionsHelper.buildListFrom("ku(ru)", "kita(ru)", "kita(su)")
            ),
    FEUER("火", "Feuer", "火", "ひ", 20, 73,
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("hi", "[ho]")
            ),
    BAUM_HOLZ("木", "Baum, Holz", "木", "き", 22, 74,
            CollectionsHelper.buildListFrom("BOKU", "MOKU"),
            CollectionsHelper.buildListFrom("ki", "[ka]")
            ),
    WENIG("少", "wenig", "少し", "すこし", 144, 91,
            CollectionsHelper.buildListFrom("SHOU"),
            CollectionsHelper.buildListFrom("suko(shi)", "suku(nai)")
            ),
    VIEL_ZAHLREICH("多", "viel,  zahlreich", "多い", "おおい", 229, 103,
            CollectionsHelper.buildListFrom("TA"),
            CollectionsHelper.buildListFrom("oo(i)")
            ),
    WAS("何", "was", "何", "なに", 392, 126,
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("nani", "[nan]")
            ),
    HOEREN_FRAGEN("聞", "hören, fragen", "聞く", "きく", 64, 80,
            CollectionsHelper.buildListFrom("BUN", "MON"),
            CollectionsHelper.buildListFrom("ki(ku)", "ki(koru)")
            ),
    KAUFEN("買", "kaufen", "買う", "かう", 241, 105,
            CollectionsHelper.buildListFrom("BAI"),
            CollectionsHelper.buildListFrom("ka(u)")
            ),
    HOCH_TEUER("高", "hoch, teuer", "高い", "たかい", 190, 98,
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("taka(i)", "taka", "taka(maru)", "taka(meru)")
            ),
    BILLIG_SICHER("安", "billig, (sicher)", "安い", "やすい", 105, 85,
            CollectionsHelper.buildListFrom("AN"),
            CollectionsHelper.buildListFrom("yasu(i)")
            ),
    KIND("子", "Kind", "子供", "こども", 103, 85,
            CollectionsHelper.buildListFrom("SHI", "SU"),
            CollectionsHelper.buildListFrom("ko")
            ),
    WEISS("白", "Weiß", "白", "しろ", 205, 100,
            CollectionsHelper.buildListFrom("HAKU", "BYAKU"),
            CollectionsHelper.buildListFrom("shiro(i)", "shiro")
            ),
    LADEN_GESCHAEFT("店", "Laden, Geschäft", "店", "みせ", 168, 94,
            CollectionsHelper.buildListFrom("TEN"),
            CollectionsHelper.buildListFrom("mise")
            ),
    MOND_MONAT("月", "Mond, Monat", "月", "つき", 17, 73,
            CollectionsHelper.buildListFrom("GETSU", "GATSU"),
            CollectionsHelper.buildListFrom("tsuki")
            ),
    FUENF_5("五", "fünf, 5", "五", "ご", 7, 71,
            CollectionsHelper.buildListFrom("GO"),
            CollectionsHelper.buildListFrom("itsu(tsu)", "itsu-")
            ),
    MITTE_INNEN("中", "Mitte, innen", "中", "なか", 28, 74,
            CollectionsHelper.buildListFrom("CHUU", "-CHUU", "-JUU"),
            CollectionsHelper.buildListFrom("naka")
            ),
    MITTAG("午", "Mittag", "午後", "ごご", 49, 77,
            CollectionsHelper.buildListFrom("GO"),
            CollectionsHelper.buildListFrom()
            ),
    EINS_1("一", "eins, 1", "一", "いち", 2, 71,
            CollectionsHelper.buildListFrom("ICHI", "ITSU"),
            CollectionsHelper.buildListFrom("hito(tsu)", "hito-")
            ),
    ZWEI_2("二", "zwei, 2", "二", "に", 3, 71,
            CollectionsHelper.buildListFrom("NI"),
            CollectionsHelper.buildListFrom("futa(tsu)", "futa-")
            ),
    DREI_3("三", "drei, 3", "三", "ご", 4, 71,
            CollectionsHelper.buildListFrom("SAN"),
            CollectionsHelper.buildListFrom("mit(tsu)", "mi(tsu)", "mi-")
            ),
    VIER_4("四", "vier, 4", "四", "し", 6, 71,
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("yot(tsu)", "yo(tsu)", "yo-", "yon")
            ),
    SECHS_6("六", "sechs, 6", "六", "ろく", 8, 72,
            CollectionsHelper.buildListFrom("ROKU"),
            CollectionsHelper.buildListFrom("mut(tsu)", "mu(tsu)", "mu-", "[mui-]")
            ),
    SIEBEN_7("七", "sieben, 7", "七", "なな", 9, 72,
            CollectionsHelper.buildListFrom("SHICHI"),
            CollectionsHelper.buildListFrom("nana(tsu)", "nana-", "[nano-]")
            ),
    ACHT_8("八", "acht, 8", "八", "はち", 10, 72,
            CollectionsHelper.buildListFrom("HACHI"),
            CollectionsHelper.buildListFrom("yat(tsu)", "ya(tsu)", "ya-", "[yo-]")
            ),
    NEUN_9("九", "neun, 9", "九", "きゅう", 11, 72,
            CollectionsHelper.buildListFrom("KYUU", "KU"),
            CollectionsHelper.buildListFrom("kokono(tsu)", "kokono-")
            ),
    ZEHN_10("十", "zehn, 10", "十", "じゅう", 12, 72,
            CollectionsHelper.buildListFrom("JUU", "JI'"),
            CollectionsHelper.buildListFrom("tou", "to-")
            ),
    HUNDERT_100("百", "hundert, 100", "百", "ひゅく", 14, 72,
            CollectionsHelper.buildListFrom("HYAKU"),
            CollectionsHelper.buildListFrom()
            ),
    TAUSEND_1000("千", "tausend, 1000", "千", "せん", 15, 73,
            CollectionsHelper.buildListFrom("SEN"),
            CollectionsHelper.buildListFrom("chi-")
            ),
    GOLD("金", "Gold, Metall, Geld", "金", "きん", 23, 74,
            CollectionsHelper.buildListFrom("KIN", "KON"),
            CollectionsHelper.buildListFrom("kane", "[kana]")
            ),
    BODEN_ERDE_GRUND("土", "Boden, Erde, Grund", "土", "ど", 24, 74,
            CollectionsHelper.buildListFrom("DO", "TO"),
            CollectionsHelper.buildListFrom("tsuchi")
            ),
    OBEN_RAUF_UEBER("上", "oben, rauf, über", "上", "うえ", 32, 75,
            CollectionsHelper.buildListFrom("JOU", "[SHOU]"),
            CollectionsHelper.buildListFrom("ue", "kami", "[uwa-]", "a(geru)", "nobo(ru)",
                    "nobo(seru)", "nobo(su)")
            ),
    UNTEN_RUNTER_UNTER("下", "unten, runter, unter", "下", "した", 31, 75,
            CollectionsHelper.buildListFrom("KA", "GE"),
            CollectionsHelper.buildListFrom("shita", "moto", "shimo", "sa(geru)", "o(riru)",
                    "kuda(ru)", "kuda(saru)")
            ),
    LINKS("左", "links", "左", "ひだり", 75, 81,
            CollectionsHelper.buildListFrom("SA"),
            CollectionsHelper.buildListFrom("hidari")
            ),
    RECHTS("右", "rechts", "右", "みぎ", 76, 81,
            CollectionsHelper.buildListFrom("U", "YUU"),
            CollectionsHelper.buildListFrom("migi")
            ),
    KREIS_YEN_RUND("円", "Kreis, Yen, rund", "円", "えん", 13, 72,
            CollectionsHelper.buildListFrom("EN"),
            CollectionsHelper.buildListFrom("maru(i)")
            ),
    AUGE("目", "Auge", "目", "め", 55, 78,
            CollectionsHelper.buildListFrom("MOKU", "[BOKU]"),
            CollectionsHelper.buildListFrom("me", "[ma]")
            ),
    HAELFTE_HALB("半", "Hälfte, Halb-, halb, halb nach", "半", "はん", 88, 83,
            CollectionsHelper.buildListFrom("HAN"),
            CollectionsHelper.buildListFrom("naka(ba)")
            ),
    OHR("耳", "Ohr", "耳", "みみ", 56, 78,
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("mimi")
            ),
    MUND("口", "Mund", "口", "くち", 54, 78,
            CollectionsHelper.buildListFrom("KOU", "KU"),
            CollectionsHelper.buildListFrom("kuchi")
            ),
    HAND("手", "Hand", "手", "て", 57, 79,
            CollectionsHelper.buildListFrom("SHU"),
            CollectionsHelper.buildListFrom("te", "[ta]")
            ),
    BEIN_FUSS("足", "Bein, Fuß", "足", "あし", 58, 79,
            CollectionsHelper.buildListFrom("SOKU"),
            CollectionsHelper.buildListFrom("ashi", "ta(ru)", "ta(iru)", "ta(su)")
            ),
    KRAFT("力", "Kraft", "力", "ちから", 100, 85,
            CollectionsHelper.buildListFrom("RYOKU", "RIKI"),
            CollectionsHelper.buildListFrom("chikara")
            ),
    STEHEN_AUFSTEHEN("立", "stehen, aufstehen, aufstellen", "立つ", "たつ", 121, 88,
            CollectionsHelper.buildListFrom("RITSU", "[RYUU]"),
            CollectionsHelper.buildListFrom("ta(tsu)", "ta(teru)")
            ),
    HIMMEL_LEER("空", "Himmel, leer", "空", "そら", 140, 90,
            CollectionsHelper.buildListFrom("KUU"),
            CollectionsHelper.buildListFrom("sora", "a(keru)", "a(ku)", "kara")
            ),
    BERG("山", "Berg", "山", "やま", 34, 75,
            CollectionsHelper.buildListFrom("SAN"),
            CollectionsHelper.buildListFrom("yama")
            ),
    FLUSS("川", "Fluss", "川", "かわ", 33, 75,
            CollectionsHelper.buildListFrom("SEN"),
            CollectionsHelper.buildListFrom("kawa")
            ),
    WORT_SAGEN("言", "Wort, sagen", "言う", "いう", 66, 80,
            CollectionsHelper.buildListFrom("GEN", "GON"),
            CollectionsHelper.buildListFrom("-koto", "i(u)")
            ),
    LESEN("読", "lesen", "読む", "よむ", 244, 105,
            CollectionsHelper.buildListFrom("DOKU", "TOKU", "[TOU]"),
            CollectionsHelper.buildListFrom("yo(mu)")
            ),
    SCHREIBEN("書", "schreiben", "書く", "かく", 131, 89,
            CollectionsHelper.buildListFrom("SHO"),
            CollectionsHelper.buildListFrom("ka(ku)")
            ),
    WAELCHEN_WALD_FORST("林", "Wäldchen, Wald, Forst", "林", "はやし", 127, 89,
            CollectionsHelper.buildListFrom("RIN"),
            CollectionsHelper.buildListFrom("hayashi")
            ),
    WALD_FORST("森", "Wald, Forst", "森", "もり", 128, 89,
            CollectionsHelper.buildListFrom("SHIN"),
            CollectionsHelper.buildListFrom("mori")
            ),
    STEIN("石", "Stein", "石", "いし", 78, 82,
            CollectionsHelper.buildListFrom("SEKI", "[SHAKU]", "[KOKU]"),
            CollectionsHelper.buildListFrom("ishi")
            ),
    HUND("犬", "Hund", "犬", "いぬ", 280, 110,
            CollectionsHelper.buildListFrom("KEN"),
            CollectionsHelper.buildListFrom("inu")
            ),
    INSEKT_WURM("虫", "Insekt, Wurm", "虫", "むし", 893, 198,
            CollectionsHelper.buildListFrom("CHUU"),
            CollectionsHelper.buildListFrom("mushi")
            ),
    JAHR("年", "Jahr", "年", "ねん", 45, 77,
            CollectionsHelper.buildListFrom("NEN"),
            CollectionsHelper.buildListFrom("toshi")
            ),
    DORF("村", "Dorf", "村", "むら", 191, 98,
            CollectionsHelper.buildListFrom("SON"),
            CollectionsHelper.buildListFrom("mura")
            ),
    REISFELD("田", "Reisfeld, Feld", "田", "た", 35, 75,
            CollectionsHelper.buildListFrom("DEN"),
            CollectionsHelper.buildListFrom("ta")
            ),
    ABEND("夕", "Abend, Nacht", "夕べ", "ゆうべ", 81, 82,
            CollectionsHelper.buildListFrom("SEKI"),
            CollectionsHelper.buildListFrom("yuu")
            ),
    ROT("赤", "rot", "赤", "あか", 207, 100,
            CollectionsHelper.buildListFrom("SEKI", "[SHAKU]"),
            CollectionsHelper.buildListFrom("aka(i)", "aka", "aka(ramu)", "aka(rameru)")
            ),
    BLAU("青", "grün, blau, unreif", "青", "あお", 208, 100,
            CollectionsHelper.buildListFrom("SEI", "[SHOU]"),
            CollectionsHelper.buildListFrom("ao(i)", "ao")
            ),
    HERAUSNEHMEN_HERAUSKOMMEN("出", "herausnehmen, abschicken, hinausgehen, herauskommen",
            "出口", "でぐち", 53, 78,
            CollectionsHelper.buildListFrom("SHUTSU", "[SUI]"),
            CollectionsHelper.buildListFrom("da(su)", "de(ru)")
            ),
    HINEINGEHEN_EINLASSEN("入", "hineingehen, eintreten, hereinstecken, einlassen",
            "入り口", "いりぐち", 52, 78,
            CollectionsHelper.buildListFrom("NYUU"),
            CollectionsHelper.buildListFrom("hai(ru)", "i(ru)", "i(eru)")
            ),
    FRUEH_SCHNELL("早", "früh, schnell", "早い", "はやい", 248, 106,
            CollectionsHelper.buildListFrom("SOU", "[SA']"),
            CollectionsHelper.buildListFrom("haya(i)", "haya(maru)", "haya(meru)")
            ),
    SATZ_TEXT_LITERATUR_BRIEF("文", "Satz, Text, Literatur, Brief", "文学", "ぶんがく", 111, 86,
            CollectionsHelper.buildListFrom("BUN", "MON"),
            CollectionsHelper.buildListFrom("fumi")
            ),
    NORDEN("北", "Norden", "北", "きた", 73, 81,
            CollectionsHelper.buildListFrom("HOKU"),
            CollectionsHelper.buildListFrom("kita")
            ),
    OSTEN("東", "Osten", "東", "ひがし", 71, 81,
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("higashi")
            ),
    SUEDEN("南", "Süden", "南", "みなみ", 74, 81,
            CollectionsHelper.buildListFrom("NAN", "[NA]"),
            CollectionsHelper.buildListFrom("minami")
            ),
    WESTEN("西", "Westen", "西", "にし", 72, 81,
            CollectionsHelper.buildListFrom("SEI", "SAI"),
            CollectionsHelper.buildListFrom("nishi")
            ),
    NAME_RUF_REPUTATION("名", "Name, Ruf, Reputation", "名前", "なまえ", 82, 82,
            CollectionsHelper.buildListFrom("MEI", "MYOU"),
            CollectionsHelper.buildListFrom("na")
            ),
    SCHRIFTZEICHEN_BUCHSTABE("字", "Schriftzeichen, Buchstabe", "文字", "もじ", 110, 86,
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("aza")
            ),
    SCHULE_KORREKTUR("校", "Schule, Korrektur", "学校", "がっこう", 115, 87,
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom()
            ),
    RICHTIG_GERECHT("正", "richtig, gerecht", "正す", "ただす", 275, 110,
            CollectionsHelper.buildListFrom("SEI", "SHOU"),
            CollectionsHelper.buildListFrom("tada(shii)", "tada(su)", "masa(ni)")
            ),
    KOENIG("王", "König", "王様", "おうさま", 294, 112,
            CollectionsHelper.buildListFrom("OU"),
            CollectionsHelper.buildListFrom()
            ),
    TON_LAUT("音", "Ton, Laut", "音楽", "おんがく", 347, 120,
            CollectionsHelper.buildListFrom("ON", "IN"),
            CollectionsHelper.buildListFrom("oto", "ne")
            ),
    FADEN("糸", "Faden", "糸", "いと", 242, 105,
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("ito")
            ),
    MUSCHEL("貝", "Muschel", "貝", "かい", 240, 105,
            CollectionsHelper.buildListFrom(),
            CollectionsHelper.buildListFrom("kai")
            ),
    KUGEL_BALL("玉", "Edelstein, Juwel, Kugel, Ball", "玉", "たま", 295, 113,
            CollectionsHelper.buildListFrom("GYOKU"),
            CollectionsHelper.buildListFrom("tama")
            ),
    GRAS_KRAUT("草", "Gras, Kraut, Entwurf", "草", "くさ", 249, 106,
            CollectionsHelper.buildListFrom("SOU"),
            CollectionsHelper.buildListFrom("kusa")
            ),
    BAMBUS("竹", "Bambus", "竹", "たけ", 129, 89,
            CollectionsHelper.buildListFrom("CHIKU"),
            CollectionsHelper.buildListFrom("take")
            ),
    ZEHNTAUSEND("万", "10000, 10.000, zehntausend, Myriaden, viele , alle", "万", "まん", 16, 73,
            CollectionsHelper.buildListFrom("MAN", "BAN"),
            CollectionsHelper.buildListFrom()
            ),
    ZIEHEN_BILLIGER_MACHEN("引", "ziehen, anziehen, schließen, billiger machen", "引く", "ひく", 216, 101,
            CollectionsHelper.buildListFrom("IN"),
            CollectionsHelper.buildListFrom("hi(ku)")
            ),
    WOLKE("雲", "Wolke", "雲", "くも", 646, 163,
            CollectionsHelper.buildListFrom("UN"),
            CollectionsHelper.buildListFrom("kumo")
            ),
    WEIT_FERN_ENTFERNT("遠", "weit, fern, entfernt", "遠い", "とおい", 453, 135,
            CollectionsHelper.buildListFrom("EN", "[ON]"),
            CollectionsHelper.buildListFrom("too(i)")
            ),
    LEHRFACH("科", "Lehrfach, Abteilung, Fakultät, Schulfach", "科学", "かがく", 320, 116,
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom()
            ),
    SOMMER("夏", "Sommer", "夏", "なつ", 469, 137,
            CollectionsHelper.buildListFrom("KA", "[GE]"),
            CollectionsHelper.buildListFrom("natsu")
            ),
    HAN_CHINA("漢", "Han, China, Mann, Kerl", "漢字", "かんじ", 565, 151,
            CollectionsHelper.buildListFrom("KAN"),
            CollectionsHelper.buildListFrom()
            ),
    HAUPTSTADT("京", "Hauptstadt", "京都", "きょうと", 189, 97,
            CollectionsHelper.buildListFrom("KYOU", "KEI"),
            CollectionsHelper.buildListFrom()
            ),
    LICHT_HELL("明", "Licht, hell, Helligkeit", "明かり", "あかり", 18, 73,
            CollectionsHelper.buildListFrom("MEI", "MYOU"),
            CollectionsHelper.buildListFrom("a(kari)", "aka(rui)", "aki(raka)", "a(keru)",
                    "aka(rumu)", "aka(ramu)", "a(ku)", "a(kasu)", "a(kuru)")
            ),
    HAUS_HEIM("家", "Haus, Heim", "家", "いえ", 165, 94,
            CollectionsHelper.buildListFrom("KA", "KE"),
            CollectionsHelper.buildListFrom("ie", "ya")
            ),
    GEDICHT_LIED_SINGEN("歌", "Gedicht, Lied, singen", "歌", "うた", 395, 127,
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("uta", "uta(u)")
            ),
    BILD_STRICH("画", "Bild, Strich", "映画", "えいが", 343, 119,
            CollectionsHelper.buildListFrom("GA", "KAKU"),
            CollectionsHelper.buildListFrom()
            ),
    MAL_WIEDERKEHR_WEITERGABE_DREHEN("回", "Mal, Wiederkehr, Weitergabe, drehen", "一回", "いっかい",
            90, 83,
            CollectionsHelper.buildListFrom("KAI", "[E]"),
            CollectionsHelper.buildListFrom("mawa(su)", "mawa(ru)")
            ),
    MEER_SEE("海", "Meer, See", "海", "うみ", 117, 87,
            CollectionsHelper.buildListFrom("KAI"),
            CollectionsHelper.buildListFrom("umi")
            ),
    JEDER_JEDES("毎", "jeder, jeden, jedes", "毎日", "まいにち", 116, 87,
            CollectionsHelper.buildListFrom("MAI"),
            CollectionsHelper.buildListFrom()
            ),
    EINZELSTUECK("箇", "Einzelstück, Stück", "箇所", "かしょ", 1570, 295,
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom()
            ),
    EINZELSTUECK_KURZFORM("ヶ", "Einzelstück, Stück", "一ヶ月", "いっかげつ", 1570, 295,
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom()
            ),
    ICH_PRIVAT("私", "ich, privat", "私", "わたし", 125, 88,
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("watakushi", "watashi")
            ),
    KATZE("猫", "Katze", "猫", "ねこ", 1567, 294,
            CollectionsHelper.buildListFrom("BYOU"),
            CollectionsHelper.buildListFrom("neko")
            ),
    BAHNHOF("駅", "Bahnhof", "駅", "えき", 284, 111,
            CollectionsHelper.buildListFrom("EKI"),
            CollectionsHelper.buildListFrom()
            ),
    OEFFENTLICH_OFFIZIELL("公", "öffentlich, offiziell", "公園", "こうえん", 126, 88,
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("ooyake")
            ),
    GARTEN_PARK("園", "Garten, Park", "公園", "こうえん", 454, 135,
            CollectionsHelper.buildListFrom("EN"),
            CollectionsHelper.buildListFrom("sono")
            ),
    AUSSENSEITE_DRAUSSEN("外", "Außenseite, draußen", "外", "そと", 83, 82,
            CollectionsHelper.buildListFrom("GAI", "GE"),
            CollectionsHelper.buildListFrom("soto", "hoka", "hazu(reru)", "hazu(su)")
            ),
    ZEICHNUNG_PLAN("図", "Zeichnung, Plan, Skizze, planen", "図書館", "としょかん", 339, 119,
            CollectionsHelper.buildListFrom("ZU", "TO"),
            CollectionsHelper.buildListFrom("haka(ru)")
            ),
    GEBAUEDE_HALLE("館", "großes Gebäude, Gebäude, Halle", "図書館", "としょかん", 327, 117,
            CollectionsHelper.buildListFrom("KAN"),
            CollectionsHelper.buildListFrom("yakata")
            ),
    MOEGEN_LIEBEN("好", "gern haben, mögen, lieben", "好き", "すき", 104, 85,
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("kono(mu)", "su(ku)")
            ),
    TEE("茶", "Teestrauch, Teeblätter, Tee", "お茶", "おちゃ", 251, 106,
            CollectionsHelper.buildListFrom("CHA", "SA"),
            CollectionsHelper.buildListFrom()
            ),
    VOR_VORNE_FRUEHER("前", "vorne, vor, früher", "前", "まえ", 47, 77,
            CollectionsHelper.buildListFrom("ZEN"),
            CollectionsHelper.buildListFrom("mae")
            ),
    HINTER_DANACH_SPAETER("後", "danach, später, Rückseite, hinten, nach, hinter, zurück",
            "後ろ", "うしろ", 48, 77,
            CollectionsHelper.buildListFrom("GO", "KOU"),
            CollectionsHelper.buildListFrom("nochi", "ushi(ro)", "ato", "oku(reru)")
            ),
    SPRACHE_DIALEKT_UNTERSCHEIDUNG_BLATT("弁", "Sprache, Dialekt, Unterscheidung, Blatt",
            "弁当", "べんとう", 724, 174,
            CollectionsHelper.buildListFrom("BEN"),
            CollectionsHelper.buildListFrom()
            ),
    TREFFEN_ZUTREFFEN("当", "treffen, zutreffen", "弁当", "べんとう", 77, 81,
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("a(teru)", "a(taru)")
            ),
    GOTTHEIT_GOTT("神", "Gottheit, Gott", "神", "かみ", 310, 115,
            CollectionsHelper.buildListFrom("SHIN", "JIN"),
            CollectionsHelper.buildListFrom("kami", "[kan]", "[kou]")
            ),
    SHINTO_SCHREIN_GESELLSCHAFT_FIRMA("社", "Shinto-Schrein, Gesellschaft, Firma",
            "会社", "かいしゃ", 308, 114,
            CollectionsHelper.buildListFrom("SHA"),
            CollectionsHelper.buildListFrom("yashiro")
            ),
    MACHEN_ANFERTIGEN("作", "machen, anfertigen, zubereiten, herstellen", "作る", "べんとう", 361, 122,
            CollectionsHelper.buildListFrom("SAKU", "SA"),
            CollectionsHelper.buildListFrom("tsuku(ru)")
            ),
    SCHLOSS_BURG("城", "Schloss, Burg", "城", "しろ", 733, 175,
            CollectionsHelper.buildListFrom("JOU"),
            CollectionsHelper.buildListFrom("shiro")
            ),
    TEMPEL("寺", "Tempel", "寺", "てら", 41, 76,
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("tera")
            ),
    ANSTRENGUNG_HARTE_ARBEIT("勉", "Anstrengung, harte Arbeit", "勉強", "べんきょう", 748, 177,
            CollectionsHelper.buildListFrom("BEN"),
            CollectionsHelper.buildListFrom()
            ),
    STAR_STAERKER_WERDEN("強", "stark, stärker werden, zwingen", "勉強", "べんきょう", 217, 101,
            CollectionsHelper.buildListFrom("KYOU", "GOU"),
            CollectionsHelper.buildListFrom("tsuyo(i)", "tsuyo(maru)", "tsuyo(meru)", "shi(iru)")
            ),
    LEHREN_UNTERRICHTEN("教", "lehren, unterrichten, lernen", "教える", "おしえる", 245, 105,
            CollectionsHelper.buildListFrom("KYOU"),
            CollectionsHelper.buildListFrom("oshi(eru)", "oso(waru)")
            ),
    LAND_STAAT("国", "Land, Staat", "国", "くに", 40, 76,
            CollectionsHelper.buildListFrom("KOKU"),
            CollectionsHelper.buildListFrom("kuni")
            ),
    EINSTEIGEN("乗", "einsteigen, reiten, fahren", "乗る", "のる", 532, 146,
            CollectionsHelper.buildListFrom("JOU"),
            CollectionsHelper.buildListFrom("no(ru)", "no(seru)")
            ),
    STADT_MARKT("市", "Stadt, Markt", "市", "し", 181, 96,
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("ichi")
            ),
    PRAEFEKTUR("県", "Präfektur", "県", "けん", 194, 98,
            CollectionsHelper.buildListFrom("KEN"),
            CollectionsHelper.buildListFrom()
            ),
    GESICHT_MASKE_FLAECHE("面", "Gesicht, Maske, Fläche, Oberfläche, Seite", "面白い", "おもしろい",
            274, 110,
            CollectionsHelper.buildListFrom("MEN"),
            CollectionsHelper.buildListFrom("omote", "omo", "tsura")
            ),
    KOPF_URSPRUNG_ANFANG("元", "Kopf, Ursprung, Anfang", "元気", "げんき", 137, 90,
            CollectionsHelper.buildListFrom("GEN", "GAN"),
            CollectionsHelper.buildListFrom("moto")
            ),
    SEIN_EXISTIEREN("有", "sein, existieren, sich befinden, haben", "有名", "ゆうめい", 265, 108,
            CollectionsHelper.buildListFrom("YUU", "U"),
            CollectionsHelper.buildListFrom("a(ru)")
            ),
    SCHNEIDEN_ZU_ENDE_GEHEN("切", "schneiden, gut schneiden, zu Ende gehen", "大切", "たいせつ", 39, 76,
            CollectionsHelper.buildListFrom("SETSU", "[SAI]"),
            CollectionsHelper.buildListFrom("ki(ru)", "ki(eru)")
            ),
    UEBERSCHREITEN_UEBERGEBEN("渡", "überschreiten, übergeben", "渡す", "わたす", 379, 125,
            CollectionsHelper.buildListFrom("TO"),
            CollectionsHelper.buildListFrom("wata(ru)", "wata(su)")
            ),
    NEHMEN("取", "nehmen", "取る", "とる", 65, 80,
            CollectionsHelper.buildListFrom("SHU"),
            CollectionsHelper.buildListFrom("to(ru)")
            ),
    SETZEN_LEGEN_DALSSSEN("置", "setzen, stellen, legen, dalassen, lassen, errichten", "置く", "おく",
            432, 132,
            CollectionsHelper.buildListFrom("CHI"),
            CollectionsHelper.buildListFrom("o(ku)")
            ),
    SCHNEE("雪", "Schnee", "雪", "ゆき", 979, 210,
            CollectionsHelper.buildListFrom("SETSU"),
            CollectionsHelper.buildListFrom("yuki")
            ),
    FALLEN("降", "fallen, herabsteigen, aussteigen, regnen, schneien", "降る", "ふる", 977, 210,
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("o(riru)", "o(rosu)", "fu(ru)")
            ),
    WIND("風", "Wind, Aussehen, Mode, Stil", "風", "かぜ", 29, 75,
            CollectionsHelper.buildListFrom("FUU", "[FU]"),
            CollectionsHelper.buildListFrom("kaze", "[kaza]")
            ),
    BLASEN_WEHEN("吹", "blasen, wehen", "吹く", "ふく", 1326, 260,
            CollectionsHelper.buildListFrom("SUI"),
            CollectionsHelper.buildListFrom("fu(ku)")
            ),
    HEISS__WETTER_("暑", "heiß, heiß (Wetter)", "暑", "あつい", 648, 163,
            CollectionsHelper.buildListFrom("SHO"),
            CollectionsHelper.buildListFrom("atsu(i)")
            ),
    HITZE_FIEBER_HEISS("熱", "Hitze, Fieber, heiß, heiß (anfassbar)", "熱い", "あつい", 655, 164,
            CollectionsHelper.buildListFrom("NETSU"),
            CollectionsHelper.buildListFrom("atsu(i)")
            ),
    KAELTE_WINTER_KALT("寒", "Kälte, kalte Jahreszeit, Winter, kalt", "寒い", "さむい", 464, 137,
            CollectionsHelper.buildListFrom("KAN"),
            CollectionsHelper.buildListFrom("samu(i)")
            ),
    KALT_KUEHLEN_KALT_WERDEN("冷", "kalt, kühlen, kalt werden, kaltes Wasser, kalter Sake",
            "冷たい", "おく", 851, 192,
            CollectionsHelper.buildListFrom("REI"),
            CollectionsHelper.buildListFrom("tsume(tai)", "hi(yasu)", "sa(masu)", "hi(eru)",
                    "sa(meru)", "hi(ya)", "hi(yakasu)")
            ),
    WARM_WARM_WERDEN_SICH_ERWAERMEN__WETTER__("暖",
            "warm, erwärmen, aufwärmen, warm werden, sich erwärmen, (warmes Wetter)",
            "暖かい", "あたたかい", 645, 163,
            CollectionsHelper.buildListFrom("DAN"),
            CollectionsHelper.buildListFrom("atata(kai)", "atata(ka)", "atata(meru)", "atata(maru)")
            ),
    WARM_WARM_WERDEN_SICH_ERWAERMEN__ANFASSEN__("温",
            "warm, erwärmen, aufwärmen, warm werden, sich erwärmen, (warm beim Anfassen)",
            "温かい", "あたたかい", 644, 162,
            CollectionsHelper.buildListFrom("ON"),
            CollectionsHelper.buildListFrom("atata(kai)", "atata(ka)", "atata(meru)", "atata(maru)")
            ),
    KUEHL_FRISCH_ERFRISCHEND("涼", "kühl, angenehm kühl, frisch, erfrischend, "
            + "sich draußen erfrischen, frische Luft schöpfen", "涼しい", "すずしい", 1267, 251,
            CollectionsHelper.buildListFrom("RYOU"),
            CollectionsHelper.buildListFrom("suzu(shii)", "suzu(mu)")
            ),
    SCHLECHT_SCHLIMM_BOESE("悪", "schlecht, schlimm, böse", "悪い", "わるい", 304, 114,
            CollectionsHelper.buildListFrom("AKU", "O"),
            CollectionsHelper.buildListFrom("waru(i)")
            ),
    PFERD("馬", "Pferd", "馬", "うま", 283, 111,
            CollectionsHelper.buildListFrom("BA"),
            CollectionsHelper.buildListFrom("uma", "[ma]")
            ),
    VOGEL("鳥", "Vogel", "鳥", "とり", 285, 111,
            CollectionsHelper.buildListFrom("CHOU"),
            CollectionsHelper.buildListFrom("tori")
            ),
    RIND_KUH("牛", "Rind, Kuh", "牛", "うし", 281, 111,
            CollectionsHelper.buildListFrom("GYUU"),
            CollectionsHelper.buildListFrom("ushi")
            ),
    SCHWEIN_HAUSSCHWEIN("豚", "Schwein, Hausschwein", "豚", "ぶた", 810, 186,
            CollectionsHelper.buildListFrom("TON"),
            CollectionsHelper.buildListFrom("buta")
            ),
    BEWEGUNG_SICH_BEWEGEN("動", "Bewegung, sich bewegen, bewegen", "動物", "どうぶつ", 231, 103,
            CollectionsHelper.buildListFrom("DOU"),
            CollectionsHelper.buildListFrom("ugo(ku)", "ugo(kasu)")
            ),
    DING_SACHE_GEGENSTAND("物", "Ding, Sache, Gegenstand", "動物", "どうぶつ", 79, 82,
            CollectionsHelper.buildListFrom("BUTSU", "MOTSU"),
            CollectionsHelper.buildListFrom("mono")
            ),
    MUSIK_BEQUEMLICHKEIT_ERFREULICH("楽",
            "Musik, Bequemlichkeit, sich freuen, genießen, froh, erfreulich",
            "楽しい", "たのしい", 359, 122,
            CollectionsHelper.buildListFrom("GAKU", "RAKU"),
            CollectionsHelper.buildListFrom("tano(shimu)", "tano(shii)")
            ),
    SCHWIERIG("難", "schwierig", "難しい", "むずかしい", 566, 151,
            CollectionsHelper.buildListFrom("NAN"),
            CollectionsHelper.buildListFrom("muzuka(shii)", "kata(i)")
            ),
    BEQUEMLICHKEIT_EXKREMENTE_POST("便",
            "Bequemlichkeit, Exkremente, Gelegenheit, Post, Nachricht, Brief",
            "便利", "べんり", 330, 118,
            CollectionsHelper.buildListFrom("BEN", "BIN"),
            CollectionsHelper.buildListFrom("tayo(ri)")
            ),
    VORTEIL_ZINS_WIRKEN("利", "Vorteil, Zins, wirksam sein, wirken", "便利", "べんり", 329, 117,
            CollectionsHelper.buildListFrom("RI"),
            CollectionsHelper.buildListFrom("ki(ku)")
            ),
    UN_NICHT("不", "un-, nicht", "不便", "ふべん", 94, 84,
            CollectionsHelper.buildListFrom("FU", "BU"),
            CollectionsHelper.buildListFrom()
            ),
    WACHE_DIENST_NUMMER("番", "Wache, Aufsicht, Dienst, Reihe, Nummer", "番号", "ばんごう", 185, 97,
            CollectionsHelper.buildListFrom("BAN"),
            CollectionsHelper.buildListFrom()
            ),
    NUMMER_PSEUDONYM("号", "Nummer, Pseudonym, verzweifeltes Schreien", "番号", "ばんごう", 266, 108,
            CollectionsHelper.buildListFrom("GOU"),
            CollectionsHelper.buildListFrom()
            ),
    BENUTZUNG_BOTE_BENUTZEN("使", "Benutzung, Bote, benutzen", "使う", "つかう", 331, 118,
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("tsuka(u)")
            ),
    SELBST("自", "selbst", "自転車", "じてんしゃ", 62, 79,
            CollectionsHelper.buildListFrom("JI", "SHI"),
            CollectionsHelper.buildListFrom("mizuka(ra)")
            ),
    ROLLEN_FALLEN_WAELZEN("転", "rollen, fallen, wälzen, hinfallen, umwerfen",
            "自転車", "じてんしゃ", 439, 133,
            CollectionsHelper.buildListFrom("TEN"),
            CollectionsHelper.buildListFrom("koro(bu)", "koro(garu)", "koro(geru)", "koro(gasu)")
            ),
    SCHWERT_MESSER("刀", "Schwert, Messer", "刀", "かたな", 37, 76,
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("katana")
            ),
    VATER("父", "Vater", "父", "ちち", 113, 87,
            CollectionsHelper.buildListFrom("FU"),
            CollectionsHelper.buildListFrom("chichi")
            ),
    MUTTER("母", "Mutter", "母", "はは", 112, 86,
            CollectionsHelper.buildListFrom("BO"),
            CollectionsHelper.buildListFrom("haha")
            ),
    GROSS_STARK_STATUR_GROESSE("丈", "groß, stark, Statur, Größe", "大丈夫", "だいじょうぶ", 1408, 272,
            CollectionsHelper.buildListFrom("JOU"),
            CollectionsHelper.buildListFrom("take")
            ),
    EHEMANN_MANN("夫", "Ehemann, Mann", "大丈夫", "だいじょうぶ", 315, 115,
            CollectionsHelper.buildListFrom("FU", "[FUU]"),
            CollectionsHelper.buildListFrom("otto")
            ),
    BITTEN_BETEN_WUENSCHEN("願", "bitten, beten, wünschen", "お願いします", "お願いします", 590, 155,
            CollectionsHelper.buildListFrom("GAN"),
            CollectionsHelper.buildListFrom("nega(u)")
            ),
    ORT_STELLE("所", "Ort, Stelle", "所", "ところ", 153, 92,
            CollectionsHelper.buildListFrom("SHO"),
            CollectionsHelper.buildListFrom("tokoro")
            ),
    STRASSE_WEG_PFAD("道", "Straße, Weg, Pfad, Dao/Tao", "道", "みち", 149, 92,
            CollectionsHelper.buildListFrom("DOU", "[TOU]"),
            CollectionsHelper.buildListFrom("michi")
            ),
    ZURUECKKEHREN("帰", "zurückkehren, nach Hause schicken", "帰る", "かえる", 317, 116,
            CollectionsHelper.buildListFrom("KI"),
            CollectionsHelper.buildListFrom("kae(ru)", "kae(su)")
            ),
    BAUEN_ERRICHTEN("建", "bauen, errichten, errichtet werden", "建物", "たてもの", 913, 201,
            CollectionsHelper.buildListFrom("KEN", "[KON]"),
            CollectionsHelper.buildListFrom("ta(teru)", "ta(tsu)")
            ),
    STOCKWERK_ETAGE_EBENE("階", "Stockwerk, Etage, Ebene", "階", "かい", 597, 156,
            CollectionsHelper.buildListFrom("KAI"),
            CollectionsHelper.buildListFrom()
            ),
    ERDE_LAND("地", "Erde, Land", "地下", "ちか", 118, 87,
            CollectionsHelper.buildListFrom("CHI", "JI"),
            CollectionsHelper.buildListFrom()
            ),
    BLATT_LAUB("葉", "Blatt, Laub", "言葉", "ことば", 253, 107,
            CollectionsHelper.buildListFrom("YOU"),
            CollectionsHelper.buildListFrom("ha")
            ),
    GLAENZEND_BEGABT_ENGLAND("英", "glänzend, brillant, begabt, England", "英語", "えいご", 354, 121,
            CollectionsHelper.buildListFrom("EI"),
            CollectionsHelper.buildListFrom()
            ),
    QUALITAET_NATUR_PFAD("質", "Qualität, Natur, Pfad", "質問", "しつもん", 176, 96,
            CollectionsHelper.buildListFrom("SHITSU", "SHICHI", "[CHI]"),
            CollectionsHelper.buildListFrom()
            ),
    FRAGE_PROBLEM_FRAGEN("問", "Frage, Problem, fragen, sich kümmern (um)", "質問", "しつもん", 162, 94,
            CollectionsHelper.buildListFrom("MON"),
            CollectionsHelper.buildListFrom("to(i)", "[tou]", "to(u)")
            ),
    ANTWORT_ANTWORTEN("答", "Antwort, antworten", "答え", "こたえ", 160, 93,
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("kota(e)", "kota(eru)")
            ),
    ER_JENER("彼", "er, jener", "彼", "かれ", 1010, 215,
            CollectionsHelper.buildListFrom("HI"),
            CollectionsHelper.buildListFrom("kare", "[kano]")
            ),
    FARBE_EROTIK("色", "Farbe, Erotik", "色", "いろ", 204, 100,
            CollectionsHelper.buildListFrom("SHOKU", "SHIKI"),
            CollectionsHelper.buildListFrom("iro")
            ),
    KANJI_WIEDERHOLUNGSZEICHEN("々", "Wiederholungszeichen, Kanji Wiederholungszeichen", "々", "々",
            0, 0,
            CollectionsHelper.buildListFrom(),
            CollectionsHelper.buildListFrom()
            ),
    FRUEHLING("春", "Frühling, Frühjahr", "春", "はる", 468, 137,
            CollectionsHelper.buildListFrom("SHUN"),
            CollectionsHelper.buildListFrom("haru")
            ),
    HERBST("秋", "Herbst", "秋", "あき", 470, 138,
            CollectionsHelper.buildListFrom("SHUU"),
            CollectionsHelper.buildListFrom("aki")
            ),
    WINTER("冬", "Winter", "冬", "ふゆ", 467, 137,
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("fuyu")
            ),
    ;


    /** Das Kanji-Zeichen in UTF-8 Darstellung. */
    private final String character;

    /** Die deutsche Bedeutung. */
    private final String germanMeaning;

    /** Das Wort in Kanji und Hiragana in der häufigsten Bedeutung. */
    private final String normalKanjiHiraganaWord;

    /** Das Wort in Hiragana in der häufigsten Bedeutung. */
    private final String normalHiraganaWord;

    /** Die Nummer des Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky. */
    private final int numberInKanjiAndKanaBook;

    /**
     * Die Seite auf der das Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky zu finden ist.
     */
    private final int pageNumberInKanjiAndKanaBook;

    /** Die Liste der sinojapanischen ON-Lesungen des Kanji (groß geschrieben). */
    private final List<String> onLesungen;

    /** Die Liste der rein japanischen kun-Lesungen des Kanji (klein geschrieben). */
    private final List<String> kunLesungen;

    /**
     * Konstruktor.
     *
     * @param character
     *            Das Kanji-Zeichen in UTF-8 Darstellung.
     * @param germanMeaning
     *            Die deutsche Bedeutung.
     * @param normalKanjiHiraganaWord
     *            Das Wort in Kanji und Hiragana in der häufigsten Bedeutung.
     * @param normalHiraganaWord
     *            Das Wort in Hiragana in der häufigsten Bedeutung.
     * @param numberInKanjiAndKanaBook
     *            Die Nummer des Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky.
     * @param pageNumberInKanjiAndKanaBook
     *            Die Seite auf der das Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky zu
     *            finden ist.
     * @param onLesungen
     *            Die Liste der sinojapanischen ON-Lesungen des Kanji (groß geschrieben).
     * @param kunLesungen
     *            Die Liste der rein japanischen kun-Lesungen des Kanji (klein geschrieben).
     */
    private Kanji(String character, String germanMeaning, String normalKanjiHiraganaWord,
            String normalHiraganaWord, int numberInKanjiAndKanaBook,
            int pageNumberInKanjiAndKanaBook, List<String> onLesungen, List<String> kunLesungen) {
        this.character = character;
        this.germanMeaning = germanMeaning;
        this.normalKanjiHiraganaWord = normalKanjiHiraganaWord;
        this.normalHiraganaWord = normalHiraganaWord;
        this.numberInKanjiAndKanaBook = numberInKanjiAndKanaBook;
        this.pageNumberInKanjiAndKanaBook = pageNumberInKanjiAndKanaBook;
        this.onLesungen = onLesungen;
        this.kunLesungen = kunLesungen;
    }

    /** Getter für das Kanji-Zeichen in UTF-8 Darstellung. */
    public String getCharacter() {
        return character;
    }

    /** Getter für die deutsche Bedeutung. */
    public String getGermanMeaning() {
        return germanMeaning;
    }

    /** Getter für das Wort in Kanji und Hiragana in der häufigsten Bedeutung. */
    public String getNormalKanjiHiraganaWord() {
        return normalKanjiHiraganaWord;
    }

    /** Getter für das Wort in Hiragana in der häufigsten Bedeutung. */
    public String getNormalHiraganaWord() {
        return normalHiraganaWord;
    }

    /** Getter für die Nummer des Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky. */
    public int getNumberInKanjiAndKanaBook() {
        return numberInKanjiAndKanaBook;
    }

    /**
     * Getter für die Seite auf der das Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky zu
     * finden ist.
     */
    public int getPageNumberInKanjiAndKanaBook() {
        return pageNumberInKanjiAndKanaBook;
    }

    /** Getter für die Liste der sinojapanischen ON-Lesungen des Kanji (groß geschrieben). */
    public List<String> getOnLesungen() {
        return onLesungen;
    }

    /** Getter für die Liste der rein japanischen kun-Lesungen des Kanji (klein geschrieben). */
    public List<String> getKunLesungen() {
        return kunLesungen;
    }

    /** Gibt eine Liste mit den bekannten Kanji zurück. */
    public static List<Kanji> getAllKanjiAsList() {
        return CollectionsHelper.arrayToList(Kanji.values());
    }

    /** Gibt eine Liste mit den Zeichen der Schrift Kanji (in UTF8-Zeichen) zurück. */
    public static List<String> getAllKanjiAsStringList() {
        List<String> kanjiList = new ArrayList<>();

        for (Kanji kanji : Kanji.values()) {
            kanjiList.add(kanji.getCharacter());
        }

        return kanjiList;
    }

    /** Gibt an, ob in dem Text ein Kanji-Zeichen vorkommt. */
    public static boolean containsKanji(String text) {
        return Text.contains(text, getAllKanjiAsStringList());
    }

    /** Gibt an, ob der Text ausschließlich aus Kanji-Zeichen besteht. */
    public static boolean containsOnlyKanji(String text) {
        if (text.isEmpty()) {
            return false;
        }
        else {
            List<String> kanji = getAllKanjiAsStringList();
            for (String textCharacter : Text.textToCharactersList(text)) {
                if (!kanji.contains(textCharacter)) {
                    return false;
                }
            }

            return true;
        }
    }

    /**
     * Gibt den Index des ersten Kanji-Zeichen im Text zurück. Gibt es kein solches, wird -1
     * zurückgegeben.
     */
    public static int firstIndexOfKanji(String text) {
        List<String> kanji = getAllKanjiAsStringList();
        List<String> textCharacters = Text.textToCharactersList(text);

        for (int index = 0; index < textCharacters.size(); ++index) {
            String textCharacter = textCharacters.get(index);
            if (kanji.contains(textCharacter)) {
                return index;
            }
        }

        return -1;
    }

    /**
     * Gibt den Index des letzten Kanji-Zeichen im Text zurück. Gibt es kein solches, wird -1
     * zurückgegeben.
     */
    public static int lastIndexOfKanji(String text) {
        List<String> kanji = getAllKanjiAsStringList();
        List<String> textCharacters = Text.textToCharactersList(text);

        for (int index = textCharacters.size() - 1; index >= 0; --index) {
            String textCharacter = textCharacters.get(index);
            if (kanji.contains(textCharacter)) {
                return index;
            }
        }

        return -1;
    }

    /** Gibt die Anzahl der bekannten Kanji zurück. */
    public static int getNumberOfKnownKanji() {
        return Kanji.values().length;
    }

    /** Ermittelt das Kanji zu einem Kanji-Charakter. */
    public static Kanji getByCharacter(String character) {
        for (Kanji kanji : Kanji.values()) {
            if (character.equals(kanji.getCharacter())) {
                return kanji;
            }
        }

        throw new RuntimeException("Kein Kanji zum Zeichen '" + character + "' gefunden.");
    }

    /**
     * Gibt eine Liste mit den im Text enthaltenen Kanji zurück.
     *
     * Die Kanji kommen in der Liste in der Reihenfolge ihres Vorkommens im Text vor und sind auch
     * mehrfach enthalten, wenn sie mehrfach im Text vorkommen.
     */
    public static List<Kanji> getAllKanjiFromText(String text) {
        List<Kanji> kanjiInText = new ArrayList<>();

        for (String textCharacter : Text.textToCharactersList(text)) {
            for (Kanji kanji : Kanji.values()) {
                String kanjiCharacter = kanji.getCharacter();
                if (textCharacter.equals(kanjiCharacter)) {
                    kanjiInText.add(kanji);
                }
            }
        }

        return kanjiInText;
    }

}
