package de.duehl.vocabulary.japanese.logic.success;

import java.awt.Color;
import java.util.List;

import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalVocableData;
import de.duehl.vocabulary.japanese.common.data.TranslationDirection;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;

/**
 * Diese Klasse ermittelt den Erfolg bei den letzten zehn Abfragen eines Vokabulars und erzeugt
 * daraus die passende Farbe sowie die Prozentzahl zur Anzeige in der VocabularyBar.
 *
 * @version 1.01     2024-07-30
 * @author Christian Dühl
 */

public class VocabularyTestSuccesssCalculator {

    /** Die Liste mit den Vokabeln, zu welchen der Erfolg berechnet werden soll. */
    private final List<Vocable> vocables;

    /** Die Programmoptionen. */
    private final Options options;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalDataRequester requester;

    /** Die Anzahl der maximal zehn Abfragen aller Vokabeln des Vokabulars. */
    private int lastCountSum;

    /** Die Anzahl der korrekten maximal zehn Abfragen aller Vokabeln des Vokabulars. */
    private int lastCorrectCountSum;

    /** Die Vordergrundfarbe für den Namen des Vokabulars in der VocabularyBar. */
    private Color foregroundColor;

    /** Der Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar. */
    private String percentText;

    /** Die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    private double percent;

    /**
     * Der Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur Anzeige in der
     * VocabularyBar.
     */
    private String percentTextWithTwoDigitsAfterComma;

    /**
     * Konstruktor.
     *
     * @param vocabulary
     *            Das Vokabular zu welchem der Erfolg berechnet werden soll.
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     */
    public VocabularyTestSuccesssCalculator(Vocabulary vocabulary, Options options,
            InternalDataRequester requester) {
        this(vocabulary.getVocables(), options, requester);
    }

    /**
     * Konstruktor.
     *
     * @param vocables
     *            Die Liste mit den Vokabeln,  zu welchen der Erfolg berechnet werden soll.
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     */
    public VocabularyTestSuccesssCalculator(List<Vocable> vocables, Options options,
            InternalDataRequester requester) {
        this.vocables = vocables;
        this.options = options;
        this.requester = requester;
    }

    /** Führt die Berechnung durch. */
    public void calculate() {
        calculateCounts();
        foregroundColor = determineForegroundColor();
        determinePercentAndPercentText();
    }

    private void calculateCounts() {
        lastCountSum = 0;
        lastCorrectCountSum = 0;

        for (Vocable vocable : vocables) {
            calculateCountsForVocable(vocable);
        }
    }

    private void calculateCountsForVocable(Vocable vocable) {
        InternalAdditionalVocableData data = requester.getInternalDataForVocable(vocable);
        int lastCount;
        int lastCorrectCount;
        TranslationDirection translationDirection = options.getTranslationDirection();
        if (translationDirection == TranslationDirection.JAPANESE_TO_GERMAN) {
            lastCount = data.getLastTenJapaneseToGermanTestsCount();
            lastCorrectCount = data.getLastCorrectJapaneseToGermanTestsCount();
        }
        else if (translationDirection == TranslationDirection.GERMAN_TO_JAPANESE) {
            lastCount = data.getLastGermanToJapaneseTestsCount();
            lastCorrectCount = data.getLastCorrectGermanToJapaneseTestsCount();
        }
        else {
            throw new RuntimeException("Unbekannte Übersetzungsrichtung " + translationDirection);
        }
        lastCountSum += lastCount;
        lastCorrectCountSum += lastCorrectCount;

        /* Damit nicht abgefragte Vokabeln (und Vokabularien) nicht gut aussehen: */
        if (lastCount == 0) {
            lastCountSum += 10;
        }
    }

    private Color determineForegroundColor() {
        if (options.isColorVocabularyDependingOnLastSuccess()) {
            VocableColors vocableColors = new VocableColors(options);
            return vocableColors.determineForegroundColor(lastCountSum, lastCorrectCountSum);
        }
        else {
            return Color.BLACK;
        }
    }

    private void determinePercentAndPercentText() {
        percentTextWithTwoDigitsAfterComma = NumberString.percent(lastCorrectCountSum, lastCountSum);

        percent = NumberString.percentAsNumber(lastCorrectCountSum, lastCountSum);

        String percentString = String.format("%.0f", percent);
        percentText = Text.fillWithSpacesAtFront(percentString, 3);
    }

    /** Getter für die Vordergrundfarbe für den Namen des Vokabulars in der VocabularyBar. */
    public Color getForegroundColor() {
        return foregroundColor;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar.
     */
    public String getPercentText() {
        return percentText;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur
     * Anzeige in der VocabularyBar.
     */
    public String getPercentTextWithTwoDigitsAfterComma() {
        return percentTextWithTwoDigitsAfterComma;
    }

    /** Getter für die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    public double getPercent() {
        return percent;
    }

}
