package de.duehl.vocabulary.japanese.startup.ui;

import java.awt.BorderLayout;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JTextArea;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.NonModalFrameDialogBase;

/**
 * Diese Klasse ist die grafische Oberfläche beim Startup (vgl. Klasse StartUp), diese lädt beim
 * Starten des Vokabeltrainers die Vokabulare ein, sucht die Sound-Files und stellt die Verbindung
 * zu den benutzerabhängigen, internen Daten jeder Variablen her.
 *
 * Hier wird das ganze grafisch dargestellt und die angezeigten Informationen gesammelt, damit man
 * diese später über das Menü abfragen kann.
 *
 * @version 1.01     2024-10-26
 * @author Christian Dühl
 */

public class SplashScreen extends NonModalFrameDialogBase implements MessageAppender {

    /** Der StringBuilder für den Text, der beim Starten des Vokabeltrainers angezeigt wird. */
    private final StringBuilder splashTextBuilder;

    /** Das Textfeld, wo die Daten über den Einleseprozess angezeigt werden. */
    private final JTextArea textArea;

    /**
     * Konstruktor.
     *
     * @param programImage
     *            Das Icon für das Programm.
     */
    public SplashScreen(Image programImage) {
        super(new Point(150, 100), programImage, "Start des Vokabeltrainers");

        splashTextBuilder = new StringBuilder();
        textArea = new JTextArea();

        fillDialog();
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        initElements();

        // Später das Bild ...

        add(GuiTools.createScrollPane(textArea), BorderLayout.CENTER);
    }

    private void initElements() {
        initTextArea();
    }

    private void initTextArea() {
        textArea.setRows(45);
        textArea.setColumns(120);
        textArea.setEditable(false);
    }

    /** Fügt eine Nachricht hinzu. */
    @Override
    public void appendMessage(String message) {
        appendMessageToSplashTextBuilder(message);
        appendMessageToTextArea(message);
    }

    private void appendMessageToSplashTextBuilder(String message) {
        splashTextBuilder.append(message).append("\n");
    }

    private void appendMessageToTextArea(String message) {
        try {
            tryToAppendMessageToTextArea(message);
        }
        catch (BadLocationException e) {
            throw new RuntimeException("Kann die Nachricht [" + message + "] nicht an das Ende "
                    + "des Dokuments der JTextArea des SplashScreens anhängen.");
        }
    }

    private void tryToAppendMessageToTextArea(String message) throws BadLocationException {
        Document doc = textArea.getDocument();
        doc.insertString(doc.getLength(), message + "\n", null);
        textArea.setCaretPosition(textArea.getDocument().getLength());
    }

    /** Getter für den Text, der beim Starten des Vokabeltrainers angezeigt wurde. */
    public String getSplashText() {
        return splashTextBuilder.toString();
    }

}
