package de.duehl.vocabulary.japanese.ui.components;

import java.awt.BorderLayout;
import java.awt.CardLayout;

import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.vocabulary.japanese.common.data.TranslationDirection;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.common.ui.resources.IconDefinitions;
import de.duehl.vocabulary.japanese.ui.VocabularyTrainerGui;
import de.duehl.vocabulary.japanese.ui.tools.VocabularyTrainerUiTools;

/**
 * Diese Klasse stellt die grafische Oberfläche der Auswahl der Übersetzungsrichtung im
 * Vokabel-Trainers dar.
 *
 * @version 1.01     2024-08-07
 * @author Christian Dühl
 */

public class TranslationDirectionGuiWithSwitchButton {

    private static final String GERMAN_JAPANESE_ACTIVE_NAME = "GERMAN_JAPANESE_ACTIVE_NAME";
    private static final String JAPANESE_GERMAN_ACTIVE_NAME = "JAPANISCH_DEUTSCH_AKTIV_NAME";


    /** Die Programmoptionen. */
    private final Options options;

    /** Die grafische Oberfläche des Vokabel-Trainers. */
    private final VocabularyTrainerGui gui;

    /** Der Panel auf dem die Auswahl dargestellt wird. */
    private final JPanel panel;

    /**
     * Der Panel der entweder den Panel mit den Buttons zum Ansehen und Anfragen des Vokabulars
     * oder aber den Panel mit Buttons zum Verschieben nach oben und unten anzeigen kann.
     */
    private final JPanel switchingContentButtonPanel;

    /**
     * Das Layout für den Panel mit den wechselnden Inhalten (switchingContentButtonPanel mit den
     * beiden Buttons).
     */
    private final CardLayout cardLayout;

    /** Der Button für die Übersetzung von Japanisch in Deutsch. */
    private final JButton japaneseToGermanTranslationButton;

    /** Der Button für die Übersetzung von Japanisch in Deutsch. */
    private final JButton germanToJapaneseTranslationButton;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     * @param gui
     *            Die grafische Oberfläche des Vokabel-Trainers.
     */
    public TranslationDirectionGuiWithSwitchButton(Options options, VocabularyTrainerGui gui) {
        this.options = options;
        this.gui = gui;

        panel = new JPanel();

        japaneseToGermanTranslationButton = createUmschalterJapanischDeutschIsActiveButton();
        germanToJapaneseTranslationButton = createUmschalterDeutschJapanischIsActiveButton();
        switchingContentButtonPanel = new JPanel();
        cardLayout = new CardLayout();

        init();
        switchToCorrectButtonDependingOnOptions();
    }

    private JButton createUmschalterJapanischDeutschIsActiveButton() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.UMSCHALTER_JAPANISCH_DEUTSCH,
                e -> switchToDeutschJapanisch(),
                "Zur Übersetzungsrichtung Deutsch-Japanisch umschalten");
        return button;
    }

    private JButton createUmschalterDeutschJapanischIsActiveButton() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.UMSCHALTER_DEUTSCH_JAPANISCH,
                e -> switchToJapanischDeutsch(),
                "Zur Übersetzungsrichtung Japanisch-Deutsch umschalten");
        return button;
    }

    private void switchToDeutschJapanisch() {
        TranslationDirection direction = TranslationDirection.GERMAN_TO_JAPANESE;
        String cardName = GERMAN_JAPANESE_ACTIVE_NAME;
        switchCardAndActivateTranslationDirection(direction, cardName);
    }

    private void switchToJapanischDeutsch() {
        TranslationDirection direction = TranslationDirection.JAPANESE_TO_GERMAN;
        String cardName = JAPANESE_GERMAN_ACTIVE_NAME;
        switchCardAndActivateTranslationDirection(direction, cardName);
    }

    private void switchCardAndActivateTranslationDirection(TranslationDirection direction,
            String cardName) {
        showButton(cardName);

        options.setTranslationDirection(direction);
        gui.showTranslationDirectionOnBarButtons();
        gui.setCorrectForegroundColorOfVocabularyBarsLater();
    }

    private void showButton(String cardName) {
        cardLayout.show(switchingContentButtonPanel, cardName);
        panel.repaint();
    }

    private void init() {
        initSwitchingButtonCardPane();
        initPanel();
    }

    private void initSwitchingButtonCardPane() {
        switchingContentButtonPanel.setLayout(cardLayout);
        switchingContentButtonPanel.add(japaneseToGermanTranslationButton,
                JAPANESE_GERMAN_ACTIVE_NAME);
        switchingContentButtonPanel.add(germanToJapaneseTranslationButton,
                GERMAN_JAPANESE_ACTIVE_NAME);
    }

    private void initPanel() {
        panel.setLayout(new BorderLayout());
        //GuiTools.createTitle(panel);

        panel.add(switchingContentButtonPanel, BorderLayout.CENTER);

        // Eigentlich braucht man den Panel gar nicht mehr ...
    }

    private void switchToCorrectButtonDependingOnOptions() {
        TranslationDirection direction = options.getTranslationDirection();

        if (direction == TranslationDirection.JAPANESE_TO_GERMAN) {
            showButton(JAPANESE_GERMAN_ACTIVE_NAME);
        }
        else if (direction == TranslationDirection.GERMAN_TO_JAPANESE) {
            showButton(GERMAN_JAPANESE_ACTIVE_NAME);
        }
        else {
            throw new RuntimeException("Unbekannte TranslationDirection " + direction + ".");
        }
    }

    /** Getter für den Panel auf dem die Auswahl dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

}
