package de.duehl.vocabulary.japanese.ui.dialog.detail.related;

import java.awt.Image;
import java.awt.Point;

import javax.swing.JPanel;

import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.grammar.AdjectivePositivePresenceFinder;
import de.duehl.vocabulary.japanese.grammar.VerbWoerterbuchformFinder;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.ui.data.MessageSetter;

/**
 * Diese Klasse stellt Buttons bereit, um zu einer Vokabel zugehörige Verb- und Adjektivformen zu
 * finden, wenn die Vokabel ein Verb bzw. ein Adjektiv ist.
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

public class VerbsAndAdjectivesFinderGui {

    /** Die angezeigte Vokabel. */
    private final Vocable vocable;

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /** Findet zur angezeigten Vokabel gehörige Verb- und Adjektivformen. */
    private final VerbsAndAdjectivesFinder verbsAndAdjectivesFinder;

    /** Stellt die Buttons wirklich dar. */
    private final VerbAndAdjectiveButtonsCreator creator;

    /**
     * Konstruktor.
     *
     * @param vocable
     *            Die angezeigte Vokabel.
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param ownLists
     *            Die Verwaltung der eigenen Vokabellisten.
     * @param kanjiRequester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param messageSetter
     *            Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann.
     * @param location
     *            Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese
     *            Klasse verwendet.
     * @param programImage
     *            Das anzuzeigende ProgrammIcon.
     * @param longTimeProcessInformer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     */
    public VerbsAndAdjectivesFinderGui(Vocable vocable, Options options,
            InternalDataRequester requester, OwnLists ownLists,
            InternalKanjiDataRequester kanjiRequester, MessageSetter messageSetter, Point location,
            Image programImage, LongTimeProcessInformer longTimeProcessInformer) {
        this.vocable = vocable;
        this.ownLists = ownLists;

        verbsAndAdjectivesFinder = new VerbsAndAdjectivesFinder(options,
                requester, ownLists, kanjiRequester, messageSetter, location, programImage,
                longTimeProcessInformer);
        creator = new VerbAndAdjectiveButtonsCreator(vocable, verbsAndAdjectivesFinder);
    }

    /**
     * Fügt zum übergebenen Panel Buttons zur Anzeige der Verb- bzw. Adjektivformen an, abhängig
     * davon, ob die Vokabel ein Verb bzw. Adjektiv ist.
     *
     * @param panel
     *            Der Panel dem die Buttons hinzugefügt werden sollen. Dieser sollte ein Layout
     *            haben, das das Hinzufügen von Elemente ohne weitere Parameter gestattet, wie z.B.
     *            das VerticalLayout.
     */
    public void appendButtonsToPanel(JPanel panel) {
        if (vocable.isVerbInWoerterbuchform()) {
            panel.add(creator.createFindRelatedVerbsButton());
        }
        else if (vocable.isVerb()) {
            Vocable woerterbuchform = VerbWoerterbuchformFinder.determineWoerterbuchform(vocable,
                    ownLists.getVocabularies());
            if (!woerterbuchform.equals(VerbWoerterbuchformFinder.NO_WOERTERBUCHFORM_FOUND)) {
                panel.add(
                        creator.createFindRelatedVerbsButtonViaWoerterbuchform(woerterbuchform));
            }
        }
        else if (vocable.isAdjectivInPositivePresence()) {
            panel.add(creator.createFindRelatedAdjectiveButton());
        }
        else if (vocable.isAdjective()) {
            Vocable positivePresence = AdjectivePositivePresenceFinder.determinePositivePresence(
                    vocable, ownLists.getVocabularies());
            if (!positivePresence.equals(
                    AdjectivePositivePresenceFinder.NO_POSITIVE_PRESENCE_FOUND)) {
                panel.add(
                        creator.createFindRelatedAdjectiveButtonViaPositivePresence(
                                positivePresence));
            }
        }
    }

}
