package de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjiset;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.Text;
import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.NonModalFrameDialogBase;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.data.KanjiSet;
import de.duehl.vocabulary.japanese.logic.VocabularyTrainerLogic;
import de.duehl.vocabulary.japanese.ui.components.bars.KanjiSetBar;
import de.duehl.vocabulary.japanese.ui.data.MessageSetter;

/**
 * Diese Klasse lässt den Benutzer die Kanji-Mengen verwalten (neue anlegen, Listen bearbeiten,
 * Listen löschen).
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

public class KanjiSetManagementDialog extends NonModalFrameDialogBase
        implements KanjiSetBarsRefresher {

    private static final int MIN_WIDTH = 500;


    /** Die Logik des Vokabeltrainers. */
    private final VocabularyTrainerLogic logic;

    /** Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen kann. */
    private final LongTimeProcessInformer informer;

    /** Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann. */
    private final MessageSetter messageSetter;

    /** Die Liste der benutzerdefinierten Kanji-Mengen. */
    private final List<KanjiSet> kanjiSets;

    /** Die Menge der dargestellten Bars mit je einer Kanji-Menge. */
    private final List<KanjiSetBar> bars;

    /** Das Panel auf dem die Bars mit je einer Kanji-Menge dargestellt werden. */
    private final JPanel barsPanel;

    /** Der Button zum Anlegen einer neuen Kanji-Menge. */
    private final JButton newKanjiSetButton;

    /** Der Button zum Beenden des Dialogs. */
    private final JButton okButton;

    /**
     * Konstruktor.
     *
     * @param logic
     *            Die Logik des Vokabeltrainers.
     * @param informer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     * @param kanjiSets
     *            Die Liste der benutzerdefinierten Kanji-Mengen.
     * @param messageSetter
     *            Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Anzuzeigendes ProgrammIcon.
     */
    public KanjiSetManagementDialog(VocabularyTrainerLogic logic, LongTimeProcessInformer informer,
            MessageSetter messageSetter, Point parentLocation, Image programImage) {
        super(parentLocation, programImage, "Verwaltung der Kanji-Mengen");
        addClosingWindowListener(() -> quit());
        addEscapeBehaviour();

        this.logic = logic;
        this.informer = informer;
        this.messageSetter = messageSetter;

        kanjiSets = logic.getKanjiSets();

        bars = new ArrayList<>();
        barsPanel = new JPanel();

        newKanjiSetButton = new JButton("Neue Kanji-Menge hinzufügen");
        okButton = new JButton();

        init();
        createAndShowBars();
        fillDialog();
    }

    private void init() {
        initBarsPanel();
        initNewKanjiSetButton();
    }

    private void initBarsPanel() {
        barsPanel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));
    }

    private void initNewKanjiSetButton() {
        newKanjiSetButton.addActionListener(e -> addNewKanjiList());
    }

    private void addNewKanjiList() {
        List<String> notAllowedNames = new ArrayList<>();
        for (KanjiSet kanjiSet : kanjiSets) {
            notAllowedNames.add(kanjiSet.getName());
        }

        String title = "Den Namen der neuen Kanji-Menge eingeben";
        String message = "Bitte den Namen der neuen Kanji-Menge eingeben.";
        String name = GuiTools.askUserToEnterAStringValue(getWindowAsComponent(), title, message);
        if (name.isBlank()) {
            return;
        }

        if (notAllowedNames.contains(name)) {
            title = "Der Name ist bereits vergeben";
            message = "Den Name '" + name + "' ist bereits vergeben.";
            GuiTools.informUser(getWindowAsComponent(), title, message);
            return;
        }

        KanjiSet kanjiSet = new KanjiSet(name);
        kanjiSets.add(kanjiSet);
        createAndShowBars();
    }

    private void createAndShowBars() {
        bars.clear();
        barsPanel.removeAll();

        sortKanjiSets();
        List<String> allNames = new ArrayList<>();
        for (KanjiSet kanjiSet : kanjiSets) {
            allNames.add(kanjiSet.getName());
        }

        for (KanjiSet kanjiSet : kanjiSets) {
            KanjiSetBar bar = new KanjiSetBar(kanjiSet, allNames, (KanjiSetBarsRefresher) this,
                    logic, informer, messageSetter, getLocation(), getProgramImage());
            bars.add(bar);
            barsPanel.add(bar.getPanel());
        }

        barsPanel.invalidate();
        barsPanel.validate();
        barsPanel.repaint();
        pack();
    }

    private void sortKanjiSets() {
        Collections.sort(kanjiSets, new Comparator<KanjiSet>() {
            @Override
            public int compare(KanjiSet set1, KanjiSet set2) {
                String name1 = set1.getName();
                String name2 = set2.getName();
                return Text.compareStringsBetter(name1, name2);
                //return name1.compareTo(name2);
            }
        });
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createHeaderPart(), BorderLayout.NORTH);
        add(barsPanel, BorderLayout.CENTER);
        add(createButtonsPart(), BorderLayout.SOUTH);

        SwingUtilities.invokeLater(() -> okButton.requestFocus());
    }

    private Component createHeaderPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(createDummyMinWidthLabelPart());
        panel.add(createUpperButtonPart());

        return panel;
    }

    private Component createDummyMinWidthLabelPart() {
        JLabel label = new JLabel("");
        label.setPreferredSize(new Dimension(MIN_WIDTH, 0));
        return label;
    }

    private Component createUpperButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(newKanjiSetButton, BorderLayout.WEST);

        return panel;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createQuitButton() {
        okButton.setText("Beenden");
        okButton.addActionListener(e -> quit());
        return okButton;
    }

    private void quit() {
        saveKanjiLists();
        closeDialog();
    }

    /** Speichert die Kanji-Mengen ab. */
    private void saveKanjiLists() {
        logic.saveKanjiSets();
    }

    /** Löscht die übergebene Kanji-Menge aus der Liste der Kanji-Mengen. */
    @Override
    public void deleteKanjiSetFromListAndFileFromDisk(KanjiSet kanjiSet) {
        kanjiSets.remove(kanjiSet);
        String filename = kanjiSet.determineFilename();
        if (FileHelper.isFile(filename)) { // gibt es nicht, wenn man sie gerade erst angelegt hat
            FileHelper.deleteFile(filename);
        }
    }

    /** Stellt die Bars mit den Kanji-Mengen neu dar. */
    @Override
    public void refreshKanjiSetBars() {
        createAndShowBars();
    }

}
