package de.duehl.vocabulary.japanese.website.update.ownlists.parser;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.website.update.ownlists.data.NewestOwnListVersionListEntry;

/**
 * Diese Klasse parst den Inhalt der Datei mit den Versionen der Gruppen von eigenen Listen
 * (Neuste_Version_Listen.txt) die es lokal und auf dem Webserver gibt.
 *
 * @version 1.01     2024-10-13
 * @author Christian Dühl
 */

public class NewestOwnListsVersionParser {

    /** Der Inhalt der eingelesenen Datei Neuste_Version_Listen.txt. */
    private final String content;

    /** Die interessanten Zeilen des Inhalts der Datei. */
    private List<String> lines;

    /** Gibt an, ob der String mit der Version erfolgreich geparst werden konnte. */
    private boolean success;

    /**
     * Der Grund, warum der String mit der Version nicht geparst werden konnte (nur gefüllt wenn
     * success false ist).
     */
    private String failReason;

    /** Die Liste mit den Versionen der Gruppen von eigenen Listen. */
    private List<NewestOwnListVersionListEntry> newestOwnListVersionListEntries;

    /**
     * Konstruktor.
     *
     * @param content
     *            Der Inhalt der eingelesenen Datei Neuste_Version_Listen.txt.
     */
    public NewestOwnListsVersionParser(String content) {
        this.content = content;
    }

    /** Führt das Parsen durch. */
    public void parse() {
        init();
        splitContent();
        parseLines();
        removeAllEntriesIfErrorOccured();
    }

    private void init() {
        success = true;
        failReason = "";
        newestOwnListVersionListEntries = new ArrayList<>();
    }

    private void splitContent() {
        lines = Text.splitByLineBreaks(content);
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(lines);
    }

    private void parseLines() {
        for (String line : lines) {
            NewestOwnListVersionListEntry entry = tryTpParseLine(line);

            newestOwnListVersionListEntries.add(entry);
        }
    }

    private NewestOwnListVersionListEntry tryTpParseLine(String line) {
        try {
            return NewestOwnListVersionListEntry.createFromLine(line);
        }
        catch (Exception exception) {
            fail("Die Zeile '" + line + "' lässt sich nicht parsen: " + exception.getMessage());
            return new NewestOwnListVersionListEntry("", 0);
        }
    }

    private void removeAllEntriesIfErrorOccured() {
        if (!success) {
            newestOwnListVersionListEntries.clear();
        }
    }

    private void fail(String failReason) {
        success = false;
        this.failReason = failReason;
    }

    /** Gibt an, ob der String mit der Version erfolgreich geparst werden konnte. */
    public boolean isSuccess() {
        return success;
    }

    /**
     * Getter für den Grund, warum der String mit der Version nicht geparst werden konnte (nur
     * gefüllt wenn isSuccess() false ist).
     */
    public String getFailReason() {
        return failReason;
    }

    /** Getter für die Liste mit den Versionen der Gruppen von eigenen Listen. */
    public List<NewestOwnListVersionListEntry> getNewestOwnListVersionListEntries() {
        return newestOwnListVersionListEntries;
    }

}
