package de.duehl.basics.io.textfile.dictionary.io.reader;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.Charset;
import de.duehl.basics.io.textfile.StringsFromNormalTextFileReader;

/**
 * Diese Klasse liest eine Liste mit den Texten im Format des DictionaryFiles aus den Zeilen einer
 * normalen Textdatei ein.
 *
 * Das Format sieht wie folgt aus:
 *
 * Englisches Wort
 *     Deutsches Wort 1
 *     Deutsches Wort 2
 *     Deutsches Wort 3
 *     Deutsches Wort 4
 *
 * Kommt in einer Datenzeile weiter hinten das Kommentarzeichen vor, so wird ab dort abgeschnitten
 * und getrimmt.
 *
 * @version 1.01     2024-05-17
 * @author Christian Dühl
 */

public class DictionaryFromNormalTextFileReader extends DictionaryFromTextFileReader {

    /** Kommentarzeichen '#' für Konstruktoren ohne diese Angabe. */
    private static final String DEFAULT_COMMENT_CHARACTER = "#";

    /**
     * Konstruktor für Dateien im Format UTF-8 mit Standard-Kommentarzeichen '#'.
     *
     * @param filename
     *            Name der einzulesenden Datei.
     */
    public DictionaryFromNormalTextFileReader(String filename) {
        this(filename, Charset.UTF_8, DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor für Dateien mit Kodierung und Standard-Kommentarzeichen '#'.
     *
     * @param filename
     *            Name der einzulesenden Datei.
     * @param charset
     *            Kodierung der einzulesenden Datei.
     */
    public DictionaryFromNormalTextFileReader(String filename, Charset charset) {
        this(filename, charset, DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor.
     *
     * @param filename
     *            Name der einzulesenden Datei.
     * @param charset
     *            Kodierung der einzulesenden Datei.
     * @param commentChar
     *            Das Kommentarzeichen.
     */
    public DictionaryFromNormalTextFileReader(String filename, Charset charset,
            String commentChar) {
        super(new StringsFromNormalTextFileReader(filename, charset, commentChar));
    }

}
