package de.duehl.basics.datetime;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.Calendar;

import org.junit.Test;

import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.basics.datetime.time.ImmutualTime;

public class DateAndTimeHelperTest {

    @Test
    public void now() {
        String now = DateAndTimeHelper.now();
        assertNotNull(now);
    }

    @Test
    public void millisToDateAndTime() {
        //System.out.println("millis : " + TimeHelper.getCurrentTimeInMilliseconds());
        //System.out.println(DateAndTimeHelper.now());
        DateAndTime actual = DateAndTimeHelper.millisToDateAndTime(1616069139593L);
        DateAndTime expected = new DateAndTime(new ImmutualDate("18.03.2021"),
                new ImmutualTime("13:05:39"));
        assertEquals(expected, actual);
    }

    @Test
    public void millisToDateAndTimeString() {
        //System.out.println("millis : " + TimeHelper.getCurrentTimeInMilliseconds());
        //System.out.println(DateAndTimeHelper.now());
        String actual = DateAndTimeHelper.millisToDateAndTimeString(1616069139593L);
        String expected = "18.03.2021, 13:05:39";
        assertEquals(expected, actual);
    }

    @Test
    public void millisToDateAndTime2() {
        DateAndTime actual = DateAndTimeHelper.millisToDateAndTime(1490622889000L);
        DateAndTime expected = new DateAndTime(new ImmutualDate("27.03.2017"),
                new ImmutualTime("15:54:49"));
        assertEquals(expected, actual);
    }

    @Test
    public void millisToDateAndTimeString2() {
        String actual = DateAndTimeHelper.millisToDateAndTimeString(1490622889000L);
        String expected = "27.03.2017, 15:54:49";
        assertEquals(expected, actual);
    }

    @Test
    public void timeWithoutSecondsWithoutDateIfFromToday() {
        ImmutualDate today = new ImmutualDate();
        ImmutualTime time = new ImmutualTime("13:05");
        DateAndTime dateAndTime = new DateAndTime(today, time);
        String actual = DateAndTimeHelper.timeWithoutSecondsWithoutDateIfFromToday(dateAndTime);
        String expected = "13:05";
        assertEquals(expected, actual);
    }

    @Test
    public void timeWithoutSecondsWithoutDateIfFromTodayOtherDay() {
        ImmutualDate date = new ImmutualDate("11.02.2021");
        ImmutualTime time = new ImmutualTime("13:05");
        DateAndTime dateAndTime = new DateAndTime(date, time);
        String actual = DateAndTimeHelper.timeWithoutSecondsWithoutDateIfFromToday(dateAndTime);
        String expected = "13:05 am 11.02.2021";
        assertEquals(expected, actual);
    }

    @Test
    public void timePeriodWithoutSecondsWithoutDateIfFromToday() {
        ImmutualDate today = new ImmutualDate();
        ImmutualTime fromTime = new ImmutualTime("13:05");
        ImmutualTime toTime = new ImmutualTime("13:55");
        DateAndTime from = new DateAndTime(today, fromTime);
        DateAndTime to = new DateAndTime(today, toTime);
        String actual = DateAndTimeHelper.timePeriodWithoutSecondsWithoutDateIfFromToday(from, to);
        String expected = "13:05 - 13:55";
        assertEquals(expected, actual);
    }

    @Test
    public void timePeriodWithoutSecondsWithoutDateIfFromTodayFromOtherDays() {
        ImmutualDate today = new ImmutualDate();
        ImmutualDate fromDate =  new ImmutualDate("11.02.2021");
        ImmutualTime fromTime = new ImmutualTime("13:05");
        ImmutualTime toTime = new ImmutualTime("13:55");
        DateAndTime from = new DateAndTime(fromDate, fromTime);
        DateAndTime to = new DateAndTime(today, toTime);
        String actual = DateAndTimeHelper.timePeriodWithoutSecondsWithoutDateIfFromToday(from, to);
        String expected = "13:05 am 11.02.2021 - 13:55 am " + today.toString();
        assertEquals(expected, actual);
    }

    @Test
    public void timePeriodWithoutSecondsWithoutDateIfFromTodayToOtherDays() {
        ImmutualDate today = new ImmutualDate();
        ImmutualTime fromTime = new ImmutualTime("13:05");
        ImmutualTime toTime = new ImmutualTime("13:55");
        ImmutualDate toDate =  new ImmutualDate("11.02.2021");
        DateAndTime from = new DateAndTime(today, fromTime);
        DateAndTime to = new DateAndTime(toDate, toTime);
        String actual = DateAndTimeHelper.timePeriodWithoutSecondsWithoutDateIfFromToday(from, to);
        String expected = "13:05 am " + today.toString() + " - 13:55 am 11.02.2021";
        assertEquals(expected, actual);
    }

    @Test
    public void timePeriodWithoutSecondsWithoutDateIfFromTodayBothOtherDays() {
        ImmutualDate date = new ImmutualDate("11.02.2021");
        ImmutualTime fromTime = new ImmutualTime("13:05");
        ImmutualTime toTime = new ImmutualTime("13:55");
        DateAndTime from = new DateAndTime(date, fromTime);
        DateAndTime to = new DateAndTime(date, toTime);
        String actual = DateAndTimeHelper.timePeriodWithoutSecondsWithoutDateIfFromToday(from, to);
        String expected = "13:05 am 11.02.2021 - 13:55 am 11.02.2021";
        assertEquals(expected, actual);
    }

    @Test
    public void calendarToDateAndTimeTestConcreteDate() {
        Calendar cal = Calendar.getInstance();
        cal.set(2016, 10 - 1, 12, 11, 25, 0);

        DateAndTime dateAndTime = DateAndTimeHelper.calendarToDateAndTime(cal);

        ImmutualDate date = dateAndTime.getDate();

        int actualYear = date.getYear();
        int actualMonth = date.getMonth();
        int actualDay = date.getDay();

        int expectedYear = 2016;
        int expectedMonth = 10;
        int expectedDay = 12;

        assertEquals(expectedYear, actualYear);
        assertEquals(expectedMonth, actualMonth);
        assertEquals(expectedDay, actualDay);

        ImmutualTime time = dateAndTime.getTime();

        int actualHour = time.getHour();
        int actualMinute = time.getMinute();
        int actualSecond = time.getSecond();

        int expectedHour = 11;
        int expectedMinute = 25;
        int expectedSecond = 0;

        assertEquals(expectedHour, actualHour);
        assertEquals(expectedMinute, actualMinute);
        assertEquals(expectedSecond, actualSecond);
    }

    @Test
    public void toEpoch() {
        String date = "27.03.2017";
        String time = "15:54:49";
        long actualMillis = DateAndTimeHelper.toEpoch(date, time);
        long expectedMillis = 1490622889000L;
        assertEquals(expectedMillis, actualMillis);
    }

}
