package de.duehl.swing.ui.dialogs;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.basics.datetime.DateAndTime;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.components.selections.datetime.SimpleDateAndTimeSelection;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zur Eingabe von Uhrzeit und Datum.
 *
 * @version 1.02     2020-01-24
 * @author Christian Dühl
 */

public class SimpleDateAndTimeSelectionDialog extends ModalDialogBase {

    private final SimpleDateAndTimeSelection dateAndTimeSelection;

    private DateAndTime selectedDateAndTime;

    /** Konstruktor. */
    public SimpleDateAndTimeSelectionDialog() {
        this("Bitte Uhrzeit und Datum auswählen");
    }

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     */
    public SimpleDateAndTimeSelectionDialog(String title) {
        this(new Point(250, 100), title);
    }

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param title
     *            Titel des Dialogs.
     */
    public SimpleDateAndTimeSelectionDialog(Point parentLocation, String title) {
        this(parentLocation, null, title);
    }

    /**
     * Konstruktor.
     *
     * @param thingsToSelect
     *            Liste mit der Auswahl, die man hier modifizieren kann.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Image für den Dialog.
     * @param title
     *            Titel des Dialogs.
     */
    public SimpleDateAndTimeSelectionDialog(Point parentLocation, Image programImage,
            String title) {
        super(parentLocation, programImage, title);
        ignoreSize();
        setMinimumWidth(400);

        dateAndTimeSelection = new SimpleDateAndTimeSelection();
        dateAndTimeSelection.addReturnListener(() -> apply());

        fillDialog();
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createMainPart(), BorderLayout.CENTER);
    }

    private Component createMainPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        //GuiTools.createTitle(panel);

        panel.add(createDateAndTimeSelectionPart(), BorderLayout.CENTER);
        panel.add(createButtonPart(), BorderLayout.SOUTH);

        return panel;
    }

    private Component createDateAndTimeSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        panel.add(dateAndTimeSelection.getPanel(), BorderLayout.CENTER);

        return panel;
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createOkButton(), BorderLayout.EAST);

        return panel;
    }

    private JButton createOkButton() {
        JButton button = new JButton("OK");
        setColors(button);
        button.addActionListener(e -> apply());

        return button;
    }

    private void apply() {
        if (dateAndTimeSelection.checkValidity()) {
            selectedDateAndTime = dateAndTimeSelection.getDateAndTime();
            closeDialog();
        }
        else {
            createErrorHandler().warning(dateAndTimeSelection.getValidityCheckFailedCause());
        }
    }

    public void initWithNow() {
        dateAndTimeSelection.initWithNow();
        repaint();
    }

    public void initWithSmoothTimeInAbout10Minutes() {
        dateAndTimeSelection.initWithSmoothTimeInAbout10Minutes();
        repaint();
    }

    public DateAndTime getSelectedDateAndTime() {
        return selectedDateAndTime;
    }

    public void setSelectedDateAndTime(DateAndTime selectedDateAndTime) {
        this.selectedDateAndTime = selectedDateAndTime;
    }

}
