package de.duehl.swing.ui.elements.navigator;

import java.util.List;

import de.duehl.basics.io.FileHelper;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

/**
 * Diese Klasse stellt die anzuzeigenden und zu bearbeitenden Daten, zwischen denen navigiert wird
 * dar und ist die abstrakte Basis für solche Datensätze, die zusätzlich gespeichert und vom
 * vorigen oder nächsten Datensatz kopiert werden können.
 *
 * @version 1.01     2021-06-15
 * @author Christian Dühl
 */

public abstract class NavigatedDataWithSaveAndCopyData<Data> extends NavigatedData<Data> {

    /** Gibt an, ob die Daten gespeichert wurden. */
    private boolean saved;

    /**
     * Konstruktor.
     *
     * @param uiDatasets
     *            Liste der anzuzeigenden und zu bearbeitenden Daten, zwischen denen navigiert
     *            wird.
     */
    public NavigatedDataWithSaveAndCopyData(List<Data> uiDatasets) {
        super(uiDatasets);
        saved = false;
    }

    /** Alle Daten speichern. */
    public final void save() {
        saveInternal();
        saved = true;
    }

    /** Alle Daten speichern. */
    protected abstract void saveInternal();

    /** Inhalte dieses Datensatzes mit denen vom vorigen Datensatz überschreiben. */
    public abstract void copyFromPrevious();

    /** Inhalte dieses Datensatzes mit denen vom nächsten Datensatz überschreiben. */
    public abstract void copyFromNext();

    /**
     * Erzeugt eine Datei die anzeigt, dass die Bearbeitung abgeschlossen ist, falls die Daten
     * gespeichert wurden.
     */
    public final void createDoneFileIfDataWasSaved() {
        if (saved) {
            reallyCreateDoneFileIfDataWasSaved();
        }
    }

    private void reallyCreateDoneFileIfDataWasSaved() {
        String filename = createDoneFilename();
        FileHelper.createEmptyFile(filename);
    }

    /**
     * Erzeugt den Namen für die Datei die anzeigt, dass die Bearbeitung abgeschlossen ist, falls
     * die Daten gespeichert wurden.
     */
    protected abstract String createDoneFilename();

}
