package de.duehl.swing.ui.elements.navigator;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;

import javax.swing.JMenu;
import javax.swing.JMenuItem;

import de.duehl.swing.ui.menu.MyMenuItem;
import de.duehl.swing.ui.menu.SingleMenuCreation;

/**
 * Diese Klasse erzeugt das Navigation-Menü einer graphischen Oberfläche mit einem
 * Navigationselement.
 *
 * @version 1.01     2020-09-22
 * @author Christian Dühl
 */

public class NavigationMenuCreation implements SingleMenuCreation {

    private static final char DEFAULT_MNEMONIC = 'v';  // n

    /** Grafisches Element mit der Navigation. */
    private final NavigatorPanel navigator;

    /** Aktivierungszeichen zum Öffnen des Menüs. */
    private char menuMnemonic;

    /** Modifier für die alternative Navigation. */
    private int alternativeModifier;

    /**
     * Konstruktor.
     *
     * @param logic
     *            Verbindung zur Programmlogik.
     * @param navigator
     *            Grafisches Element mit der Navigation.
     */
    public NavigationMenuCreation(NavigatorPanel navigator) {
        this.navigator = navigator;
        menuMnemonic = DEFAULT_MNEMONIC;
        alternativeModifier = InputEvent.ALT_DOWN_MASK;
    }

    /** Setter für das Aktivierungszeichen zum Öffnen des Menüs. */
    public NavigationMenuCreation setMenuMnemonic(char menuMnemonic) {
        this.menuMnemonic = menuMnemonic;
        return this;
    }

    /**
     * Legt fest, dass die alternative Navigation ohne zusätzliches Drücken der Alt-Taste
     * funktioniert.
     */
    public NavigationMenuCreation alternativeWithoutAlt() {
        alternativeModifier = 0;
        return this;
    }

    /** Erzeugt das Navigation-Menü. */
    @Override
    public JMenu createTheMenu() {
        JMenu menu = new JMenu("Navigation");
        menu.setMnemonic(menuMnemonic);

        menu.add(createShowFirstDocumentMenuItem());
        menu.add(createShowPreviousDocumentMenuItem());
        menu.add(createShowNextDocumentMenuItem());
        menu.add(createShowLastDocumentMenuItem());
        menu.addSeparator();
        menu.add(createAlternativeShowPreviousDocumentMenuItem());
        menu.add(createAlternativeShowNextDocumentMenuItem());

        addAdditionalNavigationMenuItems(menu);

        return menu;
    }

    private JMenuItem createShowFirstDocumentMenuItem() {
        return new MyMenuItem("erster Datensatz")
                .mnemonic('e')
                .accelerator(KeyEvent.VK_HOME, InputEvent.ALT_DOWN_MASK)
                .actionListener(e -> navigator.doClickOnFirstButton());
    }

    private JMenuItem createShowPreviousDocumentMenuItem() {
        return new MyMenuItem("voriger Datensatz")
            .mnemonic('v')
            .accelerator(KeyEvent.VK_LEFT, InputEvent.ALT_DOWN_MASK)
            .actionListener(e -> navigator.doClickOnPreviousButton());
    }

    private JMenuItem createShowNextDocumentMenuItem() {
        return new MyMenuItem("nächster Datensatz")
                .mnemonic('n')
                .accelerator(KeyEvent.VK_RIGHT, InputEvent.ALT_DOWN_MASK)
                .actionListener(e -> navigator.doClickOnNextButton());
    }

    private JMenuItem createShowLastDocumentMenuItem() {
        return new MyMenuItem("letzter Datensatz")
                .mnemonic('l')
                .accelerator(KeyEvent.VK_END, InputEvent.ALT_DOWN_MASK)
                .actionListener(e -> navigator.doClickOnLastButton());
    }

    private JMenuItem createAlternativeShowPreviousDocumentMenuItem() {
        return new MyMenuItem("voriger Datensatz (Alternative)")
                //.mnemonic('v')
                .accelerator(KeyEvent.VK_PAGE_UP, alternativeModifier)
                .actionListener(e -> navigator.doClickOnPreviousButton());
    }

    private JMenuItem createAlternativeShowNextDocumentMenuItem() {
        return new MyMenuItem("nächster Datensatz (Alternative)")
                //.mnemonic('n')
                .accelerator(KeyEvent.VK_PAGE_DOWN, alternativeModifier)
                .actionListener(e -> navigator.doClickOnNextButton());
    }

    /**
     * Bietet die Möglichkeit, weitere Menüpunkte zum Navigations-Menü hinzuzufügen.
     *
     * @param menu
     *            Das JMenu in das diese Menüpunkte eingefügt werden.
     */
    protected void addAdditionalNavigationMenuItems(JMenu menu) {
        // falls ein Projekt weitere Navigations-Menüpunkte einfügen möchte.
    }

}
