package de.duehl.swing.ui.menu.collection.tools;

import java.awt.Image;
import java.awt.Point;

import de.duehl.swing.ui.key.KeyDefinition;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.swing.ui.menu.collection.AutomaticMyMenuItemCollection;
import de.duehl.swing.ui.menu.collection.StoredMyMenuItem;
import de.duehl.swing.ui.text.html.HtmlFrame;

/**
 * Diese Hilfsklasse bietet Methoden rund um die Sammlung von Menüeinträgen sammelt die gemerkten
 * Eigenschaften von der Menüeinträgen.
 *
 * @version 1.01     2020-12-09
 * @author Christian Dühl
 */

public class MyMenuItemCollectionHelper {

    /**
     * Erzeugt eine HTML-Beschreibung der automatisch gesammelten Menüeinträge mit individuellem
     * CSS und zeigt diese in einem Frame an.
     *
     * @param title
     *            Titel des Dialogs.
     * @param css
     *            Individueller CSS-Code.
     * @param programImage
     *            Image vom aufrufenden Programm.
     * @param location
     *            Image vom aufrufenden Programm.
     */
    public static void generateAndShowMyMenuItemsHtmlDescriptionDialog(String title, String css,
            Image programImage, Point location) {
        String html = MyMenuItemCollectionHelper.generateMyMenuItemsHtmlDescription(css);
        generateAndShowMyMenuItemsHtmlDescription(title, programImage, location, html);
    }

    /**
     * Erzeugt eine HTML-Beschreibung der automatisch gesammelten Menüeinträge in Reihenfolge der
     * Tasten mit individuellem CSS und zeigt diese in einem Frame an.
     *
     * @param title
     *            Titel des Dialogs.
     * @param css
     *            Individueller CSS-Code.
     * @param programImage
     *            Image vom aufrufenden Programm.
     * @param location
     *            Image vom aufrufenden Programm.
     */
    public static void generateAndShowMyMenuItemsHtmlDescriptionSortByKeysDialog(String title,
            String css, Image programImage, Point location) {
        String html = MyMenuItemCollectionHelper.generateMyMenuItemsHtmlDescriptionSortByKeys(css);
        generateAndShowMyMenuItemsHtmlDescription(title, programImage, location, html);
    }

    private static void generateAndShowMyMenuItemsHtmlDescription(String title, Image programImage,
            Point location, String html) {
        HtmlFrame dialog = new HtmlFrame(title, programImage, location);
        dialog.setText(html);
        dialog.scrollScrollbarToMinimumLater();

        dialog.setVisible(true);
    }

    /** Erzeugt eine HTML-Beschreibung der automatisch gesammelten Menüeinträge ohne CSS. */
    public static String generateMyMenuItemsHtmlDescription() {
        return generateMyMenuItemsHtmlDescription("");
    }

    /**
     * Erzeugt eine HTML-Beschreibung der automatisch gesammelten Menüeinträge mit individuellem
     * CSS.
     */
    public static String generateMyMenuItemsHtmlDescription(String css) {
        MyMenuItemsHtmlDescriptionGenerator generator = createMyMenuItemsHtmlDescriptionGenerator(
                css);
        generator.buildDescription();
        return generator.getHtmlDescription();
    }

    /**
     * Erzeugt eine HTML-Beschreibung der automatisch gesammelten Menüeinträge in Reihenfolge der
     * Tasten ohne CSS.
     */
    public static String generateMyMenuItemsHtmlDescriptionSortByKeys() {
        return generateMyMenuItemsHtmlDescriptionSortByKeys("");
    }

    /**
     * Erzeugt eine HTML-Beschreibung der automatisch gesammelten Menüeinträge in Reihenfolge der
     * Tasten mit individuellem CSS.
     */
    public static String generateMyMenuItemsHtmlDescriptionSortByKeys(String css) {
        MyMenuItemsHtmlDescriptionGenerator generator = createMyMenuItemsHtmlDescriptionGenerator(
                css);
        generator.sortByKeys();
        generator.buildDescription();
        return generator.getHtmlDescription();
    }

    private static MyMenuItemsHtmlDescriptionGenerator createMyMenuItemsHtmlDescriptionGenerator(
            String css) {
        MyMenuItemsHtmlDescriptionGenerator generator = new MyMenuItemsHtmlDescriptionGenerator(
                AutomaticMyMenuItemCollection.getListOfCollectedMyMenuItems());
        if (!css.isEmpty()) {
            generator.setCss(css);
        }
        return generator;
    }

    /** Erzeugt eine lesbare Darstellung eines Tastaturkommandos eines Menüeintrags. */
    public static String generateKeyboardCommandDescription(StoredMyMenuItem storedMyMenuItem) {
        int keyCode = storedMyMenuItem.getKeyCode();
        int modifiers = storedMyMenuItem.getModifiers();

        return generateKeyboardCommandDescription(keyCode, modifiers);
    }

    /** Erzeugt eine lesbare Darstellung eines Tastaturkommandos einer KeyDefinition. */
    public static String generateKeyboardCommandDescription(KeyDefinition keyDefinition) {
        int keyCode = keyDefinition.getKeyCode();
        int modifiers = keyDefinition.getModifiers();

        return generateKeyboardCommandDescription(keyCode, modifiers);
    }

    private static String generateKeyboardCommandDescription(int keyCode, int modifiers) {
        String keyCodeDescription = generateReadableKeyCode(keyCode);

        if (0 == modifiers) {
            return keyCodeDescription;
        }
        else {
            String modifierDescription = generateReadableModifiers(modifiers);
            return ""
                    + modifierDescription
                    + " + "
                    + keyCodeDescription
                    ;
        }
    }

    /** Erzeugt eine lesbare Darstellung des KeyCodes eines Tastaturkommandos. */
    private static String generateReadableKeyCode(int keyCode) {
        ReadableKeyCodeGenerator generator = new ReadableKeyCodeGenerator(keyCode);
        return generator.generate();
    }

    /** Erzeugt eine lesbare Darstellung von Modifikatoren eines Tastaturkommandos. */
    private static String generateReadableModifiers(int modifiers) {
        ReadableModifiersGenerator generator = new ReadableModifiersGenerator(modifiers);
        return generator.generate();
    }

}
