package de.duehl.swing.ui.dragndrop.filler;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

import de.duehl.basics.text.Text;

public class AdressFillerTest {

    @Test
    public void fillAdressWithoutCountry() {
        String text = "Deciusstr. 6" + Text.LINE_BREAK + "33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithoutCountryInSingleLine() {
        String text = "Deciusstr. 6 33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithoutCountryInSingleLineWithKomma() {
        String text = "Deciusstr. 6, 33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithoutCountryInSingleLineWithKommaWithoutSpace() {
        String text = "Deciusstr. 6,33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithLetterCountry() {
        String text = "Deciusstr. 6" + Text.LINE_BREAK + "D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("D", filler.getCountry());
    }

    @Test
    public void fillAdressWithLetterCountryInSingleLine() {
        String text = "Deciusstr. 6 D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("D", filler.getCountry());
    }

    @Test
    public void fillAdressWithLetterCountryInSingleLineWithKomma() {
        String text = "Deciusstr. 6, D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("D", filler.getCountry());
    }

    @Test
    public void fillAdressWithLetterCountryInSingleLineWithKommaWithoutSpace() {
        String text = "Deciusstr. 6,D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("D", filler.getCountry());
    }

    @Test
    public void fillAdressWithFullCountry() {
        String text = "Deciusstr. 6"
                + Text.LINE_BREAK + "33611 Bielefeld"
                + Text.LINE_BREAK + "Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("Deutschland", filler.getCountry());
    }

    @Test
    public void fillAdressWithFullCountryInSingleLine() {
        String text = "Deciusstr. 6 33611 Bielefeld Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("Deutschland", filler.getCountry());
    }

    @Test
    public void fillAdressWithFullCountryInSingleLineWithKomma() {
        String text = "Deciusstr. 6, 33611 Bielefeld, Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("Deutschland", filler.getCountry());
    }

    @Test
    public void fillAdressWithFullCountryInSingleLineWithKommaWithoutSpace() {
        String text = "Deciusstr. 6,33611 Bielefeld,Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("Deutschland", filler.getCountry());
    }

    @Test
    public void fillAdressWithLetterCountryAndFullCountry() {
        String text = "Deciusstr. 6"
                        + Text.LINE_BREAK + "D-33611 Bielefeld"
                        + Text.LINE_BREAK + "Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("D Deutschland", filler.getCountry());
    }

    @Test (expected = RuntimeException.class)
    public void errorWhenAnalyseMultipleTimes() {
        String text = "Deciusstr. 6"
                        + Text.LINE_BREAK + "D-33611 Bielefeld"
                        + Text.LINE_BREAK + "Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.analyze();
        filler.analyze();
    }

    @Test
    public void fillAdressWithoutCountryOnlyInland() {
        String text = "Deciusstr. 6" + Text.LINE_BREAK + "33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithoutCountryInSingleLineOnlyInland() {
        String text = "Deciusstr. 6 33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithoutCountryInSingleLineWithKommaOnlyInland() {
        String text = "Deciusstr. 6, 33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithoutCountryInSingleLineWithKommaWithoutSpaceOnlyInland() {
        String text = "Deciusstr. 6,33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithLetterCountryOnlyInland() {
        String text = "Deciusstr. 6" + Text.LINE_BREAK + "D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertFalse(filler.foundAddress());
    }

    @Test
    public void fillAdressWithLetterCountryInSingleLineOnlyInland() {
        String text = "Deciusstr. 6 D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertFalse(filler.foundAddress());
    }

    @Test
    public void fillAdressWithLetterCountryInSingleLineWithKommaOnlyInland() {
        String text = "Deciusstr. 6, D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertFalse(filler.foundAddress());
    }

    @Test
    public void fillAdressWithLetterCountryInSingleLineWithKommaWithoutSpaceOnlyInland() {
        String text = "Deciusstr. 6,D-33611 Bielefeld";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertFalse(filler.foundAddress());
    }

    @Test
    public void fillAdressWithFullCountryOnlyInland() {
        String text = "Deciusstr. 6"
                + Text.LINE_BREAK + "33611 Bielefeld"
                + Text.LINE_BREAK + "Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld Deutschland", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithFullCountryInSingleLineOnlyInland() {
        String text = "Deciusstr. 6 33611 Bielefeld Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld Deutschland", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithFullCountryInSingleLineWithKommaOnlyInland() {
        String text = "Deciusstr. 6, 33611 Bielefeld, Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld, Deutschland", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithFullCountryInSingleLineWithKommaWithoutSpaceOnlyInland() {
        String text = "Deciusstr. 6,33611 Bielefeld,Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertTrue(filler.foundAddress());
        assertEquals("Deciusstr. 6", filler.getStreet());
        assertEquals("33611", filler.getPostCode());
        assertEquals("Bielefeld,Deutschland", filler.getTown());
        assertEquals("", filler.getCountry());
    }

    @Test
    public void fillAdressWithLetterCountryAndFullCountryOnlyInland() {
        String text = "Deciusstr. 6"
                        + Text.LINE_BREAK + "D-33611 Bielefeld"
                        + Text.LINE_BREAK + "Deutschland";
        AddressFiller filler = new AddressFiller(text);
        filler.onlyInland();
        filler.analyze();
        assertFalse(filler.foundAddress());
    }

}
