package de.duehl.swing.ui.start.progress;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Point;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.system.SystemTools;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.elements.progress.MultipleProgressDialog;

public class MultipleProgressDialogDemo {

    private static final String IDENTIFIER_1 = "eins";
    private static final String IDENTIFIER_2 = "zwei";
    private static final String IDENTIFIER_3 = "drei";

    private static final int NUMBER_OF_TASKS_1 = 5;
    private static final int NUMBER_OF_TASKS_2 = 2;
    private static final int NUMBER_OF_TASKS_3 = 4;

    private static final Random RANDOM = new Random();

    private List<String> taskDescriptions;

    private final MultipleProgressDialog progressDialog;

    public MultipleProgressDialogDemo() {
        GuiTools.setNiceLayoutManager();

        progressDialog = new MultipleProgressDialog(
                "Fortschritt bei der Bearbeitung von mehreren Demo-Aufgaben",
                new Point(300, 200), null);

        progressDialog.addProgress(IDENTIFIER_1)
                .setProgressTitle(IDENTIFIER_1,
                        "Fortschritt beim Ausführen der ersten Demo-Aufgaben:")
                .setCountPrefix(IDENTIFIER_1, "Anzahl ausgeführter Aufgaben: ")
                .setTimerPrefix(IDENTIFIER_1, "Laufzeit: ")
                .setActualElementPrefix(IDENTIFIER_1, "Führe aus: ")
                .setActualElementPrefixBeforeStart(IDENTIFIER_1, "Noch nichts ausgeführt.")
                .setActualElementWhenDone(IDENTIFIER_1, "Alle Demo-Aufgaben wurden ausgeführt.")
                .createProgressPanel(IDENTIFIER_1);

        progressDialog.addProgress(IDENTIFIER_2)
                .setProgressTitle(IDENTIFIER_2,
                        "Fortschritt beim Ausführen der zweiten Demo-Aufgaben:")
                .setCountPrefix(IDENTIFIER_2, "Anzahl ausgeführter Aufgaben: ")
                .setTimerPrefix(IDENTIFIER_2, "Laufzeit: ")
                .setActualElementPrefix(IDENTIFIER_2, "Führe aus: ")
                .setActualElementPrefixBeforeStart(IDENTIFIER_2, "Noch nichts ausgeführt.")
                .setActualElementWhenDone(IDENTIFIER_2, "Alle Demo-Aufgaben wurden ausgeführt.")
                .createProgressPanel(IDENTIFIER_2);

        progressDialog.addProgress(IDENTIFIER_3)
                .setProgressTitle(IDENTIFIER_3,
                        "Fortschritt beim Ausführen der dritten Demo-Aufgaben:")
                .setCountPrefix(IDENTIFIER_3, "Anzahl ausgeführter Aufgaben: ")
                .setTimerPrefix(IDENTIFIER_3, "Laufzeit: ")
                .setActualElementPrefix(IDENTIFIER_3, "Führe aus: ")
                .setActualElementPrefixBeforeStart(IDENTIFIER_3, "Noch nichts ausgeführt.")
                .setActualElementWhenDone(IDENTIFIER_3, "Alle Demo-Aufgaben wurden ausgeführt.")
                .createProgressPanel(IDENTIFIER_3);

        progressDialog.createUi();
    }

    public void run() {
        List<String> identifiers = CollectionsHelper.buildListFrom(IDENTIFIER_1, IDENTIFIER_2,
                IDENTIFIER_3);

        new Thread(() -> runTasksOfAllIdentifiersInOwnThread(identifiers)).start();
    }

    private void runTasksOfAllIdentifiersInOwnThread(List<String> identifiers) {
        for (String identifier : identifiers) {
            createTaskDescriptions(identifier);
            progressDialog.initNumberOfTasksToDo(identifier, taskDescriptions.size());
            runTasks(identifier);
        }
    }

    private void createTaskDescriptions(String identifier) {
        taskDescriptions = new ArrayList<>();
        int numberOfTasks = createNumberOfTasks(identifier);
        for (int number = 1; number <= numberOfTasks; ++number) {
            taskDescriptions.add("Aufgabe " + number + " (" + identifier + ")");
        }
    }

    private int createNumberOfTasks(String identifier) {
        switch (identifier) {
            case IDENTIFIER_1:
                return NUMBER_OF_TASKS_1;
            case IDENTIFIER_2:
                return NUMBER_OF_TASKS_2;
            case IDENTIFIER_3:
                return NUMBER_OF_TASKS_3;
            default:
                throw new IllegalArgumentException("Unbekannter Identifier '" + identifier + "'.");
        }
    }

    private void runTasks(String identifier) {
        progressDialog.startingWithTask(identifier);

        for (String taskDescription : taskDescriptions) {
            runTask(identifier, taskDescription);
        }

        progressDialog.closeUi(identifier);
    }

    private void runTask(String identifier, String taskDescription) {
        progressDialog.aboutToExceuteOneTaskSoon(identifier, taskDescription);

        int sleepSeconds = 1 + RANDOM.nextInt(1);
        SystemTools.sleepSeconds(sleepSeconds);

        progressDialog.oneTaskDone(identifier, taskDescription);
    }

    public static void main(String[] args) {
        GuiTools.setNiceLayoutManager();
        MultipleProgressDialogDemo demo = new MultipleProgressDialogDemo();
        demo.run();
    }

}
