package de.duehl.vocabulary.japanese.data.symbol;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse stellt die Zeichen der Schrift Kanji (in UTF-Zeichen) dar, die ich bislang gelernt
 * habe. Da es davon über 50.000 Stück gibt, wird diese Klasse niemals vollständig sein.
 *
 * Ich hoffe aber, sie enthält irgendwann die 2.136 常用漢字 (Jōyō-Kanji), vgl.
 * https://de.wikipedia.org/wiki/J%C5%8Dy%C5%8D-Kanji.
 *
 * Die Klasse beinhaltet auch Methoden rund darum, um zu fragen, ob eine Zeichenkette
 * ausschließlich aus diesen Zeichen besteht oder wo das erste bzw. letzte dieser Zeichen zu finden
 * ist.
 *
 * Möchte man an die Übersetzungen, so kann man nach dem Wort über normalKanjiHiraganaWord oder
 * normalHiraganaWord suchen, das muss man nicht hier auch noch vorhalten.
 *
 * https://www-kanshudo-com.translate.goog/kanji/%E6%B0%97?_x_tr_sl=en&_x_tr_tl=de&_x_tr_hl=de&_x_tr_pto=rq
 *
 * @version 1.01     2026-01-23
 * @author Christian Dühl
 */

public enum Kanji {

    FISCH("魚",
            CollectionsHelper.buildListFrom("Fisch"),
            CollectionsHelper.buildListFrom("GYO"),
            CollectionsHelper.buildListFrom("sakana", "uo"),
            "魚", "さかな", 290, 112,
            1
            ),
    FLEISCH("肉",
            CollectionsHelper.buildListFrom("Fleisch"),
            CollectionsHelper.buildListFrom("NIKU"),
            CollectionsHelper.buildListFrom(),
            "肉", "にく", 223, 102,
            2
            ),
    EI("卵",
            CollectionsHelper.buildListFrom("Ei"),
            CollectionsHelper.buildListFrom("RAN"),
            CollectionsHelper.buildListFrom("tamago"),
            "卵", "たまご", 1103, 228,
            3
            ),
    WASSER("水",
            CollectionsHelper.buildListFrom("Wasser"),
            CollectionsHelper.buildListFrom("SUI"),
            CollectionsHelper.buildListFrom("mizu"),
            "水", "みず", 21, 73,
            4
            ),
    ESSEN("食",
            CollectionsHelper.buildListFrom("essen"),
            CollectionsHelper.buildListFrom("SHOKU", "[JIKI]"),
            CollectionsHelper.buildListFrom("ta(beru)", "ku(u)", "ku(rau)"),
            "食べる", "たべる", 322, 116,
            5
            ),
    TRINKEN("飲",
            CollectionsHelper.buildListFrom("trinken"),
            CollectionsHelper.buildListFrom("IN"),
            CollectionsHelper.buildListFrom("no(mu)"),
            "飲む", "のむ", 323, 117,
            6
            ),
    GROSS("大",
            CollectionsHelper.buildListFrom("groß"),
            CollectionsHelper.buildListFrom("DAI", "TAI"),
            CollectionsHelper.buildListFrom("oo(kii)", "oo-", "oo(ini)"),
            "大きい", "ちいさい", 26, 74,
            7
            ),
    KLEIN("小",
            CollectionsHelper.buildListFrom("klein"),
            CollectionsHelper.buildListFrom("SHOU"),
            CollectionsHelper.buildListFrom("chii(sai)", "ko-", "o-"),
            "小さい", "ちいさい", 27, 74,
            8
            ),
    NEU("新",
            CollectionsHelper.buildListFrom("neu"),
            CollectionsHelper.buildListFrom("SHIN"),
            CollectionsHelper.buildListFrom("atara(shii)", "ara(ta)", "nii-"),
            "新しい", "あたらしい", 174, 95,
            9
            ),
    ALT("古",
            CollectionsHelper.buildListFrom("alt"),
            CollectionsHelper.buildListFrom("KO"),
            CollectionsHelper.buildListFrom("furu(i)", "furu(su)"),
            "古い", "ふるい", 172, 95,
            10
            ),
    TAG_SONNE("日",
            CollectionsHelper.buildListFrom("Tag", "Sonne"),
            CollectionsHelper.buildListFrom("NICHI", "JITSU"),
            CollectionsHelper.buildListFrom("hi", "ka"),
            "日", "にち", 5, 71,
            11
            ),
    BUCH_URSPRUNG("本",
            CollectionsHelper.buildListFrom("Buch", "Ursprung", "Haupt-"),
            CollectionsHelper.buildListFrom("HON"),
            CollectionsHelper.buildListFrom("moto"),
            "本", "ほん", 25, 74,
            12
            ),
    PERSON_MENSCH("人",
            CollectionsHelper.buildListFrom("Person", "Mensch"),
            CollectionsHelper.buildListFrom("JIN", "NIN"),
            CollectionsHelper.buildListFrom("hito"),
            "人", "ひと", 1, 71,
            13
            ),
    GEHEN_FAHREN("行",
            CollectionsHelper.buildListFrom("gehen", "fahren", "fliegen"),
            CollectionsHelper.buildListFrom("KOU", "[AN]", "GYOU"),
            CollectionsHelper.buildListFrom("i(ku)", "yu(ku)", "okona(u)"),
            "行く", "いく", 68, 80,
            14
            ),
    ELEKTRIZITAET("電",
            CollectionsHelper.buildListFrom("Elektrizität"),
            CollectionsHelper.buildListFrom("DEN"),
            CollectionsHelper.buildListFrom(),
            "電車", "でんしゃ", 108, 86,
            15
            ),
    AUTO_WAGEN_RAD("車",
            CollectionsHelper.buildListFrom("Auto", "Wagen", "Rad"),
            CollectionsHelper.buildListFrom("SHA"),
            CollectionsHelper.buildListFrom("kuruma"),
            "車", "くるま", 133, 89,
            16
            ),
    FRAU("女",
            CollectionsHelper.buildListFrom("Frau", "weiblich"),
            CollectionsHelper.buildListFrom("JO", "NYO", "[NYOU]"),
            CollectionsHelper.buildListFrom("onna", "me"),
            "女", "おんな", 102, 85,
            17
            ),
    MANN("男",
            CollectionsHelper.buildListFrom("Mann"),
            CollectionsHelper.buildListFrom("DAN", "NAN"),
            CollectionsHelper.buildListFrom("otoko"),
            "男", "おとこ", 101, 85,
            18
            ),
    SPRECHEN("話",
            CollectionsHelper.buildListFrom("sprechen", "reden"),
            CollectionsHelper.buildListFrom("WA"),
            CollectionsHelper.buildListFrom("hanashi", "hana(su)"),
            "話す", "はなす", 238, 104,
            19
            ),
    SICH_TREFFEN("会",
            CollectionsHelper.buildListFrom("sich treffen", "sich versammeln"),
            CollectionsHelper.buildListFrom("KAI", "E"),
            CollectionsHelper.buildListFrom("a(u)"),
            "会う", "あう", 158, 93,
            20
            ),
    SICH_AUSRUHEN("休",
            CollectionsHelper.buildListFrom("sich ausruhen", "ruhen", "ruhen lassen"),
            CollectionsHelper.buildListFrom("KYUU"),
            CollectionsHelper.buildListFrom("yasu(mu)", "yasu(meru)", "yasu(maru)"),
            "休む", "休", 60, 79,
            21
            ),
    SEHEN_ANSEHEN("見",
            CollectionsHelper.buildListFrom("sehen", "ansehen"),
            CollectionsHelper.buildListFrom("KEN"),
            CollectionsHelper.buildListFrom("mi(ru)", "mi(eru)", "mi(seru)"),
            "見る", "みる", 63, 79,
            22
            ),
    HIMMEL_PARADIES("天",
            CollectionsHelper.buildListFrom("Himmel", "Paradies"),
            CollectionsHelper.buildListFrom("TEN"),
            CollectionsHelper.buildListFrom("ame", "[ama]"),
            "天気", "てんき", 141, 91,
            23
            ),
    GEIST_SEELE_ENERGIE("気",
            CollectionsHelper.buildListFrom("Geist", "Seele", "Energie"),
            CollectionsHelper.buildListFrom("KI", "KE"),
            CollectionsHelper.buildListFrom(),
            "元気", "げんき", 134, 90,
            24
            ),
    STADT_ORTSCHAFT("町",
            CollectionsHelper.buildListFrom("Stadt", "Ortschaft"),
            CollectionsHelper.buildListFrom("CHOU"),
            CollectionsHelper.buildListFrom("machi"),
            "町", "まち", 182, 96,
            25
            ),
    LERNEN("学",
            CollectionsHelper.buildListFrom("lernen", "Wissenschaft", "Studium"),
            CollectionsHelper.buildListFrom("GAKU"),
            CollectionsHelper.buildListFrom("mana(bu)"),
            "大学", "だいがく", 109, 86,
            26
            ),
    BLUME("花",
            CollectionsHelper.buildListFrom("Blume", "Blüte"),
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("hana"),
            "花", "はな", 255, 107,
            27
            ),
    JETZT("今",
            CollectionsHelper.buildListFrom("jetzt", "gegenwärtig", "dieser"),
            CollectionsHelper.buildListFrom("KON", "KIN"),
            CollectionsHelper.buildListFrom("ima"),
            "今", "いま", 51, 78,
            28
            ),
    ZEIT_STUNDE("時",
            CollectionsHelper.buildListFrom("Zeit", "Stunde", "... Uhr"),
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("toki"),
            "時", "じ", 42, 76,
            29
            ),
    TEILEN_VERSTEHEN_TEIL_MINUTE("分",
            CollectionsHelper.buildListFrom("teilen", "Teil", "Minute", "verstehen", "Prozent",
                    "geteilt sein"),
            CollectionsHelper.buildListFrom("BUN", "BU", "FUN", "PUN"),
            CollectionsHelper.buildListFrom("wa(keru)", "wa(katsu)", "wa(kareru)", "wa(karu)"),
            "分かる", "わかる", 38, 76,
            30
            ),
    SPITZE_ZUVOR("先",
            CollectionsHelper.buildListFrom("Spitze", "zuvor", "früher", "voraus", "künftig"),
            CollectionsHelper.buildListFrom("SEN"),
            CollectionsHelper.buildListFrom("saki"),
            "先", "さき", 50, 78,
            31
            ),
    GEBAEREN_LEBEN("生",
            CollectionsHelper.buildListFrom("gebären", "leben", "lebendig sein", "beleben"),
            CollectionsHelper.buildListFrom("SEI", "SHOU"),
            CollectionsHelper.buildListFrom("i(kiru)", "i(keru)", "i(kasu)", "u(mu)", "ha(yasu)",
                    "ha(eru)", "o(u)"),
            "生む", "うむ", 44, 77,
            32
            ),
    SPRACHE("語",
            CollectionsHelper.buildListFrom("Sprache", "sprechen", "erzählen"),
            CollectionsHelper.buildListFrom("GO"),
            CollectionsHelper.buildListFrom("kata(ru)", "kata(rau)"),
            "語", "ご", 67, 80,
            33
            ),
    REGEN("雨",
            CollectionsHelper.buildListFrom("Regen"),
            CollectionsHelper.buildListFrom("U"),
            CollectionsHelper.buildListFrom("ame", "[ama]"),
            "雨", "あめ", 30, 75,
            34
            ),
    KOMMEN("来",
            CollectionsHelper.buildListFrom("kommen", "führen zu"),
            CollectionsHelper.buildListFrom("RAI"),
            CollectionsHelper.buildListFrom("ku(ru)", "kita(ru)", "kita(su)"),
            "来る", "くる", 69, 80,
            35
            ),
    FEUER("火",
            CollectionsHelper.buildListFrom("Feuer"),
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("hi", "[ho]"),
            "火", "ひ", 20, 73,
            36
            ),
    BAUM_HOLZ("木",
            CollectionsHelper.buildListFrom("Baum", "Holz"),
            CollectionsHelper.buildListFrom("BOKU", "MOKU"),
            CollectionsHelper.buildListFrom("ki", "[ka]"),
            "木", "き", 22, 74,
            37
            ),
    WENIG("少",
            CollectionsHelper.buildListFrom("wenig", "gering", "ein wenig", "etwas"),
            CollectionsHelper.buildListFrom("SHOU"),
            CollectionsHelper.buildListFrom("suko(shi)", "suku(nai)"),
            "少し", "すこし", 144, 91,
            38
            ),
    VIEL_ZAHLREICH("多",
            CollectionsHelper.buildListFrom("viel",  "zahlreich", "viele", "groß"),
            CollectionsHelper.buildListFrom("TA"),
            CollectionsHelper.buildListFrom("oo(i)"),
            "多い", "おおい", 229, 103,
            39
            ),
    WAS("何",
            CollectionsHelper.buildListFrom("was", "welcher", "wie viele"),
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("nani", "[nan]"),
            "何", "なに", 392, 126,
            40
            ),
    HOEREN_FRAGEN("聞",
            CollectionsHelper.buildListFrom("hören", "fragen", "zuhören", "vernehmbar sein"),
            CollectionsHelper.buildListFrom("BUN", "MON"),
            CollectionsHelper.buildListFrom("ki(ku)", "ki(koru)"),
            "聞く", "きく", 64, 80,
            41
            ),
    KAUFEN("買",
            CollectionsHelper.buildListFrom("kaufen"),
            CollectionsHelper.buildListFrom("BAI"),
            CollectionsHelper.buildListFrom("ka(u)"),
            "買う", "かう", 241, 105,
            42
            ),
    HOCH_TEUER("高",
            CollectionsHelper.buildListFrom("hoch", "teuer", "steigen", "zunehmen"),
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("taka(i)", "taka", "taka(maru)", "taka(meru)"),
            "高い", "たかい", 190, 98,
            43
            ),
    BILLIG_SICHER("安",
            CollectionsHelper.buildListFrom("billig", "(sicher)", "Friede", "Ruhe"),
            CollectionsHelper.buildListFrom("AN"),
            CollectionsHelper.buildListFrom("yasu(i)"),
            "安い", "やすい", 105, 85,
            44
            ),
    KIND("子",
            CollectionsHelper.buildListFrom("Kind"),
            CollectionsHelper.buildListFrom("SHI", "SU"),
            CollectionsHelper.buildListFrom("ko"),
            "子供", "こども", 103, 85,
            45
            ),
    WEISS("白",
            CollectionsHelper.buildListFrom("Weiß"),
            CollectionsHelper.buildListFrom("HAKU", "BYAKU"),
            CollectionsHelper.buildListFrom("shiro(i)", "shiro"),
            "白", "しろ", 205, 100,
            46
            ),
    LADEN_GESCHAEFT("店",
            CollectionsHelper.buildListFrom("Laden", "Geschäft"),
            CollectionsHelper.buildListFrom("TEN"),
            CollectionsHelper.buildListFrom("mise"),
            "店", "みせ", 168, 94,
            47
            ),
    MOND_MONAT("月",
            CollectionsHelper.buildListFrom("Mond", "Monat"),
            CollectionsHelper.buildListFrom("GETSU", "GATSU"),
            CollectionsHelper.buildListFrom("tsuki"),
            "月", "つき", 17, 73,
            48
            ),
    FUENF_5("五",
            CollectionsHelper.buildListFrom("fünf", "5"),
            CollectionsHelper.buildListFrom("GO"),
            CollectionsHelper.buildListFrom("itsu(tsu)", "itsu-"),
            "五", "ご", 7, 71,
            49
            ),
    MITTE_INNEN("中",
            CollectionsHelper.buildListFrom("Mitte", "innen", "China", "Inneres", "zwischen"),
            CollectionsHelper.buildListFrom("CHUU", "-CHUU", "-JUU"),
            CollectionsHelper.buildListFrom("naka"),
            "中", "なか", 28, 74,
            50
            ),
    MITTAG("午",
            CollectionsHelper.buildListFrom("Mittag"),
            CollectionsHelper.buildListFrom("GO"),
            CollectionsHelper.buildListFrom(),
            "午後", "ごご", 49, 77,
            51
            ),
    EINS_1("一",
            CollectionsHelper.buildListFrom("eins", "1"),
            CollectionsHelper.buildListFrom("ICHI", "ITSU"),
            CollectionsHelper.buildListFrom("hito(tsu)", "hito-"),
            "一", "いち", 2, 71,
            52
            ),
    ZWEI_2("二",
            CollectionsHelper.buildListFrom("zwei", "2"),
            CollectionsHelper.buildListFrom("NI"),
            CollectionsHelper.buildListFrom("futa(tsu)", "futa-"),
            "二", "に", 3, 71,
            53
            ),
    DREI_3("三",
            CollectionsHelper.buildListFrom("drei", "3"),
            CollectionsHelper.buildListFrom("SAN"),
            CollectionsHelper.buildListFrom("mit(tsu)", "mi(tsu)", "mi-"),
            "三", "ご", 4, 71,
            54
            ),
    VIER_4("四",
            CollectionsHelper.buildListFrom("vier", "4"),
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("yot(tsu)", "yo(tsu)", "yo-", "yon"),
            "四", "し", 6, 71,
            55
            ),
    SECHS_6("六",
            CollectionsHelper.buildListFrom("sechs", "6"),
            CollectionsHelper.buildListFrom("ROKU"),
            CollectionsHelper.buildListFrom("mut(tsu)", "mu(tsu)", "mu-", "[mui-]"),
            "六", "ろく", 8, 72,
            56
            ),
    SIEBEN_7("七",
            CollectionsHelper.buildListFrom("sieben", "7"),
            CollectionsHelper.buildListFrom("SHICHI"),
            CollectionsHelper.buildListFrom("nana(tsu)", "nana-", "[nano-]"),
            "七", "なな", 9, 72,
            57
            ),
    ACHT_8("八",
            CollectionsHelper.buildListFrom("acht", "8"),
            CollectionsHelper.buildListFrom("HACHI"),
            CollectionsHelper.buildListFrom("yat(tsu)", "ya(tsu)", "ya-", "[yo-]"),
            "八", "はち", 10, 72,
            58
            ),
    NEUN_9("九",
            CollectionsHelper.buildListFrom("neun", "9"),
            CollectionsHelper.buildListFrom("KYUU", "KU"),
            CollectionsHelper.buildListFrom("kokono(tsu)", "kokono-"),
            "九", "きゅう", 11, 72,
            59
            ),
    ZEHN_10("十",
            CollectionsHelper.buildListFrom("zehn", "10"),
            CollectionsHelper.buildListFrom("JUU", "JI'"),
            CollectionsHelper.buildListFrom("tou", "to-"),
            "十", "じゅう", 12, 72,
            60
            ),
    HUNDERT_100("百",
            CollectionsHelper.buildListFrom("hundert", "100"),
            CollectionsHelper.buildListFrom("HYAKU"),
            CollectionsHelper.buildListFrom(),
            "百", "ひゅく", 14, 72,
            61
            ),
    TAUSEND_1000("千",
            CollectionsHelper.buildListFrom("tausend", "1000"),
            CollectionsHelper.buildListFrom("SEN"),
            CollectionsHelper.buildListFrom("chi-"),
            "千", "せん", 15, 73,
            62
            ),
    GOLD("金",
            CollectionsHelper.buildListFrom("Gold", "Metall", "Geld"),
            CollectionsHelper.buildListFrom("KIN", "KON"),
            CollectionsHelper.buildListFrom("kane", "[kana]"),
            "金", "きん", 23, 74,
            63
            ),
    BODEN_ERDE_GRUND("土",
            CollectionsHelper.buildListFrom("Boden", "Erde", "Grund"),
            CollectionsHelper.buildListFrom("DO", "TO"),
            CollectionsHelper.buildListFrom("tsuchi"),
            "土", "ど", 24, 74,
            64
            ),
    OBEN_RAUF_UEBER("上",
            CollectionsHelper.buildListFrom("oben", "rauf", "über", "steigen", "erheben"),
            CollectionsHelper.buildListFrom("JOU", "[SHOU]"),
            CollectionsHelper.buildListFrom("ue", "kami", "[uwa-]", "a(geru)", "nobo(ru)",
                    "nobo(seru)", "nobo(su)"),
            "上", "うえ", 32, 75,
            65
            ),
    UNTEN_RUNTER_UNTER("下",
            CollectionsHelper.buildListFrom("unten", "runter", "unter", "hinuntergehen",
                    "herabsteigen"),
            CollectionsHelper.buildListFrom("KA", "GE"),
            CollectionsHelper.buildListFrom("shita", "moto", "shimo", "sa(geru)", "o(riru)",
                    "kuda(ru)", "kuda(saru)"),
            "下", "した", 31, 75,
            66
            ),
    LINKS("左",
            CollectionsHelper.buildListFrom("links"),
            CollectionsHelper.buildListFrom("SA"),
            CollectionsHelper.buildListFrom("hidari"),
            "左", "ひだり", 75, 81,
            67
            ),
    RECHTS("右",
            CollectionsHelper.buildListFrom("rechts"),
            CollectionsHelper.buildListFrom("U", "YUU"),
            CollectionsHelper.buildListFrom("migi"),
            "右", "みぎ", 76, 81,
            68
            ),
    KREIS_YEN_RUND("円",
            CollectionsHelper.buildListFrom("Kreis", "Yen", "rund", "kreisrund"),
            CollectionsHelper.buildListFrom("EN"),
            CollectionsHelper.buildListFrom("maru(i)"),
            "円", "えん", 13, 72,
            69
            ),
    AUGE("目",
            CollectionsHelper.buildListFrom("Auge"),
            CollectionsHelper.buildListFrom("MOKU", "[BOKU]"),
            CollectionsHelper.buildListFrom("me", "[ma]"),
            "目", "め", 55, 78,
            70
            ),
    HAELFTE_HALB("半",
            CollectionsHelper.buildListFrom("Hälfte", "Halb-", "halb", "halb nach", "mitten in",
                    "Mitte"),
            CollectionsHelper.buildListFrom("HAN"),
            CollectionsHelper.buildListFrom("naka(ba)"),
            "半", "はん", 88, 83,
            71
            ),
    OHR("耳",
            CollectionsHelper.buildListFrom("Ohr"),
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("mimi"),
            "耳", "みみ", 56, 78,
            72
            ),
    MUND("口",
            CollectionsHelper.buildListFrom("Mund", "mündlich", "reden", "Job", "Anfang"),
            CollectionsHelper.buildListFrom("KOU", "KU"),
            CollectionsHelper.buildListFrom("kuchi"),
            "口", "くち", 54, 78,
            73
            ),
    HAND("手",
            CollectionsHelper.buildListFrom("Hand"),
            CollectionsHelper.buildListFrom("SHU"),
            CollectionsHelper.buildListFrom("te", "[ta]"),
            "手", "て", 57, 79,
            74
            ),
    BEIN_FUSS("足",
            CollectionsHelper.buildListFrom("Bein", "Fuß", "Pfote", "addieren", "Schritt"),
            CollectionsHelper.buildListFrom("SOKU"),
            CollectionsHelper.buildListFrom("ashi", "ta(ru)", "ta(iru)", "ta(su)"),
            "足", "あし", 58, 79,
            75
            ),
    KRAFT("力",
            CollectionsHelper.buildListFrom("Kraft"),
            CollectionsHelper.buildListFrom("RYOKU", "RIKI"),
            CollectionsHelper.buildListFrom("chikara"),
            "力", "ちから", 100, 85,
            76
            ),
    STEHEN_AUFSTEHEN("立",
            CollectionsHelper.buildListFrom("stehen", "aufstehen", "aufstellen", "errichten"),
            CollectionsHelper.buildListFrom("RITSU", "[RYUU]"),
            CollectionsHelper.buildListFrom("ta(tsu)", "ta(teru)"),
            "立つ", "たつ", 121, 88,
            77
            ),
    HIMMEL_LEER("空",
            CollectionsHelper.buildListFrom("Himmel", "leer", "frei machen", "frei werden"),
            CollectionsHelper.buildListFrom("KUU"),
            CollectionsHelper.buildListFrom("sora", "a(keru)", "a(ku)", "kara"),
            "空", "そら", 140, 90,
            78
            ),
    BERG("山",
            CollectionsHelper.buildListFrom("Berg"),
            CollectionsHelper.buildListFrom("SAN"),
            CollectionsHelper.buildListFrom("yama"),
            "山", "やま", 34, 75,
            79
            ),
    FLUSS("川",
            CollectionsHelper.buildListFrom("Fluss", "Strom"),
            CollectionsHelper.buildListFrom("SEN"),
            CollectionsHelper.buildListFrom("kawa"),
            "川", "かわ", 33, 75,
            80
            ),
    WORT_SAGEN("言",
            CollectionsHelper.buildListFrom("Wort", "sagen"),
            CollectionsHelper.buildListFrom("GEN", "GON"),
            CollectionsHelper.buildListFrom("-koto", "i(u)"),
            "言う", "いう", 66, 80,
            81
            ),
    LESEN("読",
            CollectionsHelper.buildListFrom("lesen"),
            CollectionsHelper.buildListFrom("DOKU", "TOKU", "[TOU]"),
            CollectionsHelper.buildListFrom("yo(mu)"),
            "読む", "よむ", 244, 105,
            82
            ),
    SCHREIBEN("書",
            CollectionsHelper.buildListFrom("schreiben"),
            CollectionsHelper.buildListFrom("SHO"),
            CollectionsHelper.buildListFrom("ka(ku)"),
            "書く", "かく", 131, 89,
            83
            ),
    WAELCHEN_WALD_FORST("林",
            CollectionsHelper.buildListFrom("Wäldchen", "Wald", "Forst"),
            CollectionsHelper.buildListFrom("RIN"),
            CollectionsHelper.buildListFrom("hayashi"),
            "林", "はやし", 127, 89,
            84
            ),
    WALD_FORST("森",
            CollectionsHelper.buildListFrom("Wald", "Forst"),
            CollectionsHelper.buildListFrom("SHIN"),
            CollectionsHelper.buildListFrom("mori"),
            "森", "もり", 128, 89,
            85
            ),
    STEIN("石",
            CollectionsHelper.buildListFrom("Stein"),
            CollectionsHelper.buildListFrom("SEKI", "[SHAKU]", "[KOKU]"),
            CollectionsHelper.buildListFrom("ishi"),
            "石", "いし", 78, 82,
            86
            ),
    HUND("犬",
            CollectionsHelper.buildListFrom("Hund"),
            CollectionsHelper.buildListFrom("KEN"),
            CollectionsHelper.buildListFrom("inu"),
            "犬", "いぬ", 280, 110,
            87
            ),
    INSEKT_WURM("虫",
            CollectionsHelper.buildListFrom("Insekt", "Wurm"),
            CollectionsHelper.buildListFrom("CHUU"),
            CollectionsHelper.buildListFrom("mushi"),
            "虫", "むし", 893, 198,
            88
            ),
    JAHR("年",
            CollectionsHelper.buildListFrom("Jahr"),
            CollectionsHelper.buildListFrom("NEN"),
            CollectionsHelper.buildListFrom("toshi"),
            "年", "ねん", 45, 77,
            89
            ),
    DORF("村",
            CollectionsHelper.buildListFrom("Dorf"),
            CollectionsHelper.buildListFrom("SON"),
            CollectionsHelper.buildListFrom("mura"),
            "村", "むら", 191, 98,
            90
            ),
    REISFELD("田",
            CollectionsHelper.buildListFrom("Reisfeld", "Feld"),
            CollectionsHelper.buildListFrom("DEN"),
            CollectionsHelper.buildListFrom("ta"),
            "田", "た", 35, 75,
            91
            ),
    ABEND_NACHT("夕",
            CollectionsHelper.buildListFrom("Abend", "Nacht"),
            CollectionsHelper.buildListFrom("SEKI"),
            CollectionsHelper.buildListFrom("yuu"),
            "夕べ", "ゆうべ", 81, 82,
            92
            ),
    ROT("赤",
            CollectionsHelper.buildListFrom("rot", "rot werden", "erröten"),
            CollectionsHelper.buildListFrom("SEKI", "[SHAKU]"),
            CollectionsHelper.buildListFrom("aka(i)", "aka", "aka(ramu)", "aka(rameru)"),
            "赤", "あか", 207, 100,
            93
            ),
    BLAU_GRUEN_UNREIF("青",
            CollectionsHelper.buildListFrom("blau", "grün", "unreif"),
            CollectionsHelper.buildListFrom("SEI", "[SHOU]"),
            CollectionsHelper.buildListFrom("ao(i)", "ao"),
            "青", "あお", 208, 100,
            94
            ),
    HERAUSNEHMEN_HERAUSKOMMEN("出",
            CollectionsHelper.buildListFrom("herausnehmen", "abschicken", "hinausgehen",
                    "herauskommen"),
            CollectionsHelper.buildListFrom("SHUTSU", "[SUI]"),
            CollectionsHelper.buildListFrom("da(su)", "de(ru)"),
            "出口", "でぐち", 53, 78,
            95
            ),
    HINEINGEHEN_EINLASSEN("入",
            CollectionsHelper.buildListFrom("hineingehen", "eintreten", "hereinstecken",
                    "einlassen"),
            CollectionsHelper.buildListFrom("NYUU"),
            CollectionsHelper.buildListFrom("hai(ru)", "i(ru)", "i(eru)"),
            "入り口", "いりぐち", 52, 78,
            96
            ),
    FRUEH_SCHNELL("早",
            CollectionsHelper.buildListFrom("früh", "schnell"),
            CollectionsHelper.buildListFrom("SOU", "[SA']"),
            CollectionsHelper.buildListFrom("haya(i)", "haya(maru)", "haya(meru)"),
            "早い", "はやい", 248, 106,
            97
            ),
    SATZ_TEXT_LITERATUR_BRIEF("文",
            CollectionsHelper.buildListFrom("Satz", "Text", "Literatur", "Brief"),
            CollectionsHelper.buildListFrom("BUN", "MON"),
            CollectionsHelper.buildListFrom("fumi"),
            "文学", "ぶんがく", 111, 86,
            98
            ),
    NORDEN("北",
            CollectionsHelper.buildListFrom("Norden"),
            CollectionsHelper.buildListFrom("HOKU"),
            CollectionsHelper.buildListFrom("kita"),
            "北", "きた", 73, 81,
            99
            ),
    OSTEN("東",
            CollectionsHelper.buildListFrom("Osten"),
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("higashi"),
            "東", "ひがし", 71, 81,
            100
            ),
    SUEDEN("南",
            CollectionsHelper.buildListFrom("Süden"),
            CollectionsHelper.buildListFrom("NAN", "[NA]"),
            CollectionsHelper.buildListFrom("minami"),
            "南", "みなみ", 74, 81,
            101
            ),
    WESTEN("西",
            CollectionsHelper.buildListFrom("Westen", "Spanien"),
            CollectionsHelper.buildListFrom("SEI", "SAI"),
            CollectionsHelper.buildListFrom("nishi"),
            "西", "にし", 72, 81,
            102
            ),
    NAME_RUF_REPUTATION("名",
            CollectionsHelper.buildListFrom("Name", "Ruf", "Reputation"),
            CollectionsHelper.buildListFrom("MEI", "MYOU"),
            CollectionsHelper.buildListFrom("na"),
            "名前", "なまえ", 82, 82,
            103
            ),
    SCHRIFTZEICHEN_BUCHSTABE("字",
            CollectionsHelper.buildListFrom("Schriftzeichen", "Buchstabe", "Weiler",
                    "Gemeindeteil"),
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("aza"),
            "文字", "もじ", 110, 86,
            104
            ),
    SCHULE_KORREKTUR("校",
            CollectionsHelper.buildListFrom("Schule", "Korrektur"),
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom(),
            "学校", "がっこう", 115, 87,
            105
            ),
    RICHTIG_GERECHT("正",
            CollectionsHelper.buildListFrom("richtig", "gerecht", "berichtigen", "verbessern",
                    "genau", "sicher"),
            CollectionsHelper.buildListFrom("SEI", "SHOU"),
            CollectionsHelper.buildListFrom("tada(shii)", "tada(su)", "masa(ni)"),
            "正す", "ただす", 275, 110,
            106
            ),
    KOENIG("王",
            CollectionsHelper.buildListFrom("König"),
            CollectionsHelper.buildListFrom("OU"),
            CollectionsHelper.buildListFrom(),
            "王様", "おうさま", 294, 112,
            107
            ),
    TON_LAUT("音",
            CollectionsHelper.buildListFrom("Ton", "Laut"),
            CollectionsHelper.buildListFrom("ON", "IN"),
            CollectionsHelper.buildListFrom("oto", "ne"),
            "音楽", "おんがく", 347, 120,
            108
            ),
    FADEN("糸",
            CollectionsHelper.buildListFrom("Faden"),
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("ito"),
            "糸", "いと", 242, 105,
            109
            ),
    MUSCHEL("貝",
            CollectionsHelper.buildListFrom("Muschel"),
            CollectionsHelper.buildListFrom(),
            CollectionsHelper.buildListFrom("kai"),
            "貝", "かい", 240, 105,
            110
            ),
    KUGEL_BALL("玉",
            CollectionsHelper.buildListFrom("Edelstein", "Juwel", "Kugel", "Ball"),
            CollectionsHelper.buildListFrom("GYOKU"),
            CollectionsHelper.buildListFrom("tama"),
            "玉", "たま", 295, 113,
            111
            ),
    GRAS_KRAUT("草",
            CollectionsHelper.buildListFrom("Gras", "Kraut", "Entwurf"),
            CollectionsHelper.buildListFrom("SOU"),
            CollectionsHelper.buildListFrom("kusa"),
            "草", "くさ", 249, 106,
            112
            ),
    BAMBUS("竹",
            CollectionsHelper.buildListFrom("Bambus"),
            CollectionsHelper.buildListFrom("CHIKU"),
            CollectionsHelper.buildListFrom("take"),
            "竹", "たけ", 129, 89,
            113
            ),
    ZEHNTAUSEND("万",
            CollectionsHelper.buildListFrom("10000", "10.000", "zehntausend", "Myriaden", "viele",
                    "alle"),
            CollectionsHelper.buildListFrom("MAN", "BAN"),
            CollectionsHelper.buildListFrom(),
            "万", "まん", 16, 73,
            114
            ),
    ZIEHEN_BILLIGER_MACHEN("引",
            CollectionsHelper.buildListFrom("ziehen", "anziehen", "schließen", "billiger machen"),
            CollectionsHelper.buildListFrom("IN"),
            CollectionsHelper.buildListFrom("hi(ku)"),
            "引く", "ひく", 216, 101,
            115
            ),
    WOLKE("雲",
            CollectionsHelper.buildListFrom("Wolke"),
            CollectionsHelper.buildListFrom("UN"),
            CollectionsHelper.buildListFrom("kumo"),
            "雲", "くも", 646, 163,
            116
            ),
    WEIT_FERN_ENTFERNT("遠",
            CollectionsHelper.buildListFrom("weit", "fern", "entfernt"),
            CollectionsHelper.buildListFrom("EN", "[ON]"),
            CollectionsHelper.buildListFrom("too(i)"),
            "遠い", "とおい", 453, 135,
            117
            ),
    LEHRFACH("科",
            CollectionsHelper.buildListFrom("Lehrfach", "Abteilung", "Fakultät", "Schulfach"),
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom(),
            "科学", "かがく", 320, 116,
            118
            ),
    SOMMER("夏",
            CollectionsHelper.buildListFrom("Sommer"),
            CollectionsHelper.buildListFrom("KA", "[GE]"),
            CollectionsHelper.buildListFrom("natsu"),
            "夏", "なつ", 469, 137,
            119
            ),
    HAN_CHINA("漢",
            CollectionsHelper.buildListFrom("Han", "China", "Mann", "Kerl"),
            CollectionsHelper.buildListFrom("KAN"),
            CollectionsHelper.buildListFrom(),
            "漢字", "かんじ", 565, 151,
            120
            ),
    HAUPTSTADT("京",
            CollectionsHelper.buildListFrom("Hauptstadt"),
            CollectionsHelper.buildListFrom("KYOU", "KEI"),
            CollectionsHelper.buildListFrom(),
            "京都", "きょうと", 189, 97,
            121
            ),
    LICHT_HELL("明",
            CollectionsHelper.buildListFrom("Licht", "hell", "Helligkeit", "hell werden"),
            CollectionsHelper.buildListFrom("MEI", "MYOU"),
            CollectionsHelper.buildListFrom("a(kari)", "aka(rui)", "aki(raka)", "a(keru)",
                    "aka(rumu)", "aka(ramu)", "a(ku)", "a(kasu)", "a(kuru)"),
            "明かり", "あかり", 18, 73,
            122
            ),
    HAUS_HEIM("家",
            CollectionsHelper.buildListFrom("Haus", "Heim"),
            CollectionsHelper.buildListFrom("KA", "KE"),
            CollectionsHelper.buildListFrom("ie", "ya"),
            "家", "いえ", 165, 94,
            123
            ),
    GEDICHT_LIED_SINGEN("歌",
            CollectionsHelper.buildListFrom("Gedicht", "Lied", "singen"),
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom("uta", "uta(u)"),
            "歌", "うた", 395, 127,
            124
            ),
    BILD_STRICH("画",
            CollectionsHelper.buildListFrom("Bild", "Strich"),
            CollectionsHelper.buildListFrom("GA", "KAKU"),
            CollectionsHelper.buildListFrom(),
            "映画", "えいが", 343, 119,
            125
            ),
    MAL_WIEDERKEHR_WEITERGABE_DREHEN("回",
            CollectionsHelper.buildListFrom("Mal", "Wiederkehr", "Weitergabe", "drehen"),
            CollectionsHelper.buildListFrom("KAI", "[E]"),
            CollectionsHelper.buildListFrom("mawa(su)", "mawa(ru)"),
            "一回", "いっかい", 90, 83,
            126
            ),
    MEER_SEE("海",
            CollectionsHelper.buildListFrom("Meer", "See"),
            CollectionsHelper.buildListFrom("KAI"),
            CollectionsHelper.buildListFrom("umi"),
            "海", "うみ", 117, 87,
            127
            ),
    JEDER_JEDES("毎",
            CollectionsHelper.buildListFrom("jeder", "jeden", "jedes"),
            CollectionsHelper.buildListFrom("MAI"),
            CollectionsHelper.buildListFrom(),
            "毎日", "まいにち", 116, 87,
            128
            ),
    EINZELSTUECK("箇",
            CollectionsHelper.buildListFrom("Einzelstück", "Stück"),
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom(),
            "箇所", "かしょ", 1570, 295,
            129
            ),
    EINZELSTUECK_KURZFORM("ヶ",
            CollectionsHelper.buildListFrom("Einzelstück", "Stück"),
            CollectionsHelper.buildListFrom("KA"),
            CollectionsHelper.buildListFrom(),
            "一ヶ月", "いっかげつ", 1570, 295,
            129
            ),
    ICH_PRIVAT("私",
            CollectionsHelper.buildListFrom("ich", "privat"),
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("watakushi", "watashi"),
            "私", "わたし", 125, 88,
            130
            ),
    KATZE("猫",
            CollectionsHelper.buildListFrom("Katze"),
            CollectionsHelper.buildListFrom("BYOU"),
            CollectionsHelper.buildListFrom("neko"),
            "猫", "ねこ", 1567, 294,
            131
            ),
    BAHNHOF("駅",
            CollectionsHelper.buildListFrom("Bahnhof"),
            CollectionsHelper.buildListFrom("EKI"),
            CollectionsHelper.buildListFrom(),
            "駅", "えき", 284, 111,
            132
            ),
    OEFFENTLICH_OFFIZIELL("公",
            CollectionsHelper.buildListFrom("öffentlich", "offiziell"),
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("ooyake"),
            "公園", "こうえん", 126, 88,
            133
            ),
    GARTEN_PARK("園",
            CollectionsHelper.buildListFrom("Garten", "Park"),
            CollectionsHelper.buildListFrom("EN"),
            CollectionsHelper.buildListFrom("sono"),
            "公園", "こうえん", 454, 135,
            134
            ),
    AUSSENSEITE_DRAUSSEN("外",
            CollectionsHelper.buildListFrom("Außenseite", "draußen", "ein anderer", "ein weiterer"),
            CollectionsHelper.buildListFrom("GAI", "GE"),
            CollectionsHelper.buildListFrom("soto", "hoka", "hazu(reru)", "hazu(su)"),
            "外", "そと", 83, 82,
            135
            ),
    ZEICHNUNG_PLAN("図",
            CollectionsHelper.buildListFrom("Zeichnung", "Plan", "Skizze", "planen"),
            CollectionsHelper.buildListFrom("ZU", "TO"),
            CollectionsHelper.buildListFrom("haka(ru)"),
            "図書館", "としょかん", 339, 119,
            136
            ),
    GEBAUEDE_HALLE("館",
            CollectionsHelper.buildListFrom("großes Gebäude", "Gebäude", "Halle"),
            CollectionsHelper.buildListFrom("KAN"),
            CollectionsHelper.buildListFrom("yakata"),
            "図書館", "としょかん", 327, 117,
            137
            ),
    MOEGEN_LIEBEN("好",
            CollectionsHelper.buildListFrom("gern haben", "mögen", "lieben"),
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("kono(mu)", "su(ku)"),
            "好き", "すき", 104, 85,
            138
            ),
    TEE("茶",
            CollectionsHelper.buildListFrom("Teestrauch", "Teeblätter", "Tee"),
            CollectionsHelper.buildListFrom("CHA", "SA"),
            CollectionsHelper.buildListFrom(),
            "お茶", "おちゃ", 251, 106,
            139
            ),
    VOR_VORNE_FRUEHER("前",
            CollectionsHelper.buildListFrom("vorne", "vor", "früher"),
            CollectionsHelper.buildListFrom("ZEN"),
            CollectionsHelper.buildListFrom("mae"),
            "前", "まえ", 47, 77,
            140
            ),
    HINTER_DANACH_SPAETER("後",
            CollectionsHelper.buildListFrom("danach", "später", "Rückseite", "hinten", "nach",
                    "hinter", "zurück"),
            CollectionsHelper.buildListFrom("GO", "KOU"),
            CollectionsHelper.buildListFrom("nochi", "ushi(ro)", "ato", "oku(reru)"),
            "後ろ", "うしろ", 48, 77,
            141
            ),
    SPRACHE_DIALEKT_UNTERSCHEIDUNG_BLATT("弁",
            CollectionsHelper.buildListFrom("Sprache", "Dialekt", "Unterscheidung", "Blatt"),
            CollectionsHelper.buildListFrom("BEN"),
            CollectionsHelper.buildListFrom(),
            "弁当", "べんとう", 724, 174,
            142
            ),
    TREFFEN_ZUTREFFEN("当",
            CollectionsHelper.buildListFrom("treffen", "zutreffen"),
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("a(teru)", "a(taru)"),
            "弁当", "べんとう", 77, 81,
            143
            ),
    GOTTHEIT_GOTT("神",
            CollectionsHelper.buildListFrom("Gottheit", "Gott"),
            CollectionsHelper.buildListFrom("SHIN", "JIN"),
            CollectionsHelper.buildListFrom("kami", "[kan]", "[kou]"),
            "神", "かみ", 310, 115,
            144
            ),
    SHINTO_SCHREIN_GESELLSCHAFT_FIRMA("社",
            CollectionsHelper.buildListFrom("Shinto-Schrein", "Gesellschaft", "Firma"),
            CollectionsHelper.buildListFrom("SHA"),
            CollectionsHelper.buildListFrom("yashiro"),
            "会社", "かいしゃ", 308, 114,
            145
            ),
    MACHEN_ANFERTIGEN("作",
            CollectionsHelper.buildListFrom("machen", "anfertigen", "zubereiten", "herstellen"),
            CollectionsHelper.buildListFrom("SAKU", "SA"),
            CollectionsHelper.buildListFrom("tsuku(ru)"),
            "作る", "べんとう", 361, 122,
            146
            ),
    SCHLOSS_BURG("城",
            CollectionsHelper.buildListFrom("Schloss", "Burg"),
            CollectionsHelper.buildListFrom("JOU"),
            CollectionsHelper.buildListFrom("shiro"),
            "城", "しろ", 733, 175,
            147
            ),
    TEMPEL("寺",
            CollectionsHelper.buildListFrom("Tempel"),
            CollectionsHelper.buildListFrom("JI"),
            CollectionsHelper.buildListFrom("tera"),
            "寺", "てら", 41, 76,
            148
            ),
    ANSTRENGUNG_HARTE_ARBEIT("勉",
            CollectionsHelper.buildListFrom("Anstrengung", "harte Arbeit"),
            CollectionsHelper.buildListFrom("BEN"),
            CollectionsHelper.buildListFrom(),
            "勉強", "べんきょう", 748, 177,
            149
            ),
    STAR_STAERKER_WERDEN("強",
            CollectionsHelper.buildListFrom("stark", "stärker werden", "zwingen", "verstärken"),
            CollectionsHelper.buildListFrom("KYOU", "GOU"),
            CollectionsHelper.buildListFrom("tsuyo(i)", "tsuyo(maru)", "tsuyo(meru)", "shi(iru)"),
            "勉強", "べんきょう", 217, 101,
            150
            ),
    LEHREN_UNTERRICHTEN("教",
            CollectionsHelper.buildListFrom("lehren", "unterrichten", "lernen"),
            CollectionsHelper.buildListFrom("KYOU"),
            CollectionsHelper.buildListFrom("oshi(eru)", "oso(waru)"),
            "教える", "おしえる", 245, 105,
            151
            ),
    LAND_STAAT("国",
            CollectionsHelper.buildListFrom("Land", "Staat"),
            CollectionsHelper.buildListFrom("KOKU"),
            CollectionsHelper.buildListFrom("kuni"),
            "国", "くに", 40, 76,
            152
            ),
    EINSTEIGEN("乗",
            CollectionsHelper.buildListFrom("einsteigen", "reiten", "fahren"),
            CollectionsHelper.buildListFrom("JOU"),
            CollectionsHelper.buildListFrom("no(ru)", "no(seru)"),
            "乗る", "のる", 532, 146,
            153
            ),
    STADT_MARKT("市",
            CollectionsHelper.buildListFrom("Stadt", "Markt"),
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("ichi"),
            "市", "し", 181, 96,
            154
            ),
    PRAEFEKTUR("県",
            CollectionsHelper.buildListFrom("Präfektur"),
            CollectionsHelper.buildListFrom("KEN"),
            CollectionsHelper.buildListFrom(),
            "県", "けん", 194, 98,
            155
            ),
    GESICHT_MASKE_FLAECHE("面",
            CollectionsHelper.buildListFrom("Gesicht", "Maske", "Fläche", "Oberfläche", "Seite"),
            CollectionsHelper.buildListFrom("MEN"),
            CollectionsHelper.buildListFrom("omote", "omo", "tsura"),
            "面白い", "おもしろい", 274, 110,
            156
            ),
    KOPF_URSPRUNG_ANFANG("元",
            CollectionsHelper.buildListFrom("Kopf", "Ursprung", "Anfang"),
            CollectionsHelper.buildListFrom("GEN", "GAN"),
            CollectionsHelper.buildListFrom("moto"),
            "元気", "げんき", 137, 90,
            157
            ),
    SEIN_EXISTIEREN("有",
            CollectionsHelper.buildListFrom("sein", "existieren", "sich befinden", "haben"),
            CollectionsHelper.buildListFrom("YUU", "U"),
            CollectionsHelper.buildListFrom("a(ru)"),
            "有名", "ゆうめい", 265, 108,
            158
            ),
    SCHNEIDEN_ZU_ENDE_GEHEN("切",
            CollectionsHelper.buildListFrom("schneiden", "gut schneiden", "zu Ende gehen"),
            CollectionsHelper.buildListFrom("SETSU", "[SAI]"),
            CollectionsHelper.buildListFrom("ki(ru)", "ki(eru)"),
            "大切", "たいせつ", 39, 76,
            159
            ),
    UEBERSCHREITEN_UEBERGEBEN("渡",
            CollectionsHelper.buildListFrom("überschreiten", "übergeben"),
            CollectionsHelper.buildListFrom("TO"),
            CollectionsHelper.buildListFrom("wata(ru)", "wata(su)"),
            "渡す", "わたす", 379, 125,
            160
            ),
    NEHMEN("取",
            CollectionsHelper.buildListFrom("nehmen"),
            CollectionsHelper.buildListFrom("SHU"),
            CollectionsHelper.buildListFrom("to(ru)"),
            "取る", "とる", 65, 80,
            161
            ),
    SETZEN_LEGEN_DALSSSEN("置",
            CollectionsHelper.buildListFrom("setzen", "stellen", "legen", "dalassen", "lassen",
                    "errichten"),
            CollectionsHelper.buildListFrom("CHI"),
            CollectionsHelper.buildListFrom("o(ku)"),
            "置く", "おく", 432, 132,
            162
            ),
    SCHNEE("雪",
            CollectionsHelper.buildListFrom("Schnee"),
            CollectionsHelper.buildListFrom("SETSU"),
            CollectionsHelper.buildListFrom("yuki"),
            "雪", "ゆき", 979, 210,
            163
            ),
    FALLEN("降",
            CollectionsHelper.buildListFrom("fallen", "herabsteigen", "aussteigen", "regnen",
                    "schneien"),
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("o(riru)", "o(rosu)", "fu(ru)"),
            "降る", "ふる", 977, 210,
            164
            ),
    WIND("風",
            CollectionsHelper.buildListFrom("Wind", "Aussehen", "Mode", "Stil"),
            CollectionsHelper.buildListFrom("FUU", "[FU]"),
            CollectionsHelper.buildListFrom("kaze", "[kaza]"),
            "風", "かぜ", 29, 75,
            165
            ),
    BLASEN_WEHEN("吹",
            CollectionsHelper.buildListFrom("blasen", "wehen"),
            CollectionsHelper.buildListFrom("SUI"),
            CollectionsHelper.buildListFrom("fu(ku)"),
            "吹く", "ふく", 1326, 260,
            166
            ),
    HEISS__WETTER_("暑",
            CollectionsHelper.buildListFrom("heiß", "heiß (Wetter)"),
            CollectionsHelper.buildListFrom("SHO"),
            CollectionsHelper.buildListFrom("atsu(i)"),
            "暑", "あつい", 648, 163,
            167
            ),
    HITZE_FIEBER_HEISS("熱",
            CollectionsHelper.buildListFrom("Hitze", "Fieber", "heiß", "heiß (anfassbar)"),
            CollectionsHelper.buildListFrom("NETSU"),
            CollectionsHelper.buildListFrom("atsu(i)"),
            "熱い", "あつい", 655, 164,
            168
            ),
    KAELTE_WINTER_KALT("寒",
            CollectionsHelper.buildListFrom("Kälte", "kalte Jahreszeit", "Winter", "kalt"),
            CollectionsHelper.buildListFrom("KAN"),
            CollectionsHelper.buildListFrom("samu(i)"),
            "寒い", "さむい", 464, 137,
            169
            ),
    KALT_KUEHLEN_KALT_WERDEN("冷",
            CollectionsHelper.buildListFrom("kalt", "kühl", "kühlen", "kalt werden",
                    "kaltes Wasser", "kalter Sake"),
            CollectionsHelper.buildListFrom("REI"),
            CollectionsHelper.buildListFrom("tsume(tai)", "hi(yasu)", "sa(masu)", "hi(eru)",
                    "sa(meru)", "hi(ya)", "hi(yakasu)"),
            "冷たい", "おく", 851, 192,
            170
            ),
    WARM_WARM_WERDEN_SICH_ERWAERMEN__WETTER__("暖",
            CollectionsHelper.buildListFrom("warm", "erwärmen", "aufwärmen", "warm werden",
                    "sich erwärmen", "angenehm warm", "(warmes Wetter)"),
            CollectionsHelper.buildListFrom("DAN"),
            CollectionsHelper.buildListFrom("atata(kai)", "atata(ka)", "atata(meru)", "atata(maru)"),
            "暖かい", "あたたかい", 645, 163,
            171
            ),
    WARM_WARM_WERDEN_SICH_ERWAERMEN__ANFASSEN__("温",
            CollectionsHelper.buildListFrom("warm", "erwärmen", "aufwärmen", "warm werden",
                    "sich erwärmen", "angenehm warm", "(warm beim Anfassen)"),
            CollectionsHelper.buildListFrom("ON"),
            CollectionsHelper.buildListFrom("atata(kai)", "atata(ka)", "atata(meru)", "atata(maru)"),
            "温かい", "あたたかい", 644, 162,
            172
            ),
    KUEHL_FRISCH_ERFRISCHEND("涼",
            CollectionsHelper.buildListFrom("kühl", "angenehm kühl", "frisch", "erfrischend",
                    "sich draußen erfrischen", "frische Luft schöpfen"),
            CollectionsHelper.buildListFrom("RYOU"),
            CollectionsHelper.buildListFrom("suzu(shii)", "suzu(mu)"),
            "涼しい", "すずしい", 1267, 251,
            173
            ),
    SCHLECHT_SCHLIMM_BOESE("悪",
            CollectionsHelper.buildListFrom("schlecht", "schlimm", "böse"),
            CollectionsHelper.buildListFrom("AKU", "O"),
            CollectionsHelper.buildListFrom("waru(i)"),
            "悪い", "わるい", 304, 114,
            174
            ),
    PFERD("馬",
            CollectionsHelper.buildListFrom("Pferd"),
            CollectionsHelper.buildListFrom("BA"),
            CollectionsHelper.buildListFrom("uma", "[ma]"),
            "馬", "うま", 283, 111,
            175
            ),
    VOGEL("鳥",
            CollectionsHelper.buildListFrom("Vogel"),
            CollectionsHelper.buildListFrom("CHOU"),
            CollectionsHelper.buildListFrom("tori"),
            "鳥", "とり", 285, 111,
            176
            ),
    RIND_KUH("牛",
            CollectionsHelper.buildListFrom("Rind", "Kuh"),
            CollectionsHelper.buildListFrom("GYUU"),
            CollectionsHelper.buildListFrom("ushi"),
            "牛", "うし", 281, 111,
            177
            ),
    SCHWEIN_HAUSSCHWEIN("豚",
            CollectionsHelper.buildListFrom("Schwein", "Hausschwein"),
            CollectionsHelper.buildListFrom("TON"),
            CollectionsHelper.buildListFrom("buta"),
            "豚", "ぶた", 810, 186,
            178
            ),
    BEWEGUNG_SICH_BEWEGEN("動",
            CollectionsHelper.buildListFrom("Bewegung", "sich bewegen", "bewegen"),
            CollectionsHelper.buildListFrom("DOU"),
            CollectionsHelper.buildListFrom("ugo(ku)", "ugo(kasu)"),
            "動物", "どうぶつ", 231, 103,
            179
            ),
    DING_SACHE_GEGENSTAND("物",
            CollectionsHelper.buildListFrom("Ding", "Sache", "Gegenstand"),
            CollectionsHelper.buildListFrom("BUTSU", "MOTSU"),
            CollectionsHelper.buildListFrom("mono"),
            "動物", "どうぶつ", 79, 82,
            180
            ),
    MUSIK_BEQUEMLICHKEIT_ERFREULICH("楽",
            CollectionsHelper.buildListFrom("Musik", "Bequemlichkeit", "sich freuen", "genießen",
                    "froh", "erfreulich"),
            CollectionsHelper.buildListFrom("GAKU", "RAKU"),
            CollectionsHelper.buildListFrom("tano(shimu)", "tano(shii)"),
            "楽しい", "たのしい", 359, 122,
            181
            ),
    SCHWIERIG("難",
            CollectionsHelper.buildListFrom("schwierig", "kompliziert"),
            CollectionsHelper.buildListFrom("NAN"),
            CollectionsHelper.buildListFrom("muzuka(shii)", "kata(i)"),
            "難しい", "むずかしい", 566, 151,
            182
            ),
    BEQUEMLICHKEIT_EXKREMENTE_POST("便",
            CollectionsHelper.buildListFrom("Bequemlichkeit", "Exkremente", "Gelegenheit", "Post",
                    "Nachricht", "Brief"),
            CollectionsHelper.buildListFrom("BEN", "BIN"),
            CollectionsHelper.buildListFrom("tayo(ri)"),
            "便利", "べんり", 330, 118,
            183
            ),
    VORTEIL_ZINS_WIRKEN("利",
            CollectionsHelper.buildListFrom("Vorteil", "Zins", "wirksam sein", "wirken"),
            CollectionsHelper.buildListFrom("RI"),
            CollectionsHelper.buildListFrom("ki(ku)"),
            "便利", "べんり", 329, 117,
            184
            ),
    UN_NICHT("不",
            CollectionsHelper.buildListFrom("un-", "nicht"),
            CollectionsHelper.buildListFrom("FU", "BU"),
            CollectionsHelper.buildListFrom(),
            "不便", "ふべん", 94, 84,
            185
            ),
    WACHE_DIENST_NUMMER("番",
            CollectionsHelper.buildListFrom("Wache", "Aufsicht", "Dienst", "Reihe", "Nummer"),
            CollectionsHelper.buildListFrom("BAN"),
            CollectionsHelper.buildListFrom(),
            "番号", "ばんごう", 185, 97,
            186
            ),
    NUMMER_PSEUDONYM("号",
            CollectionsHelper.buildListFrom("Nummer", "Pseudonym", "verzweifeltes Schreien",
                    "verzweifeltes lautes Schreien"),
            CollectionsHelper.buildListFrom("GOU"),
            CollectionsHelper.buildListFrom(),
            "番号", "ばんごう", 266, 108,
            187
            ),
    BENUTZUNG_BOTE_BENUTZEN("使",
            CollectionsHelper.buildListFrom("Benutzung", "Bote", "benutzen"),
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("tsuka(u)"),
            "使う", "つかう", 331, 118,
            188
            ),
    SELBST("自",
            CollectionsHelper.buildListFrom("selbst"),
            CollectionsHelper.buildListFrom("JI", "SHI"),
            CollectionsHelper.buildListFrom("mizuka(ra)"),
            "自転車", "じてんしゃ", 62, 79,
            189
            ),
    ROLLEN_FALLEN_WAELZEN("転",
            CollectionsHelper.buildListFrom("rollen", "fallen", "wälzen", "hinfallen", "umwerfen"),
            CollectionsHelper.buildListFrom("TEN"),
            CollectionsHelper.buildListFrom("koro(bu)", "koro(garu)", "koro(geru)", "koro(gasu)"),
            "自転車", "じてんしゃ", 439, 133,
            190
            ),
    SCHWERT_MESSER("刀",
            CollectionsHelper.buildListFrom("Schwert", "Messer"),
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("katana"),
            "刀", "かたな", 37, 76,
            191
            ),
    VATER("父",
            CollectionsHelper.buildListFrom("Vater"),
            CollectionsHelper.buildListFrom("FU"),
            CollectionsHelper.buildListFrom("chichi"),
            "父", "ちち", 113, 87,
            192
            ),
    MUTTER("母",
            CollectionsHelper.buildListFrom("Mutter"),
            CollectionsHelper.buildListFrom("BO"),
            CollectionsHelper.buildListFrom("haha"),
            "母", "はは", 112, 86,
            193
            ),
    GROSS_STARK_STATUR_GROESSE("丈",
            CollectionsHelper.buildListFrom("groß", "stark", "Statur", "Größe"),
            CollectionsHelper.buildListFrom("JOU"),
            CollectionsHelper.buildListFrom("take"),
            "大丈夫", "だいじょうぶ", 1408, 272,
            194
            ),
    EHEMANN_MANN("夫",
            CollectionsHelper.buildListFrom("Ehemann", "Mann"),
            CollectionsHelper.buildListFrom("FU", "[FUU]"),
            CollectionsHelper.buildListFrom("otto"),
            "大丈夫", "だいじょうぶ", 315, 115,
            195
            ),
    BITTEN_BETEN_WUENSCHEN("願",
            CollectionsHelper.buildListFrom("bitten", "beten", "wünschen"),
            CollectionsHelper.buildListFrom("GAN"),
            CollectionsHelper.buildListFrom("nega(u)"),
            "お願いします", "お願いします", 590, 155,
            196
            ),
    ORT_STELLE("所",
            CollectionsHelper.buildListFrom("Ort", "Stelle"),
            CollectionsHelper.buildListFrom("SHO"),
            CollectionsHelper.buildListFrom("tokoro"),
            "所", "ところ", 153, 92,
            197
            ),
    STRASSE_WEG_PFAD("道",
            CollectionsHelper.buildListFrom("Straße", "Weg", "Pfad", "Dao/Tao"),
            CollectionsHelper.buildListFrom("DOU", "[TOU]"),
            CollectionsHelper.buildListFrom("michi"),
            "道", "みち", 149, 92,
            198
            ),
    ZURUECKKEHREN("帰",
            CollectionsHelper.buildListFrom("zurückkehren", "nach Hause schicken"),
            CollectionsHelper.buildListFrom("KI"),
            CollectionsHelper.buildListFrom("kae(ru)", "kae(su)"),
            "帰る", "かえる", 317, 116,
            199
            ),
    BAUEN_ERRICHTEN("建",
            CollectionsHelper.buildListFrom("bauen", "errichten", "errichtet werden"),
            CollectionsHelper.buildListFrom("KEN", "[KON]"),
            CollectionsHelper.buildListFrom("ta(teru)", "ta(tsu)"),
            "建物", "たてもの", 913, 201,
            200
            ),
    STOCKWERK_ETAGE_EBENE("階",
            CollectionsHelper.buildListFrom("Stockwerk", "Etage", "Ebene"),
            CollectionsHelper.buildListFrom("KAI"),
            CollectionsHelper.buildListFrom(),
            "階", "かい", 597, 156,
            201
            ),
    ERDE_LAND("地",
            CollectionsHelper.buildListFrom("Erde", "Land"),
            CollectionsHelper.buildListFrom("CHI", "JI"),
            CollectionsHelper.buildListFrom(),
            "地下", "ちか", 118, 87,
            202
            ),
    BLATT_LAUB("葉",
            CollectionsHelper.buildListFrom("Blatt", "Laub"),
            CollectionsHelper.buildListFrom("YOU"),
            CollectionsHelper.buildListFrom("ha"),
            "言葉", "ことば", 253, 107,
            203
            ),
    GLAENZEND_BEGABT_ENGLAND("英",
            CollectionsHelper.buildListFrom("glänzend", "brillant", "begabt", "England"),
            CollectionsHelper.buildListFrom("EI"),
            CollectionsHelper.buildListFrom(),
            "英語", "えいご", 354, 121,
            204
            ),
    QUALITAET_NATUR_PFAD("質",
            CollectionsHelper.buildListFrom("Qualität", "Natur", "Pfad"),
            CollectionsHelper.buildListFrom("SHITSU", "SHICHI", "[CHI]"),
            CollectionsHelper.buildListFrom(),
            "質問", "しつもん", 176, 96,
            205
            ),
    FRAGE_PROBLEM_FRAGEN("問",
            CollectionsHelper.buildListFrom("Frage", "Problem", "fragen", "sich kümmern (um)"),
            CollectionsHelper.buildListFrom("MON"),
            CollectionsHelper.buildListFrom("to(i)", "[tou]", "to(u)"),
            "質問", "しつもん", 162, 94,
            206
            ),
    ANTWORT_ANTWORTEN("答",
            CollectionsHelper.buildListFrom("Antwort", "antworten"),
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("kota(e)", "kota(eru)"),
            "答え", "こたえ", 160, 93,
            207
            ),
    ER_JENER("彼",
            CollectionsHelper.buildListFrom("er", "jener"),
            CollectionsHelper.buildListFrom("HI"),
            CollectionsHelper.buildListFrom("kare", "[kano]"),
            "彼", "かれ", 1010, 215,
            208
            ),
    FARBE_EROTIK("色",
            CollectionsHelper.buildListFrom("Farbe", "Erotik"),
            CollectionsHelper.buildListFrom("SHOKU", "SHIKI"),
            CollectionsHelper.buildListFrom("iro"),
            "色", "いろ", 204, 100,
            209
            ),
    KANJI_WIEDERHOLUNGSZEICHEN("々",
            CollectionsHelper.buildListFrom("Wiederholungszeichen", "Kanji Wiederholungszeichen"),
            CollectionsHelper.buildListFrom(),
            CollectionsHelper.buildListFrom(),
            "々", "々", 0, 0,
            0
            ),
    FRUEHLING("春",
            CollectionsHelper.buildListFrom("Frühling", "Frühjahr"),
            CollectionsHelper.buildListFrom("SHUN"),
            CollectionsHelper.buildListFrom("haru"),
            "春", "はる", 468, 137,
            210
            ),
    HERBST("秋",
            CollectionsHelper.buildListFrom("Herbst"),
            CollectionsHelper.buildListFrom("SHUU"),
            CollectionsHelper.buildListFrom("aki"),
            "秋", "あき", 470, 138,
            211
            ),
    WINTER("冬",
            CollectionsHelper.buildListFrom("Winter"),
            CollectionsHelper.buildListFrom("TOU"),
            CollectionsHelper.buildListFrom("fuyu"),
            "冬", "ふゆ", 467, 137,
            212
            ),
    BILD("絵",
            CollectionsHelper.buildListFrom("Bild"),
            CollectionsHelper.buildListFrom("KAI", "E"),
            CollectionsHelper.buildListFrom(),
            "絵", "え", 345, 120,
            213
            ),
    ZWISCHENRAUM("間",
            CollectionsHelper.buildListFrom("Zwischenraum", "Zeitraum", "Raum", "Zimmer"),
            CollectionsHelper.buildListFrom("KAN", "KEN"),
            CollectionsHelper.buildListFrom("aida"),
            "間", "あいだ", 43, 77,
            214
            ),
    GESICHT("顔",
            CollectionsHelper.buildListFrom("Gesicht"),
            CollectionsHelper.buildListFrom("GAN"),
            CollectionsHelper.buildListFrom("kao"),
            "顔", "かお", 277, 110,
            215
            ),
    DAMPF("汽",
            CollectionsHelper.buildListFrom("Dampf"),
            CollectionsHelper.buildListFrom("KI"),
            CollectionsHelper.buildListFrom(),
            "汽車", "きしゃ", 135, 90,
            216
            ),
    VERGANGEN_GESTERN("昨",
            CollectionsHelper.buildListFrom("vergangen", "letzt", "gestern"),
            CollectionsHelper.buildListFrom("SAKU"),
            CollectionsHelper.buildListFrom(),
            "昨日", "きのう", 362, 122,
            217
            ),
    SCHMERZ_LEID_BITTER("苦",
            CollectionsHelper.buildListFrom("Schmerz", "Leid", "leiden", "quälen", "schmerzhaft",
                    "bitter"),
            CollectionsHelper.buildListFrom("KU"),
            CollectionsHelper.buildListFrom("kuru(shimu)", "kuru(shimeru)", "kuru(shii)",
                    "niga(i)", "niga(ru)"),
            "苦", "にがい", 554, 150,
            218
            ),
    KLAR_AUFKLAREN_AUFKLAEREN("晴",
            CollectionsHelper.buildListFrom("klar", "aufklaren", "klären", "aufklären"),
            CollectionsHelper.buildListFrom("SEI"),
            CollectionsHelper.buildListFrom("ha(re)", "ha(reru)", "ha(rasu)"),
            "晴れ", "はれ", 673, 167,
            219
            ),
    WEIT_AUSDEHNEN("広",
            CollectionsHelper.buildListFrom("breit", "weit", "ausdehnen", "erweitern",
                    "sich ausdehnen", "verbreiten", "sich verbreiten"),
            CollectionsHelper.buildListFrom("KOU"),
            CollectionsHelper.buildListFrom("hiro(i)", "hiro(geru)", "hiro(garu)", "hiro(meru)",
                    "hiro(maru)"),
            "広い", "ひろい", 707, 171,
            220
            ),
    SICH_BEWÖLKEN("曇",
            CollectionsHelper.buildListFrom("sich bewölken", "sich trüben"),
            CollectionsHelper.buildListFrom("DON"),
            CollectionsHelper.buildListFrom("kumo(ru)"),
            "曇る", "くもる", 647, 163,
            221
            ),
    STERN("星",
            CollectionsHelper.buildListFrom("Stern"),
            CollectionsHelper.buildListFrom("SEI", "[SHOU]"),
            CollectionsHelper.buildListFrom("hoshi"),
            "星", "ほし", 743, 177,
            222
            ),
    VERKEHR_PASSIEREN("通",
            CollectionsHelper.buildListFrom("Verkehr", "passieren", "verkehren", "hindurchgehen",
                    "durchlassen", "Straße"),
            CollectionsHelper.buildListFrom("TSUU", "[TSU]"),
            CollectionsHelper.buildListFrom("too(ru)", "too(su)", "kayo(u)"),
            "通り", "とおり", 150, 92,
            223
            ),
    SCHWARZ("黒",
            CollectionsHelper.buildListFrom("schwarz"),
            CollectionsHelper.buildListFrom("KOKU"),
            CollectionsHelper.buildListFrom("kuro(i)", "kuro"),
            "黒", "くろ", 206, 100,
            224
            ),
    SUCHEN("探",
            CollectionsHelper.buildListFrom("suchen", "tasten"),
            CollectionsHelper.buildListFrom("TAN"),
            CollectionsHelper.buildListFrom("sagu(ru)", "saga(su)"),
            "探す", "さがす", 544, 148,
            225
            ),
    DENKEN_GLAUBEN("思",
            CollectionsHelper.buildListFrom("denken", "glauben"),
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("omo(u)"),
            "思う", "おもう", 99, 85,
            226
            ),
    ZURUECKKEHREN_AUFBEGEHREN("戻",
            CollectionsHelper.buildListFrom("zurückkehren", "aufbegehren", "verstoßen",
                    "zurückgeben"),
            CollectionsHelper.buildListFrom("REI"),
            CollectionsHelper.buildListFrom("modo(ru)", "modo(su)"),
            "戻る", "もどる", 1304, 257,
            227
            ),
    BEGINN_SCHNUR("緒",
            CollectionsHelper.buildListFrom("Beginn", "Schnur", "Strick", "Riemen"),
            CollectionsHelper.buildListFrom("SHO", "[CHO]"),
            CollectionsHelper.buildListFrom("o"),
            "一緒に", "いっしょに", 881, 196,
            228
            ),
    PLATZ_STELLE_ORT("場",
            CollectionsHelper.buildListFrom("Platz", "Stelle", "Ort"),
            CollectionsHelper.buildListFrom("JOU"),
            CollectionsHelper.buildListFrom("ba"),
            "場所", "ばしょ", 154, 92,
            229
            ),
    WARTEN_ERWARTEN("待",
            CollectionsHelper.buildListFrom("warten", "erwarten"),
            CollectionsHelper.buildListFrom("TAI"),
            CollectionsHelper.buildListFrom("ma(tsu)"),
            "待って", "まって", 459, 136,
            230
            ),
    WIEDERGABE_FOTOS_MACHEN("撮",
            CollectionsHelper.buildListFrom("Wiedergabe", "Wiedergabe (Foto/Inhalt)",
                    "(Fotos) machen"),
            CollectionsHelper.buildListFrom("SATSU"),
            CollectionsHelper.buildListFrom("to(ru)"),
            "撮ります", "とります", 1621, 302,
            231
            ),
    AUFSCHREIBEN_NOTIEREN("記",
            CollectionsHelper.buildListFrom("aufschreiben", "notieren"),
            CollectionsHelper.buildListFrom("KI"),
            CollectionsHelper.buildListFrom("shiru(su)"),
            "日記", "にっき", 372, 124,
            232
            ),
    HAUS_DACH_GESCHAEFT_HAENDLER("屋",
            CollectionsHelper.buildListFrom("Haus", "Dach", "Geschäft", "Händler"),
            CollectionsHelper.buildListFrom("OKU"),
            CollectionsHelper.buildListFrom("ya"),
            "魚屋", "さかなや", 167, 94,
            233
            ),
    HANG_STEIGUNG_DAMM("阪",
            CollectionsHelper.buildListFrom("Hang", "Steigung", "Damm"),
            CollectionsHelper.buildListFrom("HAN"),
            CollectionsHelper.buildListFrom("(saka)"),
            "大阪", "おおさか", 450, 135,
            234
            ),
    WINKEL_ECKE_HORN("角",
            CollectionsHelper.buildListFrom("Winkel", "Viereck", "Ecke", "Horn"),
            CollectionsHelper.buildListFrom("KAKU"),
            CollectionsHelper.buildListFrom("kado", "tsuno"),
            "角", "かど", 481, 139,
            235
            ),
    NAHE("近",
            CollectionsHelper.buildListFrom("nahe", "Nähe"),
            CollectionsHelper.buildListFrom("KIN"),
            CollectionsHelper.buildListFrom("chika(i)"),
            "近く", "ちかく", 452, 135,
            236
            ),
    FORM_GESTALT("形",
            CollectionsHelper.buildListFrom("Form", "Gestalt"),
            CollectionsHelper.buildListFrom("KEI", "GYOU"),
            CollectionsHelper.buildListFrom("katachi", "kata"),
            "形", "かたち", 399, 127,
            237
            ),
    MESSEN_PLAN_SUMME("計",
            CollectionsHelper.buildListFrom("Messen", "Plan", "Summe", "messen", "rechnen",
                    "verfahren", "vorgehen", "behandeln"),
            CollectionsHelper.buildListFrom("KEI"),
            CollectionsHelper.buildListFrom("haka(ru)", "haka(rau)"),
            "計算機", "けいさんき", 340, 119,
            238
            ),
    HAUPTSTADT_METROPOLIS("都",
            CollectionsHelper.buildListFrom("Hauptstadt", "Metropole", "Metropolis"),
            CollectionsHelper.buildListFrom("TO", "TSU"),
            CollectionsHelper.buildListFrom("miyako"),
            "京都", "きょうと", 188, 97,
            239
            ),
    WOCHENTAG_TAG("曜",
            CollectionsHelper.buildListFrom("Wochentag", "Tag"),
            CollectionsHelper.buildListFrom("YOU"),
            CollectionsHelper.buildListFrom(),
            "月曜日", "げつようび", 19, 73,
            240
            ),
    URSPRUENGLICH_ORIGINAL_FELD("原",
            CollectionsHelper.buildListFrom("ursprünglich", "original", "Grund-", "Feld",
                    "freies Feld", "Ebene"),
            CollectionsHelper.buildListFrom("GEN"),
            CollectionsHelper.buildListFrom("hara"),
            "原宿", "はらじく", 136, 90,
            241
            ),
    TUER("戸",
            CollectionsHelper.buildListFrom("Tür"),
            CollectionsHelper.buildListFrom("KO"),
            CollectionsHelper.buildListFrom("to"),
            "戸口", "とぐち", 152, 92,
            242
            ),
    JAHR_JAHRE_ALT("歳",
            CollectionsHelper.buildListFrom("Jahr", "Jahre alt"),
            CollectionsHelper.buildListFrom("SAI", "[SEI]"),
            CollectionsHelper.buildListFrom(),
            "二十歳", "はたち", 487, 140,
            243
            ),
    TALENT_BEGABUNG_JAHRE_ALT("才",
            CollectionsHelper.buildListFrom("Talent", "Begabung", "Jahre alt"),
            CollectionsHelper.buildListFrom("SAI"),
            CollectionsHelper.buildListFrom(),
            "天才", "てんさい", 560, 150,
            244
            ),
    HALTEN_ANHALTEN_STOPPEN("止",
            CollectionsHelper.buildListFrom("halten", "anhalten", "stoppen"),
            CollectionsHelper.buildListFrom("SHI"),
            CollectionsHelper.buildListFrom("to(maru)", "to(meru)"),
            "止まる", "とまる", 485, 140,
            245
            ),
    FELD_FLUR("野",
            CollectionsHelper.buildListFrom("Feld", "Flur"),
            CollectionsHelper.buildListFrom("YA"),
            CollectionsHelper.buildListFrom("no"),
            "野原", "のはら", 236, 104,
            246
            ),
    ;


    /** Das Kanji-Zeichen in UTF-8 Darstellung. */
    private final String character;

    /** Die deutschen Bedeutungen. */
    private final List<String> germanMeanings;

    /** Die Liste der sinojapanischen ON-Lesungen des Kanji (groß geschrieben). */
    private final List<String> onLesungen;

    /** Die Liste der rein japanischen kun-Lesungen des Kanji (klein geschrieben). */
    private final List<String> kunLesungen;

    /** Das Wort in Kanji und Hiragana in der häufigsten Bedeutung. */
    private final String normalKanjiHiraganaWord;

    /** Das Wort in Hiragana in der häufigsten Bedeutung. */
    private final String normalHiraganaWord;

    /** Die Nummer des Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky. */
    private final int numberInKanjiAndKanaBook;

    /**
     * Die Seite auf der das Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky zu finden ist.
     */
    private final int pageNumberInKanjiAndKanaBook;

    /** Die Nummer des Kanji in meinen handschriftlichen Heften. */
    private final int numberInMyHandwrittenKanjiNotebooks;

    /**
     * Konstruktor.
     *
     * @param character
     *            Das Kanji-Zeichen in UTF-8 Darstellung.
     * @param germanMeanings
     *            Die deutschen Bedeutungen.
     * @param onLesungen
     *            Die Liste der sinojapanischen ON-Lesungen des Kanji (groß geschrieben).
     * @param kunLesungen
     *            Die Liste der rein japanischen kun-Lesungen des Kanji (klein geschrieben).
     * @param normalKanjiHiraganaWord
     *            Das Wort in Kanji und Hiragana in der häufigsten Bedeutung.
     * @param normalHiraganaWord
     *            Das Wort in Hiragana in der häufigsten Bedeutung.
     * @param numberInKanjiAndKanaBook
     *            Die Nummer des Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky.
     * @param pageNumberInKanjiAndKanaBook
     *            Die Seite auf der das Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky zu
     *            finden ist.
     * @param numberInMyHandwrittenKanjiNotebooks
     *            Die Nummer des Kanji in meinen handschriftlichen Heften.
     */
    private Kanji(String character, List<String> germanMeanings, List<String> onLesungen,
            List<String> kunLesungen, String normalKanjiHiraganaWord, String normalHiraganaWord,
            int numberInKanjiAndKanaBook, int pageNumberInKanjiAndKanaBook,
            int numberInMyHandwrittenKanjiNotebooks) {
        this.character = character;
        this.germanMeanings = germanMeanings;
        this.onLesungen = onLesungen;
        this.kunLesungen = kunLesungen;
        this.normalKanjiHiraganaWord = normalKanjiHiraganaWord;
        this.normalHiraganaWord = normalHiraganaWord;
        this.numberInKanjiAndKanaBook = numberInKanjiAndKanaBook;
        this.pageNumberInKanjiAndKanaBook = pageNumberInKanjiAndKanaBook;
        this.numberInMyHandwrittenKanjiNotebooks = numberInMyHandwrittenKanjiNotebooks;
    }

    /** Getter für das Kanji-Zeichen in UTF-8 Darstellung. */
    public String getCharacter() {
        return character;
    }

    /** Getter für die deutschen Bedeutungen. */
    public List<String> getGermanMeanings() {
        return CollectionsHelper.copyList(germanMeanings);
    }

    /** Getter für die Liste der sinojapanischen ON-Lesungen des Kanji (groß geschrieben). */
    public List<String> getOnLesungen() {
        return CollectionsHelper.copyList(onLesungen);
    }

    /** Getter für die Liste der rein japanischen kun-Lesungen des Kanji (klein geschrieben). */
    public List<String> getKunLesungen() {
        return CollectionsHelper.copyList(kunLesungen);
    }

    /** Getter für das Wort in Kanji und Hiragana in der häufigsten Bedeutung. */
    public String getNormalKanjiHiraganaWord() {
        return normalKanjiHiraganaWord;
    }

    /** Getter für das Wort in Hiragana in der häufigsten Bedeutung. */
    public String getNormalHiraganaWord() {
        return normalHiraganaWord;
    }

    /** Getter für die Nummer des Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky. */
    public int getNumberInKanjiAndKanaBook() {
        return numberInKanjiAndKanaBook;
    }

    /**
     * Getter für die Seite auf der das Kanji im Buch "Kanji und Kana" von Wolfgang Hadamitzky zu
     * finden ist.
     */
    public int getPageNumberInKanjiAndKanaBook() {
        return pageNumberInKanjiAndKanaBook;
    }

    /** Getter für die Nummer des Kanji in meinen handschriftlichen Heften. */
    public int getNumberInMyHandwrittenKanjiNotebooks() {
        return numberInMyHandwrittenKanjiNotebooks;
    }

    /** Gibt eine Liste mit den bekannten Kanji zurück. */
    public static List<Kanji> getAllKanjiAsList() {
        return CollectionsHelper.arrayToList(Kanji.values());
    }

    /** Gibt eine Liste mit den Zeichen der Schrift Kanji (in UTF8-Zeichen) zurück. */
    public static List<String> getAllKanjiAsStringList() {
        List<String> kanjiList = new ArrayList<>();

        for (Kanji kanji : Kanji.values()) {
            kanjiList.add(kanji.getCharacter());
        }

        return kanjiList;
    }

    /** Gibt an, ob in dem Text ein Kanji-Zeichen vorkommt. */
    public static boolean containsKanji(String text) {
        return Text.contains(text, getAllKanjiAsStringList());
    }

    /** Gibt an, ob der Text ausschließlich aus Kanji-Zeichen besteht. */
    public static boolean containsOnlyKanji(String text) {
        if (text.isEmpty()) {
            return false;
        }
        else {
            List<String> kanji = getAllKanjiAsStringList();
            for (String textCharacter : Text.textToCharactersList(text)) {
                if (!kanji.contains(textCharacter)) {
                    return false;
                }
            }

            return true;
        }
    }

    /**
     * Gibt den Index des ersten Kanji-Zeichen im Text zurück. Gibt es kein solches, wird -1
     * zurückgegeben.
     */
    public static int firstIndexOfKanji(String text) {
        List<String> kanji = getAllKanjiAsStringList();
        List<String> textCharacters = Text.textToCharactersList(text);

        for (int index = 0; index < textCharacters.size(); ++index) {
            String textCharacter = textCharacters.get(index);
            if (kanji.contains(textCharacter)) {
                return index;
            }
        }

        return -1;
    }

    /**
     * Gibt den Index des letzten Kanji-Zeichen im Text zurück. Gibt es kein solches, wird -1
     * zurückgegeben.
     */
    public static int lastIndexOfKanji(String text) {
        List<String> kanji = getAllKanjiAsStringList();
        List<String> textCharacters = Text.textToCharactersList(text);

        for (int index = textCharacters.size() - 1; index >= 0; --index) {
            String textCharacter = textCharacters.get(index);
            if (kanji.contains(textCharacter)) {
                return index;
            }
        }

        return -1;
    }

    /** Gibt die Anzahl der bekannten Kanji zurück. */
    public static int getNumberOfKnownKanji() {
        return Kanji.values().length;
    }

    /** Ermittelt das Kanji zu einem Kanji-Charakter. */
    public static Kanji getByCharacter(String character) {
        for (Kanji kanji : Kanji.values()) {
            if (character.equals(kanji.getCharacter())) {
                return kanji;
            }
        }

        throw new RuntimeException("Kein Kanji zum Zeichen '" + character + "' gefunden.");
    }

    /**
     * Gibt eine Liste mit den im Text enthaltenen Kanji zurück.
     *
     * Die Kanji kommen in der Liste in der Reihenfolge ihres Vorkommens im Text vor und sind auch
     * mehrfach enthalten, wenn sie mehrfach im Text vorkommen.
     */
    public static List<Kanji> getAllKanjiFromText(String text) {
        List<Kanji> kanjiInText = new ArrayList<>();

        for (String textCharacter : Text.textToCharactersList(text)) {
            for (Kanji kanji : Kanji.values()) {
                String kanjiCharacter = kanji.getCharacter();
                if (textCharacter.equals(kanjiCharacter)) {
                    kanjiInText.add(kanji);
                }
            }
        }

        return kanjiInText;
    }

}
