package de.duehl.vocabulary.japanese.ui.components.text;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.Text;
import de.duehl.basics.text.html.HtmlTool;
import de.duehl.vocabulary.japanese.data.Vocable;

/**
 * Diese Klasse erstellt den Text zu Anzeige einer Vokabel in einer eigenen Vokabelliste oder der
 * Liste mit der Auswahl der Vokabeln dazu dar.
 *
 * Dieser wird nur in dem Dialog zum Bearbeiten einer Liste angezeigt (OwnListEditorDialog).
 *
 * @version 1.01     2024-03-12
 * @author Christian Dühl
 */

public class VocaleBarTextCreator {

    private static final String CENTERED_HTML_START = "<html><center>";
    private static final String CENTERED_HTML_END = "<center></html>";
    private static final String BIG_BOLD_START = "<span style=\"font-weight:bold; font-size:24\">";
    private static final String NORMAL_BOLD_START = "<span style=\"font-weight:bold; font-size:20\">";
    private static final String BOLD_END = "</span>";

    private static final int CHARACTERS_PER_LINE = 35;
    private static final String OPENING_BRACE_TEXT = " (";
    private static final String CLOSING_BRACE_TEXT = ") - ";

    private static final boolean INSERT_TYPE_INTO_TEXT = false;


    /** Die Vokabel in Kanji. */
    private final String kanji;

    /** Die Vokabel in Hiragana oder Katakana. */
    private final String kana;

    /** Die Vokabel in Romaji. */
    private final String romaji;

    /** De erste Übersetzung der Vokabel. */
    private final String firstTranslation;

    /** Der erzeugte Text. */
    private String text;

    /** Gibt an, ob die Kanji dargestellt werden sollen. */
    private boolean showKanji;

    /**
     * Konstruktor.
     *
     * @param vocable
     *            Die Vokabel zu der der text erzeugt werden soll.
     */
    public VocaleBarTextCreator(Vocable vocable) {
        kanji = vocable.getKanji();
        kana = vocable.getKana();
        romaji = vocable.getRomaji();
        firstTranslation = vocable.getTranslations().get(0);

        showKanji = !kanji.isBlank() && !kanji.equals(kana);
    }

    /** Erzeugt den Text für die VocabularyBar. */
    public void create() {
        String textBeforeBrace = createTextBeforeBrace();
        String firstTextInBrace = createFirstTextInBrace();
        String secondTextInBrace = romaji;
        String textAfterBrace = firstTranslation;

        String brace = OPENING_BRACE_TEXT + firstTextInBrace + secondTextInBrace
                + CLOSING_BRACE_TEXT;
        String possibleText = textBeforeBrace + brace + textAfterBrace;

        if (possibleText.length() <= CHARACTERS_PER_LINE) {
            if (showKanji) {
                text = createSingleLineTextWithKanji();
            }
            else {
                text = createSingleLineTextWithoutKanji();
            }
        }
        else if (textBeforeBrace.length() <= CHARACTERS_PER_LINE
                && brace.length() <= CHARACTERS_PER_LINE
                && textAfterBrace.length() <= CHARACTERS_PER_LINE
                ) {
            if (showKanji) {
                text = createThreeLinesTextWithKanji();
            }
            else {
                text = createThreeLinesTextWithoutKanji();
            }
        }
        else if (showKanji
                && textBeforeBrace.length() <= CHARACTERS_PER_LINE
                && firstTextInBrace.length() <= CHARACTERS_PER_LINE
                && secondTextInBrace.length() <= CHARACTERS_PER_LINE
                && textAfterBrace.length() <= CHARACTERS_PER_LINE
                ) {
            text = createFourLinesTextWithKanji();
        }
        else if (showKanji) {
            text = createLongerTextWithKanji();
        }
        else {
            text = createLongerTextWithoutKanji();
        }
    }

    private String createTextBeforeBrace() {
        if (showKanji) {
            return kanji;
        }
        else {
            return kana;
        }
    }

    private String createFirstTextInBrace() {
        if (showKanji) {
            return kana + ", ";
        }
        else {
            return "";
        }
    }

    private String createSingleLineTextWithKanji() {
        String kanjiHtml = BIG_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kanji) + BOLD_END;
        String kanaHtml = NORMAL_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kana) + BOLD_END;
        String romajiHtml = HtmlTool.encodeHtmlOnlyLtGt(romaji);
        String firstTranslationHtml = HtmlTool.encodeHtmlOnlyLtGt(firstTranslation);

        String html = CENTERED_HTML_START;
        if (INSERT_TYPE_INTO_TEXT) html += "EINS MIT KANJI" + "<br/>";
        html += kanjiHtml + OPENING_BRACE_TEXT + kanaHtml + ", ";
        html += romajiHtml + CLOSING_BRACE_TEXT + firstTranslationHtml;

        html += CENTERED_HTML_END;

        return html;
    }

    private String createSingleLineTextWithoutKanji() {
        String kanaHtml = BIG_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kana) + BOLD_END;
        String romajiHtml = HtmlTool.encodeHtmlOnlyLtGt(romaji);
        String firstTranslationHtml = HtmlTool.encodeHtmlOnlyLtGt(firstTranslation);

        String html = CENTERED_HTML_START;
        if (INSERT_TYPE_INTO_TEXT) html += "EINS OHNE KANJI" + "<br/>";
        html += kanaHtml + OPENING_BRACE_TEXT;
        html += romajiHtml + CLOSING_BRACE_TEXT + firstTranslationHtml;
        html += CENTERED_HTML_END;

        return html;
    }

    private String createThreeLinesTextWithKanji() {
        String kanjiHtml = BIG_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kanji) + BOLD_END;
        String kanaHtml = NORMAL_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kana) + BOLD_END;
        String romajiHtml = HtmlTool.encodeHtmlOnlyLtGt(romaji);
        String firstTranslationHtml = HtmlTool.encodeHtmlOnlyLtGt(firstTranslation);

        String html = CENTERED_HTML_START;
        if (INSERT_TYPE_INTO_TEXT) html += "DREI MIT KANJI" + "<br/>";
        html += kanjiHtml + "<br/>";
        html += "(" + kanaHtml + ", " + romajiHtml + ")" + "<br/>";
        html += firstTranslationHtml;
        html += CENTERED_HTML_END;

        return html;
    }

    private String createThreeLinesTextWithoutKanji() {
        String kanaHtml = BIG_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kana) + BOLD_END;
        String romajiHtml = HtmlTool.encodeHtmlOnlyLtGt(romaji);
        String firstTranslationHtml = HtmlTool.encodeHtmlOnlyLtGt(firstTranslation);

        String html = CENTERED_HTML_START;

        if (INSERT_TYPE_INTO_TEXT) html += "DREI OHNE KANJI" + "<br/>";
        html += kanaHtml + "<br/>";
        html += "(" + romajiHtml + ")" + "<br/>";
        html += firstTranslationHtml;
        html += CENTERED_HTML_END;

        return html;
    }

    private String createFourLinesTextWithKanji() {
        String kanjiHtml = BIG_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kanji) + BOLD_END;
        String kanaHtml = NORMAL_BOLD_START + HtmlTool.encodeHtmlOnlyLtGt(kana) + BOLD_END;
        String romajiHtml = HtmlTool.encodeHtmlOnlyLtGt(romaji);
        String firstTranslationHtml = HtmlTool.encodeHtmlOnlyLtGt(firstTranslation);

        String html = CENTERED_HTML_START;
        if (INSERT_TYPE_INTO_TEXT) html += "VIER" + "<br/>";
        html += kanjiHtml + "<br/>";
        html += "(" + kanaHtml + "," + "<br/>";
        html += romajiHtml + ")" + "<br/>";
        html += firstTranslationHtml;
        html += CENTERED_HTML_END;

        return html;
    }

    private String createLongerTextWithKanji() {
        List<String> kanjiLines = splitLongText(kanji);
        List<String> kanaLines = splitLongText(kana);
        List<String> romajiLines = splitLongText(romaji);
        List<String> firstTranslationLines = splitLongText(firstTranslation);

        String kanjiHtml = encodeHtmlOnlyLtGtAndJoinWithBr(kanjiLines);
        String kanaHtml = encodeHtmlOnlyLtGtAndJoinWithBr(kanaLines);
        String romajiHtml = encodeHtmlOnlyLtGtAndJoinWithBr(romajiLines);
        String firstTranslationHtml = encodeHtmlOnlyLtGtAndJoinWithBr(firstTranslationLines);

        kanjiHtml = BIG_BOLD_START + kanjiHtml + BOLD_END;
        kanaHtml = NORMAL_BOLD_START + kanaHtml + BOLD_END;

        String html = CENTERED_HTML_START;

        if (INSERT_TYPE_INTO_TEXT) html += "LANG MIT KANJI" + "<br/>";

        html += kanjiHtml + "<br/>";
        html += "(" + kanaHtml + "," + "<br/>";
        html += romajiHtml + ")" + "<br/>";
        html += firstTranslationHtml;

        html += CENTERED_HTML_END;

        return html;
    }

    private List<String> splitLongText(String input) {
        String inputWithLineBreaks = Text.addLineBreaks(input, CHARACTERS_PER_LINE);
        return Text.splitByLineBreaks(inputWithLineBreaks);
    }

    private String encodeHtmlOnlyLtGtAndJoinWithBr(List<String> lines) {
        List<String> outputLines = new ArrayList<>();

        for (String line : lines) {
            String outputLine = HtmlTool.encodeHtmlOnlyLtGt(line);
            outputLines.add(outputLine);
        }

        return Text.join("<br/>", outputLines);
    }

    private String createLongerTextWithoutKanji() {
        List<String> kanaLines = splitLongText(kana);
        List<String> romajiLines = splitLongText(romaji);
        List<String> firstTranslationLines = splitLongText(firstTranslation);

        String kanaHtml = encodeHtmlOnlyLtGtAndJoinWithBr(kanaLines);
        String romajiHtml = encodeHtmlOnlyLtGtAndJoinWithBr(romajiLines);
        String firstTranslationHtml = encodeHtmlOnlyLtGtAndJoinWithBr(firstTranslationLines);

        kanaHtml = BIG_BOLD_START + kanaHtml + BOLD_END;

        String html = CENTERED_HTML_START;

        if (INSERT_TYPE_INTO_TEXT) html += "LANG OHNE KANJI" + "<br/>";

        html += kanaHtml + "<br/>";
        html += "(" + romajiHtml + ")" + "<br/>";
        html += firstTranslationHtml;

        html += CENTERED_HTML_END;

        return html;
    }

    /** Getter für den erzeugten Text. */
    public String getText() {
        return text;
    }

}
