package de.duehl.basics.io.textfile;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.caller.CallerDeterminer;
import de.duehl.basics.io.Charset;

/**
 * Diese Klasse liest eine Liste mit den Texten im Format von Zuweisungen mit Gleichheitszeichen
 * oder Doppelpunkt aus den Zeilen einer Textdatei in einer Jar ein.
 *
 * Das Format sieht wie folgt aus:
 *
 *     Schlüssel : Wert
 *
 * oder
 *
 *     Schlüssel = Wert
 *
 * Kommt in einer Datenzeile weiter hinten das Kommentarzeichen vor, so wird ab dort abgeschnitten
 * und getrimmt.
 *
 * @version 1.01     2021-03-01
 * @author Christian Dühl
 */

public class DefinitionsFromTextFileInJarReader extends DefinitionsFromTextFileReader {

    /** Kommentarzeichen '#' für Konstruktoren ohne diese Angabe. */
    private static final String DEFAULT_COMMENT_CHARACTER = "#";

    /**
     * Konstruktor für UTF-8 Files mit Standard-Kommentarzeichen '#' und der Klasse des Aufrufers,
     * neben der also die Textdatei erwartet wird.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     */
    public DefinitionsFromTextFileInJarReader(String filename) {
        this(filename, Charset.UTF_8, CallerDeterminer.determineCallingClass(),
                DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor für UTF-8 Files mit Standard-Kommentarzeichen '#'.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     * @param clazz
     *            Klasse neben der die Datei sich im befindet.
     */
    public DefinitionsFromTextFileInJarReader(String filename, Class<?> clazz) {
        this(filename, Charset.UTF_8, clazz, DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     * @param charset
     *            Kodierung der einzulesenden Datei.
     * @param clazz
     *            Klasse neben der die Datei sich im befindet.
     * @param commentChar
     *            Das Kommentarzeichen.
     */
    public DefinitionsFromTextFileInJarReader(String filename, Charset charset, Class<?> clazz,
            String commentChar) {
        super(new StringsFromTextFileInJarReader(filename, charset, clazz, commentChar));
    }

}
