package de.duehl.basics.logging;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.Text;

/**
 * Diese Klasse stellt die Informationen zu einem Logeintrag dar, entspricht eine Zeile einer
 * Logdatei.
 *
 * @version 1.01     2017-05-04
 * @author Christian Dühl
 */

public class LogEntry {

    private final String date;
    private final String time;
    private final String className;
    private final String method;
    private final String lineNumber;
    private final String message;

    public LogEntry(String date, String time, String className, String method,
            String lineNumber, String message) {
        this.date = date;
        this.time = time;
        this.className = className;
        this.method = method;
        this.lineNumber = lineNumber;
        this.message = message;
    }

    public LogEntry(String line) {
        List<String> parts = Text.splitByTabulator(line);
        checkEnoughParts(line, parts);

        date = parts.get(0);
        time = parts.get(1);
        className = parts.get(2);
        method = parts.get(3);
        lineNumber = parts.get(4);

        if (parts.size() > 6) {
            List<String> toJoin = new ArrayList<>();
            for (int i = 5; i < parts.size(); ++i) {
                toJoin.add(parts.get(i));
            }
            message = Text.join("\t", toJoin);
        }
        else {
            this.message = parts.get(5);
        }
    }

    private void checkEnoughParts(String line, List<String> parts) {
        if (parts.size() < 6) {
            throw new IllegalArgumentException("Die Zeile hat keine sechs mit Tabulatoren "
                    + "unterteilten Spalten."
                    + "\n\t" + "Anzahl Elemente : " + parts.size()
                    + "\n\t" + "Zeile           : '" + line + "'");
        }
    }

    public String getDate() {
        return date;
    }

    public String getTime() {
        return time;
    }

    public String getClassName() {
        return className;
    }

    public String getMethod() {
        return method;
    }

    public String getLineNumber() {
        return lineNumber;
    }

    public String getMessage() {
        return message;
    }

    @Override
    public String toString() {
        return "LogLineElement [date=" + date + ", time=" + time + ", className=" + className
                + ", method=" + method + ", lineNumber=" + lineNumber + ", message=" + message
                + "]";
    }

    public String toNiceString() {
        return "LogEntry:\n\t"
                + "date       : " + date + "\n\t"
                + "time       : " + time + "\n\t"
                + "className  : " + className + "\n\t"
                + "method     : " + method + "\n\t"
                + "lineNumber : " + lineNumber + "\n\t"
                + "message    : " + message
                ;
    }

    /** Erzeugt aus dem Element eine Zeile für eine Logdatei. */
    public String asLine() {
        String messageToLog = message;
        messageToLog = Text.removeLineBreaks(messageToLog);
        messageToLog = Text.tabToSpace(messageToLog);
        return Text.join("\t", date, time, className, method, lineNumber, messageToLog);
    }

    public static String toNiceString(List<LogEntry> logEntries) {
        StringBuilder builder = new StringBuilder();

        for (int index = 0; index < logEntries.size(); ++index) {
            LogEntry logEntry = logEntries.get(index);
            int lineNumber = index + 1;
            builder.append(Text.fillWithSpacesAtFront(Integer.toString(lineNumber), 4));
            builder.append(") ");
            builder.append(logEntry.toNiceString());
            builder.append(Text.LINE_BREAK);
        }

        return builder.toString();
    }

}
