package de.duehl.basics.regex;

/*
 * Copyright 2018 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * Diese Klasse kann reguläre Ausdrücke kompilieren und danach über die Zustände befragt werden.
 * Das vermeidet das sinnlose Hantieren mit PatternSyntaxExceptions im Code.
 *
 * @version 1.01     2018-01-04
 * @author Christian Dühl
 */

public class RegexCompiler {

    /** Der reguläre Ausdruck in Textform. */
    private final String regex;

    /** Der kompilierte reguläre Ausdruck, nur sinnvoll, falls compilationSuccessfull true ist. */
    private Pattern pattern;

    /** Gibt an, ob das Kompilieren des regulären Ausdrucks erfolgreich war. */
    private boolean compilationSuccessfull;

    /**
     * Der beim Kompilieren des regulären Ausdrucks aufgetretene Ausnahme, nur sinnvoll, falls
     * compilationSuccessfull falls ist.
     */
    private PatternSyntaxException compileException;

    /**
     * Konstruktor.
     *
     * @param regex
     *            Der reguläre Ausdruck, der kompiliert werden soll.
     */
    public RegexCompiler(String regex) {
        this.regex = regex;
    }

    /** Kompiliert den regulären Ausdruck. */
    public void compile() {
        try {
            pattern = Pattern.compile(regex);
            compilationSuccessfull = true;
        }
        catch (PatternSyntaxException exception) {
            compileException = exception;
            compilationSuccessfull = false;
        }
    }

    /** Getter für den regulären Ausdruck in Textform. */
    public String getRegex() {
        return regex;
    }

    /**
     * Getter für den kompilierten regulären Ausdruck. Nur sinnvoll, falls
     * isCompilationSuccessfull() true ergibt.
     */
    public Pattern getPattern() {
        return pattern;
    }

    /** Gibt an, ob das Kompilieren des regulären Ausdrucks erfolgreich war. */
    public boolean isCompilationSuccessfull() {
        return compilationSuccessfull;
    }

    /**
     * Getter für die beim Kompilieren des regulären Ausdrucks aufgetretene Ausnahme. Nur sinnvoll,
     * falls isCompilationSuccessfull() falls ergibt.
     */
    public PatternSyntaxException getCompileException() {
        return compileException;
    }

    /** Stringrepräsentation. */
    @Override
    public String toString() {
        return "RegexCompiler [regex=" + regex + ", compilationSuccessfull="
                + compilationSuccessfull + "]";
    }

}
