package de.duehl.basics.datetime.time;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

public class ImmutualTimeTest {

    @Test
    public void create() {
        ImmutualTime time = new ImmutualTime();
        assertNotNull(time);
    }

    @Test
    public void createdTimeHasHours0() {
        ImmutualTime time = new ImmutualTime();
        int actual = time.getHour();
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void createdTimeHasMinutes0() {
        ImmutualTime time = new ImmutualTime();
        int actual = time.getMinute();
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void createdTimeHasSeconds0() {
        ImmutualTime time = new ImmutualTime();
        int actual = time.getSecond();
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void createdTimeHasDayCarryOver0() {
        ImmutualTime time = new ImmutualTime();
        int actual = time.getDayCarryOver();
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSeconds() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        assertNotNull(time);
    }

    @Test
    public void createWithHoursMinutesSecondsHasCorrectHours() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        int actual = time.getHour();
        int expected = hour;
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsHasCorrectMinutes() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        int actual = time.getMinute();
        int expected = minute;
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsHasCorrectSeconds() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        int actual = time.getSecond();
        int expected = second;
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsHasDayCarryOver0() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        int actual = time.getDayCarryOver();
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndGetHoursAsTwoDigitString() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.getHourAsTwoDigitString();
        String expected = "11";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithLowHoursMinutesSecondsAndGetHoursAsTwoDigitString() {
        int hour = 1;
        int minute = 2;
        int second = 4;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.getHourAsTwoDigitString();
        String expected = "01";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndGetMinutesAsTwoDigitString() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.getMinuteAsTwoDigitString();
        String expected = "23";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithLowHoursMinutesSecondsAndGetMinutesAsTwoDigitString() {
        int hour = 1;
        int minute = 2;
        int second = 4;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.getMinuteAsTwoDigitString();
        String expected = "02";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndGetSecondsAsTwoDigitString() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.getSecondAsTwoDigitString();
        String expected = "54";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithLowHoursMinutesSecondsAndGetSecondsAsTwoDigitString() {
        int hour = 1;
        int minute = 2;
        int second = 4;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.getSecondAsTwoDigitString();
        String expected = "04";
        assertEquals(expected, actual);
    }

    @Test
    public void createAndTestTwoDigitStrings() {
        ImmutualTime time = new ImmutualTime();
        String actualHours = time.getHourAsTwoDigitString();
        String actualMinutes = time.getMinuteAsTwoDigitString();
        String actualSeconds = time.getSecondAsTwoDigitString();
        assertEquals("00", actualHours);
        assertEquals("00", actualMinutes);
        assertEquals("00", actualSeconds);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithTooLowHours() {
        int hour = -1;
        int minute = 0;
        int second = 0;
        new ImmutualTime(hour, minute, second);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithTooHighHours() {
        int hour = 24;
        int minute = 0;
        int second = 0;
        new ImmutualTime(hour, minute, second);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithTooLowMinutes() {
        int hour = 0;
        int minute = -1;
        int second = 0;
        new ImmutualTime(hour, minute, second);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithTooHighMinutes() {
        int hour = 0;
        int minute = 60;
        int second = 0;
        new ImmutualTime(hour, minute, second);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithTooLowSeconds() {
        int hour = 0;
        int minute = 0;
        int second = -1;
        new ImmutualTime(hour, minute, second);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithTooHighSeconds() {
        int hour = 0;
        int minute = 0;
        int second = 60;
        new ImmutualTime(hour, minute, second);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOver() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        assertNotNull(time);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOverHasDayCarryOver0() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        int actual = time.getDayCarryOver();
        int expected = dayCarryOver;
        assertEquals(expected, actual);
    }

    @Test
    public void createDefaultAndTestToString() {
        ImmutualTime time = new ImmutualTime();
        String actual = time.toString();
        String expected = "00:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndTestToString() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.toString();
        String expected = "11:23:54";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithLowHoursMinutesSecondsAndTestToString() {
        int hour = 1;
        int minute = 2;
        int second = 4;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        String actual = time.toString();
        String expected = "01:02:04";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOverAndTestToString() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String actual = time.toString();
        String expected = "11:23:54 (-1 Tag)";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOverAndTestToString2() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -2;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String actual = time.toString();
        String expected = "11:23:54 (-2 Tage)";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOverAndTestToString3() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = 0;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String actual = time.toString();
        String expected = "11:23:54";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOverAndTestToString4() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = 1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String actual = time.toString();
        String expected = "11:23:54 (+1 Tag)";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOverAndTestToString5() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = 2;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String actual = time.toString();
        String expected = "11:23:54 (+2 Tage)";
        assertEquals(expected, actual);
    }

    @Test
    public void createWithHoursMinutesSecondsAndAddTwoHours() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        time = time.addHours(2);
        int actual = time.getHour();
        int expected = 13;
        assertEquals(expected, actual);
    }

    @Test
    public void createAndAddMinusTwoHours() {
        ImmutualTime time = new ImmutualTime();
        time = time.addHours(-2);

        int actualHours = time.getHour();
        int expectedHours = 22;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = -1;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndAddMinusTwentysixHours() {
        ImmutualTime time = new ImmutualTime();
        time = time.addHours(-26);

        int actualHours = time.getHour();
        int expectedHours = 22;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = -2;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndAddTwentysixHours() {
        ImmutualTime time = new ImmutualTime();
        time = time.addHours(26);

        int actualHours = time.getHour();
        int expectedHours = 2;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = 1;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndAddFiftyHours() {
        ImmutualTime time = new ImmutualTime();
        time = time.addHours(50);

        int actualHours = time.getHour();
        int expectedHours = 2;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = 2;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndAddTwoMinutes() {
        ImmutualTime time = new ImmutualTime();
        time = time.addMinutes(2);

        int actualMinutes = time.getMinute();
        int expectedMinutes = 2;
        assertEquals(expectedMinutes, actualMinutes);

        int actualHours = time.getHour();
        int expectedHours = 0;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = 0;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndSubtractTwoMinutes() {
        ImmutualTime time = new ImmutualTime();
        time = time.addMinutes(-2);

        int actualMinutes = time.getMinute();
        int expectedMinutes = 58;
        assertEquals(expectedMinutes, actualMinutes);

        int actualHours = time.getHour();
        int expectedHours = 23;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = -1;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndAddADayAndTwoMinutesInMinutes() {
        ImmutualTime time = new ImmutualTime();
        time = time.addMinutes(24 * 60 + 2);

        int actualMinutes = time.getMinute();
        int expectedMinutes = 2;
        assertEquals(expectedMinutes, actualMinutes);

        int actualHours = time.getHour();
        int expectedHours = 0;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = 1;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndSubtractADayAndTwoMinutesInMinutes() {
        ImmutualTime time = new ImmutualTime();
        time = time.addMinutes(-2 - 24 * 60);

        int actualMinutes = time.getMinute();
        int expectedMinutes = 58;
        assertEquals(expectedMinutes, actualMinutes);

        int actualHours = time.getHour();
        int expectedHours = 23;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = -2;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndAddTwoSeconds() {
        ImmutualTime time = new ImmutualTime();
        time = time.addSeconds(2);

        int actualSeconds = time.getSecond();
        int expectedSeconds = 2;
        assertEquals(expectedSeconds, actualSeconds);

        int actualMinutes = time.getMinute();
        int expectedMinutes = 0;
        assertEquals(expectedMinutes, actualMinutes);

        int actualHours = time.getHour();
        int expectedHours = 0;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = 0;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createAndSubstractTwoSeconds() {
        ImmutualTime time = new ImmutualTime();
        time = time.addSeconds(-2);

        int actualSeconds = time.getSecond();
        int expectedSeconds = 58;
        assertEquals(expectedSeconds, actualSeconds);

        int actualMinutes = time.getMinute();
        int expectedMinutes = 59;
        assertEquals(expectedMinutes, actualMinutes);

        int actualHours = time.getHour();
        int expectedHours = 23;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = -1;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void createWithHoursMinutesSecondsAndDayCarryOverAndForgetDayCarryOver() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);

        time = time.forgetDayCarryOver();

        int actualSeconds = time.getSecond();
        int expectedSeconds = 54;
        assertEquals(expectedSeconds, actualSeconds);

        int actualMinutes = time.getMinute();
        int expectedMinutes = 23;
        assertEquals(expectedMinutes, actualMinutes);

        int actualHours = time.getHour();
        int expectedHours = 11;
        assertEquals(expectedHours, actualHours);

        int actualDayCarryOver = time.getDayCarryOver();
        int expectedDayCarryOver = 0;
        assertEquals(expectedDayCarryOver, actualDayCarryOver);
    }

    @Test
    public void differenceTestWithTwoEqualTimes() {
        ImmutualTime time1 = new ImmutualTime();
        ImmutualTime time2 = new ImmutualTime();

        int actual = time1.difference(time2);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void differenceTest() {
        ImmutualTime time1 = new ImmutualTime();

        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time2 = new ImmutualTime(hour, minute, second);

        int actual = time1.difference(time2);
        int expected = 11*60*60 + 23*60 + 54;
        assertEquals(expected, actual);
    }

    @Test (expected = RuntimeException.class)
    public void differenceTestWithDayCarryOverOnCaller() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time1 = new ImmutualTime(hour, minute, second, dayCarryOver);

        ImmutualTime time2 = new ImmutualTime();

        time1.difference(time2);
    }

    @Test (expected = IllegalArgumentException.class)
    public void differenceTestWithDayCarryOverOnArgument() {
        ImmutualTime time1 = new ImmutualTime();

        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time2 = new ImmutualTime(hour, minute, second, dayCarryOver);

        time1.difference(time2);
    }

    @Test
    public void asHhMmSs() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String excpeted = "112354";
        String actual = time.asHhMmSs();
        assertEquals(excpeted, actual);
    }

    @Test
    public void toStingTestWithDayCarryOver() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String excpeted = "11:23:54 (-1 Tag)";
        String actual = time.toString();
        assertEquals(excpeted, actual);
    }

    @Test
    public void toStingTestWithoutDayCarryOver() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = 0;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String excpeted = "11:23:54";
        String actual = time.toString();
        assertEquals(excpeted, actual);
    }

    @Test
    public void toStringWithoutSecondsWithDayCarryOver() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String excpeted = "11:23 (-1 Tag)";
        String actual = time.toStringWithoutSeconds();
        assertEquals(excpeted, actual);
    }

    @Test
    public void toStringWithoutSecondsWithoutDayCarryOver() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = 0;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);
        String excpeted = "11:23";
        String actual = time.toStringWithoutSeconds();
        assertEquals(excpeted, actual);
    }

    @Test
    public void equalsTest() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time1 = new ImmutualTime(hour, minute, second, dayCarryOver);
        ImmutualTime time2 = new ImmutualTime(hour, minute, second, dayCarryOver);
        assertEquals(time1, time2);
    }

    @Test
    public void beforeWithTwoEqualTimes() {
        ImmutualTime time1 = new ImmutualTime();
        ImmutualTime time2 = new ImmutualTime();

        assertFalse(time1.before(time2));
    }

    @Test
    public void before() {
        int hour1 = 11;
        int minute1 = 23;
        int second1 = 53;
        ImmutualTime time1 = new ImmutualTime(hour1, minute1, second1);

        int hour2 = 11;
        int minute2 = 23;
        int second2 = 54;
        ImmutualTime time2 = new ImmutualTime(hour2, minute2, second2);

        assertTrue(time1.before(time2));
        assertFalse(time2.before(time1));
    }

    @Test (expected = RuntimeException.class)
    public void beforeWithDayCarryOverOnCaller() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time1 = new ImmutualTime(hour, minute, second, dayCarryOver);

        ImmutualTime time2 = new ImmutualTime();

        time1.before(time2);
    }

    @Test (expected = RuntimeException.class)
    public void beforeWithDayCarryOverOnArgument() {
        ImmutualTime time1 = new ImmutualTime();

        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time2 = new ImmutualTime(hour, minute, second, dayCarryOver);

        time1.before(time2);
    }

    @Test
    public void afterWithTwoEqualTimes() {
        ImmutualTime time1 = new ImmutualTime();
        ImmutualTime time2 = new ImmutualTime();

        assertFalse(time1.before(time2));
    }

    @Test
    public void after() {
        int hour1 = 11;
        int minute1 = 23;
        int second1 = 53;
        ImmutualTime time1 = new ImmutualTime(hour1, minute1, second1);

        int hour2 = 11;
        int minute2 = 23;
        int second2 = 54;
        ImmutualTime time2 = new ImmutualTime(hour2, minute2, second2);

        assertFalse(time1.after(time2));
        assertTrue(time2.after(time1));
    }

    @Test (expected = RuntimeException.class)
    public void afterWithDayCarryOverOnCaller() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time1 = new ImmutualTime(hour, minute, second, dayCarryOver);

        ImmutualTime time2 = new ImmutualTime();

        time1.after(time2);
    }

    @Test (expected = RuntimeException.class)
    public void afterWithDayCarryOverOnArgument() {
        ImmutualTime time1 = new ImmutualTime();

        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time2 = new ImmutualTime(hour, minute, second, dayCarryOver);

        time1.after(time2);
    }

    @Test (expected = RuntimeException.class)
    public void checkNoDayCarryOverPositive() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = 132;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);

        time.checkNoDayCarryOver();
    }

    @Test (expected = RuntimeException.class)
    public void checkNoDayCarryOverNegative() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        int dayCarryOver = -1;
        ImmutualTime time = new ImmutualTime(hour, minute, second, dayCarryOver);

        time.checkNoDayCarryOver();
    }

    @Test
    public void createFromHHMM1() {
        String timeHhMm = "12:30";
        ImmutualTime time = new ImmutualTime(timeHhMm);
        assertEquals(12, time.getHour());
        assertEquals(30, time.getMinute());
        assertEquals( 0, time.getSecond());
    }

    @Test
    public void createFromHHMMSS() {
        String timeHhMmSs = "12:30:00";
        ImmutualTime time = new ImmutualTime(timeHhMmSs);
        assertEquals(12, time.getHour());
        assertEquals(30, time.getMinute());
        assertEquals( 0, time.getSecond());
    }

    @Test (expected = IllegalArgumentException.class)
    public void failOnCreateFromHHMMSSPartNotDigitSequence() {
        String timeHhMmSs = "12:30:ab";
        new ImmutualTime(timeHhMmSs);
    }

    @Test (expected = IllegalArgumentException.class)
    public void failOnCreateFromHHMMSSTooSmallNumberOfParts() {
        String timeHhMmSs = "12";
        new ImmutualTime(timeHhMmSs);
    }

    @Test (expected = IllegalArgumentException.class)
    public void failOnCreateFromHHMMSSTooBigNumberOfParts() {
        String timeHhMmSs = "12:30:00:12";
        new ImmutualTime(timeHhMmSs);
    }

    @Test (expected = IllegalArgumentException.class)
    public void failOnCreateFromHHMMSSTooBigMinutes() {
        String timeHhMmSs = "12:99:00";
        new ImmutualTime(timeHhMmSs);
    }

    @Test
    public void setSecondsToZero() {
        int hour = 11;
        int minute = 23;
        int second = 54;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        time = time.setSecondsToZero();
        assertEquals(11, time.getHour());
        assertEquals(23, time.getMinute());
        assertEquals( 0, time.getSecond());
    }

    @Test
    public void setSecondsToZeroWithZeroSeconds() {
        int hour = 11;
        int minute = 23;
        int second = 0;
        ImmutualTime time = new ImmutualTime(hour, minute, second);
        time = time.setSecondsToZero();
        assertEquals(11, time.getHour());
        assertEquals(23, time.getMinute());
        assertEquals( 0, time.getSecond());
    }

}
