package de.duehl.swing.data;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.datetime.date.Weekday;

/**
 * Diese Klasse stellt die Daten dar, die z.B. mit einem DailyWeeklyNotSelector erhoben bzw.
 * verändert werden.
 *
 * Hier kann die Einstellung täglich, wöchentlich oder gar nicht getroffen werden. Für täglich
 * gibt es eine Uhrzeit einzustellen, für wöchentlich eine Uhrzeit und einen Wochentag.
 *
 * @version 1.01     2017-03-27
 * @author Christian Dühl
 */

public class DailyWeeklyNot {

    /** Typ der Auswahl: DAILY, WEEKLY oder NOT. */
    private DailyWeeklyNotType type;

    /** Uhrzeit für die tägliche Auswahl. */
    private MinuteHour dailyTime;

    /** Wochentag für die wöchentliche Auswahl. */
    private Weekday weeklyWeekday;

    /** Uhrzeit für die wöchentliche Auswahl. */
    private MinuteHour weeklyTime;

    /** Konstruktor. */
    public DailyWeeklyNot() {
        type = DailyWeeklyNotType.DAILY;
        dailyTime = new MinuteHour("05:00");
        weeklyWeekday = Weekday.SUNDAY;
        weeklyTime = new MinuteHour("19:00");
    }

    /** Gibt an, ob die tägliche Auswahl getroffen wurde. */
    public boolean isDaily() {
        return type == DailyWeeklyNotType.DAILY;
    }

    /** Gibt an, ob die wöchentliche Auswahl getroffen wurde. */
    public boolean isWeekly() {
        return type == DailyWeeklyNotType.WEEKLY;
    }

    /** Gibt an, ob die Auswahl gar nicht getroffen wurde. */
    public boolean isNot() {
        return type == DailyWeeklyNotType.NOT;
    }

    /** Getter für den Typ der Auswahl: DAILY, WEEKLY oder NOT. */
    public DailyWeeklyNotType getType() {
        return type;
    }

    /** Getter für die Uhrzeit für die tägliche Auswahl. */
    public MinuteHour getDailyTime() {
        return dailyTime;
    }

    /** Getter für den Wochentag für die wöchentliche Auswahl. */
    public Weekday getWeeklyWeekday() {
        return weeklyWeekday;
    }

    /** Getter für den Wochentag für die wöchentliche Auswahl als deutscher Name des Wochentags. */
    public String getWeeklyWeekdayAsGermanName() {
        return getWeeklyWeekday().getGermanName();
    }

    /** Getter für die Uhrzeit für die wöchentliche Auswahl. */
    public MinuteHour getWeeklyTime() {
        return weeklyTime;
    }

    /** Legt fest, dass die tägliche Auswahl getroffen wurde. */
    public void setDaily() {
        type = DailyWeeklyNotType.DAILY;
    }

    /** Legt fest, dass die wöchentliche Auswahl getroffen wurde. */
    public void setWeekly() {
        type = DailyWeeklyNotType.WEEKLY;
    }

    /** Legt fest, dass die Auswahl gar nicht getroffen wurde. */
    public void setNot() {
        type = DailyWeeklyNotType.NOT;
    }

    /** Getter für den Typ der Auswahl: DAILY, WEEKLY oder NOT. */
    public void setType(DailyWeeklyNotType type) {
        this.type = type;
    }

    /** Setter für die Uhrzeit für die tägliche Auswahl. */
    public void setDailyTime(MinuteHour dailyTime) {
        this.dailyTime = dailyTime;
    }

    /** Getter für den Wochentag für die wöchentliche Auswahl. */
    public void setWeeklyWeekday(Weekday weeklyWeekday) {
        this.weeklyWeekday = weeklyWeekday;
    }

    /** Setter für den Wochentag für die wöchentliche Auswahl als deutscher Name des Wochentags. */
    public void setWeeklyWeekdayAsGermanName(String weeklyWeekdayAsGermanName) {
        this.weeklyWeekday = Weekday.determineByGermanName(weeklyWeekdayAsGermanName);
    }

    /** Getter für die Uhrzeit für die wöchentliche Auswahl. */
    public void setWeeklyTime(MinuteHour weeklyTime) {
        this.weeklyTime = weeklyTime;
    }

    /** Erzeugt eine Beschreibung der Einstellungen in einer Zeile. */
    public String getOneLineDescription() {
        switch (type) {
            case DAILY:
                return "täglich um " + dailyTime + " Uhr";
            case WEEKLY:
                return "wöchentlich am " + weeklyWeekday.getGermanName() + " um " + weeklyTime
                        + " Uhr";
            case NOT:
                return "keine";
            default:
                throw new RuntimeException("unbekannter Typ " + type);
        }
    }

    @Override
    public String toString() {
        return "DailyWeeklyNot [type=" + type + ", dailyTime=" + dailyTime + ", weeklyWeekday="
                + weeklyWeekday + ", weeklyTime=" + weeklyTime + "]";
    }

}
