package de.duehl.swing.ui.components.elements;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;

import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;

/**
 * Diese Klasse stellt ein Label mit einem Titel links davor dar.
 *
 * @version 1.01     2017-04-18
 * @author Christian Dühl
 */

public class TitledLabel {

    private final JLabel titleLabel;
    private final JLabel valueLabel;
    private final Component panel;
    private final Dimension initialTitleDimension;
    private final Color initialValueColor;
    private final boolean setTitelDimension;
    private final boolean setValueColor;

    /** Konstruktor ohne spezielle Breite des Titels und ohne den Titel einzufärben. */
    public TitledLabel() {
        this("", "");
    }

    /**
     * Konstruktor ohne spezielle Breite des Titels und ohne den Titel einzufärben, Titel und Wert
     * werden schon direkt angegeben.
     */
    public TitledLabel(String title, String value) {
        initialTitleDimension = new Dimension(0, 0); // egal
        initialValueColor = Color.BLACK; // egal
        setTitelDimension = false;
        setValueColor = false;

        titleLabel = new JLabel(title);
        valueLabel = new JLabel(value);
        panel = createTitledValue();
    }

    /** Konstruktor mit spezieller Breite des Titels und mit Farbe für den Titel. */
    public TitledLabel(Dimension titleDimension, Color valueColor) {
        this(titleDimension, valueColor, "", "");
    }

    /**
     * Konstruktor mit spezieller Breite des Titels und mit Farbe für den Titel, Titel und Wert
     * werden schon direkt angegeben.
     */
    public TitledLabel(Dimension titleDimension, Color valueColor, String title, String value) {
        this.initialTitleDimension = titleDimension;
        this.initialValueColor = valueColor;
        setTitelDimension = true;
        setValueColor = true;

        titleLabel = new JLabel(title);
        valueLabel = new JLabel(value);
        panel = createTitledValue();
    }

    private Component createTitledValue() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createTitleLabel(), BorderLayout.WEST);
        panel.add(createValueLabel(), BorderLayout.CENTER);

        return panel;
    }

    private Component createTitleLabel() {
        if (setTitelDimension) {
            titleLabel.setPreferredSize(initialTitleDimension);
        }
        return titleLabel;
    }

    private JLabel createValueLabel() {
        if (setValueColor) {
            valueLabel.setForeground(initialValueColor);
        }
        return valueLabel;
    }

    /** Getter für den Panel, der diese Komponente enthält. */
    public Component getPanel() {
        return panel;
    }

    /** Ändert den Text des Titels. */
    public void setTitle(String title) {
        titleLabel.setText(title);
        panel.repaint();
    }

    /** Ändert den Text des Wertes. */
    public void setValue(String value) {
        valueLabel.setText(value);
        panel.repaint();
    }

    /** Verwendet für den Wert einen Monospaced-Font. */
    public void useMonospacedFontForTheValue() {
        GuiTools.setMonospacedFont(valueLabel);
        panel.repaint();
    }

    /** Setzt die bevorzugte Größe dieses Elements. */
    public void setPreferredSize(Dimension preferredSize) {
        panel.setPreferredSize(preferredSize);
    }

    /** Setzt die bevorzugte Größe des Titels dieses Elements. */
    public void setPreferredTitleSize(Dimension preferredSize) {
        titleLabel.setPreferredSize(preferredSize);
    }

    /** Ändert die Farbe des Wertes. */
    public void setValueColor(Color color) {
        valueLabel.setForeground(color);
    }

}
