package de.duehl.swing.ui.dialogs.base;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;
import java.awt.Dialog.ModalityType;

import javax.swing.JDialog;
import javax.swing.JMenuBar;
import javax.swing.JRootPane;

import de.duehl.swing.logic.Quitter;
import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse ist die abstrakte Basis der Dialoge mit modalem JDialog.
 *
 * @version 1.01     2024-11-20
 * @author Christian Dühl
 */

public abstract class ModalDialogBase extends AbstractDialogBase  {

    /** Der Dialog, der dargestellt wird. */
    private final JDialog dialog;

    /**
     * Konstruktor für ein gepacktes Fenster ohne fest vorgegebene Dimension.
     *
     * @param dialogTitle
     *            Der Titel des Dialoges.
     */
    public ModalDialogBase(String dialogTitle) {
        this(new Point(0, 0), null, dialogTitle, IGNORE_DIMENSION, null);
        ignoreSize();
    }

    /**
     * Konstruktor.
     *
     * @param dialogTitle
     *            Der Titel des Dialoges.
     * @param dialogDimension
     *            Die Größe des Dialogs.
     */
    public ModalDialogBase(String dialogTitle, Dimension dialogDimension) {
        this(new Point(0, 0), null, dialogTitle, dialogDimension, null);
    }

    /**
     * Konstruktor für ein gepacktes Fenster ohne fest vorgegebene Dimension.
     *
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das Icon für das Programm.
     * @param dialogTitle
     *            Der Titel des Dialoges.
     */
    public ModalDialogBase(Point parentLocation, Image programImage, String dialogTitle) {
        this(parentLocation, programImage, dialogTitle, IGNORE_DIMENSION, null);
        ignoreSize();
    }

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das Icon für das Programm.
     * @param dialogTitle
     *            Der Titel des Dialoges.
     * @param dialogDimension
     *            Die Größe des Dialogs.
     */
    public ModalDialogBase(Point parentLocation, Image programImage, String dialogTitle,
            Dimension dialogDimension) {
        this(parentLocation, programImage, dialogTitle, dialogDimension, null);
    }

    /**
     * Konstruktor für ein gepacktes Fenster ohne fest vorgegebene Dimension.
     *
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das Icon für das Programm.
     * @param dialogTitle
     *            Der Titel des Dialoges.
     * @param colorizer
     *            Die Farbverwaltung für die Gui.
     */
    public ModalDialogBase(Point parentLocation, Image programImage, String dialogTitle,
            Colorizer colorizer) {
        this(parentLocation, programImage, dialogTitle, IGNORE_DIMENSION, colorizer);
        ignoreSize();
    }

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das Icon für das Programm.
     * @param dialogTitle
     *            Der Titel des Dialoges.
     * @param dialogDimension
     *            Die Größe des Dialogs.
     * @param colorizer
     *            Die Farbverwaltung für die Gui.
     */
    public ModalDialogBase(Point parentLocation, Image programImage, String dialogTitle,
            Dimension dialogDimension, Colorizer colorizer) {
        super(parentLocation, programImage, dialogDimension, colorizer);

        dialog = new JDialog();
        dialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        dialog.setModalityType(ModalityType.APPLICATION_MODAL);
        dialog.setTitle(dialogTitle);
        setWindow(dialog);
    }

    /** Zeigt oder versteckt die Fenster-Dekorationen (oberste Zeile). */
    public void setUndecorated(boolean undecorated) {
        dialog.setUndecorated(undecorated);
    }

    /** Setzt den Titel des Rahmens neu. */
    protected final void changeTitle(String newTitle) {
        dialog.setTitle(newTitle);
        refresh();
    }

    /** Getter für den Titel des Dialogs. */
    public String getTitle() {
        return dialog.getTitle();
    }

    /** Setter für den Titel des Dialogs. */
    public void setTitle(String title) {
        changeTitle(title);
    }

    /** Gibt die RootPane des Dialogs zurück. */
    @Override
    public final JRootPane getRootPane() {
        return dialog.getRootPane();
    }

    /** Fügt einen Listener zum Beenden des Dialogs über das [x] oben rechts hinzu. */
    @Override
    public void addClosingWindowListener(Quitter quitter) {
        super.addClosingWindowListener(quitter);
        dialog.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
    }

    /** Getter für den Dialog als JDialog-Objekt. */
    public JDialog getRealDialog() {
        return dialog;
    }

    /** Trägt das Menü ein. */
    public void addMenu(JMenuBar menuBar) {
        dialog.setJMenuBar(menuBar);
        dialog.validate();
    }

    public void setLocationRelativeToNothing() {
        dialog.setLocationRelativeTo(null);
    }

}
