package de.duehl.swing.ui.elements;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JTextField;

/**
 * Dieses Klasse erweitert ein JTextField so, dass es seine Hintergrundfarbe
 * abhängig davon wechselt, ob Inhalt dargestellt wird oder nicht..
 *
 * @version 1.01     2015-02-02
 * @author Christian Dühl
 */

public class AutoBackgroundTextField extends JTextField {

    private static final long serialVersionUID = -4329507767646491689L;

    /** Hintergrundfarbe für gefüllten Zustand. */
    private final Color filledBackgroundColor;

    /** Hintergrundfarbe für leeren Zustand. */
    private final Color emptyBackgroundColor;

    /**
     * Konstruktor.
     *
     * @param columns
     *            the number of columns to use to calculate the preferred width;
     *            if columns is set to zero, the preferred width will be
     *            whatever naturally results from the component implementation
     * @param filledBackgroundColor
     *            Hintergrundfarbe für gefüllten Zustand.
     * @param emptyBackgroundColor
     *            Hintergrundfarbe für leeren Zustand.
     * @param addEditColorCheck
     *            Gibt an, ob die Überprüfung bei Bearbeitung für die passende
     *            Hintergrundfarbe hinzugefügt werden soll.
     */
    public AutoBackgroundTextField(final int columns,
            final Color filledBackgroundColor,
            final Color emptyBackgroundColor,
            final boolean addEditColorCheck) {
        super(columns);
        this.filledBackgroundColor = filledBackgroundColor;
        this.emptyBackgroundColor = emptyBackgroundColor;
        setBackground(emptyBackgroundColor);
        if (addEditColorCheck) {
            addEditColorCheck();
        }
    }

    /**
     * Setzt den anzuzeigenden Text und wählt abhängig davon, ob dieser gefüllt
     * ist, die Hintergrundfarbe des Textfeldes.
     *
     * @param text
     *            Anzuzeigender Text.
     */
    @Override
    public void setText(final String text) {
        super.setText(text);
        adjustBackgroundColor();
    }

    /**
     * Wählt abhängig davon, ob der Text gefüllt ist, die Hintergrundfarbe des
     * Textfeldes.
     */
    private void adjustBackgroundColor() {
        if (getText().isEmpty()) {
            setBackground(emptyBackgroundColor);
        }
        else {
            setBackground(filledBackgroundColor);
        }
    }

    /** Fügt Listener hinzu, die für die richtige Hintergrundfarbe sorgen. */
    private void addEditColorCheck() {
        addFocusListener(new FocusListener() {
            @Override
            public void focusLost(FocusEvent e) {
                adjustBackgroundColor();
            }
            @Override
            public void focusGained(FocusEvent e) {
                //adjustBackgroundColor();
                if (isEditable()) {
                    setBackground(filledBackgroundColor);
                }
            }
        });
        addKeyListener(new KeyListener() {
            @Override
            public void keyPressed(KeyEvent e) {
                //adjustBackgroundColor();
                if (isEditable()) {
                    setBackground(filledBackgroundColor);
                }
            }
            @Override
            public void keyTyped(KeyEvent e) {
                //adjustBackgroundColor();
            }
            @Override
            public void keyReleased(KeyEvent e) {
                //adjustBackgroundColor();
            }
        });
    }

}
