package de.duehl.swing.ui.filter.method.combination;

import de.duehl.swing.ui.filter.exceptions.FilterException;
import de.duehl.swing.ui.filter.method.FilterMethodFabricable;
import de.duehl.swing.ui.filter.method.Method;

/**
 * Diese Klasse stellt ein Element einer Filtermethodenkombination dar.
 *
 * @version 1.01     2020-11-04
 * @author Christian Dühl
 */

public class CombinationElementFabric<Data, Type> {

    /** Fabrik für die Filtermethoden. */
    private FilterMethodFabricable<Data, Type> methodFabric;

    /**
     * Konstruktor
     *
     * @param methodFabric
     *            Fabrik die konkrete Filtermethoden erzeugt.
     */
    public CombinationElementFabric(FilterMethodFabricable<Data, Type> methodFabric) {
        this.methodFabric = methodFabric;
    }

    /**
     * Erzeugt eine öffnende Klammer.
     *
     * @throws FilterException
     *             Bei Fehlern wird diese Ausnahme geworfen.
     */
    public CombinationElement<Data> createOpeningBrace() {
        return new CombinationElement<>(CombinationElementType.BRACE_OPEN);
    }

    /**
     * Erzeugt eine schließende Klammer.
     *
     * @throws FilterException
     *             Bei Fehlern wird diese Ausnahme geworfen.
     */
    public CombinationElement<Data> createClosingBrace() {
        return new CombinationElement<>(CombinationElementType.BRACE_CLOSE);
    }

    /**
     * Erzeugt eine Negation.
     *
     * @throws FilterException
     *             Bei Fehlern wird diese Ausnahme geworfen.
     */
    public CombinationElement<Data> createNegation() {
        return new CombinationElement<>(CombinationElementType.NOT);
    }

    /**
     * Erzeugt eine Schnittmengenoperation.
     *
     * @throws FilterException
     *             Bei Fehlern wird diese Ausnahme geworfen.
     */
    public CombinationElement<Data> createIntersection() {
        return new CombinationElement<>(CombinationElementType.INTERSECTION);
    }

    /**
     * Erzeugt eine Vereinigungsoperation.
     *
     * @throws FilterException
     *             Bei Fehlern wird diese Ausnahme geworfen.
     */
    public CombinationElement<Data> createUnion() {
        return new CombinationElement<>(CombinationElementType.UNION);
    }

    /**
     * Erzeugt eine Filtermethode.
     *
     * @param type
     *            Typ des Filters.
     * @throws FilterException
     *             Bei Fehlern wird diese Ausnahme geworfen.
     */
    public CombinationElement<Data> createMethod(Type type) {
        Method<Data> method = methodFabric.buildMethod(type);
        return new CombinationElement<Data>(method);
    }

    /**
     * Erzeugt eine Filtermethode mit Suchtext.
     *
     * @param type
     *            Typ des Filters.
     * @param searchForMe
     *            Suchtext
     * @throws FilterException
     *             Bei Fehlern wird diese Ausnahme geworfen.
     */
    public CombinationElement<Data> createInputMethod(Type type, String searchForMe) {
        Method<Data> method = methodFabric.buildInputMethod(type, searchForMe);
        return new CombinationElement<>(method);
    }

    /* createCombinedMethod wäre eine Meta-Verknüpfung... */

}