package de.duehl.swing.ui.text;

import java.awt.BorderLayout;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Image;
import java.awt.Point;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse stellt den Dialog zum Bearbeiten eines Textes im Editor dar.
 *
 * Beim Druck auf bearbeiten wird der Text bearbeitet und wieder angezeigt.
 *
 * @version 1.01     2020-02-12
 * @author Christian Dühl
 */

public class WorkOnTextEditor extends TextEditor {

    /**
     * Konstruktor.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     */
    public WorkOnTextEditor(String dialogTitle, EditorOkReactor reaktor) {
        this(dialogTitle, new Point(0, 0), null, reaktor);
    }

    /**
     * Konstruktor als Viewer.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param parentLocation
     *            Position des Rahmens des Aufrufers.
     */
    public WorkOnTextEditor(String dialogTitle, Point parentLocation, EditorOkReactor reaktor) {
        this(dialogTitle, parentLocation, null, reaktor);
    }

    /**
     * Konstruktor als Editor.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param parentLocation
     *            Position des Rahmens des Aufrufers.
     * @param programImage
     *            Bild des Programms.
     * @param contents
     *            Inhalt der angezeigt werden soll.
     * @param reaktor
     *            Objekt, das informiert wird, wenn der OK-Button gedrückt wird.
     */
    public WorkOnTextEditor(String dialogTitle, Point parentLocation, Image programImage,
            EditorOkReactor reaktor) {
        this(dialogTitle, parentLocation, null, programImage, reaktor);
    }

    /**
     * Konstruktor.
     *
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param parentLocation
     *            Position des Rahmens des Aufrufers.
     * @param colorizer
     *            Farbverwaltung für die Gui.
     * @param programImage
     *            Bild des Programms.
     * @param reaktor
     *            Objekt, das informiert wird, wenn der OK-Button gedrückt wird.
     */
    public WorkOnTextEditor(String dialogTitle, Point parentLocation, Colorizer colorizer,
            Image programImage, EditorOkReactor reaktor) {
        super(dialogTitle, parentLocation, colorizer, programImage, reaktor);
        rearrangeButtonPanel();
    }

    @Override
    protected void initTextComponent() {
        super.initTextComponent();
        switchToEditable();
    }

    @Override
    protected void okPressed() {
        String contents = getText();

        startLongTimeProcess("Verarbeitung läuft");
        new Thread(() -> workOnInputInThread(contents)).start();
    }

    private void workOnInputInThread(String contents) {
        String correctedContents = getReactor().workOnText(contents);
        SwingUtilities.invokeLater(() -> showTextInEdt(correctedContents));
    }

    private void showTextInEdt(String correctedContents) {
        setText(correctedContents);
        validate();
        endLongTimeProcess();
    }

    private void rearrangeButtonPanel() {
        JPanel panel = getButtonPanel();

        panel.add(getQuitButton(), BorderLayout.EAST);
        panel.add(getOkButton(), BorderLayout.CENTER);
    }

}
