package de.duehl.swing.ui.start.search;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;

import java.awt.Dimension;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import de.duehl.basics.text.Text;
import de.duehl.swing.ui.elements.search.SearchPanel;
import de.duehl.swing.ui.elements.search.SearchReaktor;
import de.duehl.swing.ui.highlightingeditor.HighlightingEditor;
import de.duehl.swing.ui.highlightingeditor.HighlightingEditorWithoutButtonBar;

/**
 * Diese Klasse testet das EnterTextField-Element.
 *
 * @version 1.01     2017-12-15
 * @author Christian Dühl
 */

public class SearchPanelDemoWithEditor {

    private static final String NO_SEARCH_TEXT = "###_NO_SEARCH_WORD_ENTERED_###";

    private int lastSearchIndex;
    private String searchText = NO_SEARCH_TEXT;
    private final HighlightingEditor editor;

    /** Das zu testende Suchfeld. */
    private SearchPanel searchPanel;

    /** Konstruktor. */
    public SearchPanelDemoWithEditor() {
        searchPanel = new SearchPanel(createSearchReaktor());

        editor = new HighlightingEditorWithoutButtonBar();
        editor.setText(Text.loremImpsum());

        lastSearchIndex = 0;

        SwingUtilities.invokeLater(() -> createAndShowGUI());
    }

    private SearchReaktor createSearchReaktor() {
        return new SearchReaktor() {
            @Override
            public void search(String searchText) {
                searchInternal(searchText);
            }

            @Override
            public void searchAgain() {
                searchAgainInternal();
            }
        };
    }

    private void searchInternal(String searchText) {
        this.searchText = searchText;
        lastSearchIndex = 0; // später mit Caret-Position initialisieren
        searchAgainInternal();
    }

    private void searchAgainInternal() {
        String text = editor.getText();
        text = Text.lineBreaksToBackslashN(text);

        int index = text.indexOf(searchText, lastSearchIndex);
        if (index < 0) {
            index = text.indexOf(searchText, 0);
            System.out.println("Suche von vorn...");
        }
        if (index < 0) {
            System.out.println("Suchbegriff '" + searchText + "' nicht im Text gefunden.");
        }
        else {
            lastSearchIndex = index + searchText.length();
            System.out.println("Suchbegriff '" + searchText + "' an Position " + index + " bis "
                    + lastSearchIndex + " im Text gefunden.");
            editor.setCaretPosition(index);
            editor.select(index, lastSearchIndex);
        }
    }

    private void createAndShowGUI() {
        JFrame frame = new JFrame("SearchPanel Test");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setLayout(new BorderLayout());

        frame.add(searchPanel, BorderLayout.NORTH);
        frame.add(editor.getComponent(), BorderLayout.CENTER);

        frame.setPreferredSize(new Dimension(800, 600));
        frame.pack();
        frame.setVisible(true);
    }

    /** Startpunkt des Programms, Parameter werden ignoriert. */
    public static void main(String[] args) {
        new SearchPanelDemoWithEditor();
    }

}

