package de.duehl.vocabulary.japanese.common.persistence.data;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.Text;

/**
 * Diese Klasse hält eine Liste von zuletzt besuchten eigenen Listen in Form von Daten-Objekten,
 * die eine eigenen Liste beschreiben (bestehend aus Kategorie, Unterkategorie und dem Namen der
 * Liste) vor.
 *
 * @version 1.01     2025-07-04
 * @author Christian Dühl
 */

public class HistoricalOwnListPersistanceDataList {

    private static final int DEFAULT_NUMBER_OF_REMEBERED_OWN_LISTS = 10;

    /**
     * Die Anzahl der eigenen Listen, die in dieser Liste von besuchten eigenen Listen gespeichert
     * wird.
     */
    private int numberOfRememberedOwnLists;

    /**
     * Die Liste mit den zuletzt angesehenen eigenen Listen.
     *
     * Das letzte Element ist die neuste (zuletzt besuchte) eigene Liste.
     */
    private final List<OwnListPersistanceData> historicalOwnLists;

    /** Konstruktor. */
    public HistoricalOwnListPersistanceDataList() {
        this(DEFAULT_NUMBER_OF_REMEBERED_OWN_LISTS);
    }

    /**
     * Konstruktor.
     *
     * @param numberOfRememberedOwnLists
     *            Die Anzahl der eigenen Listen, die in dieser Liste von besuchten eigenen Listen
     *            gespeichert wird.
     */
    public HistoricalOwnListPersistanceDataList(int numberOfRememberedOwnLists) {
        this.numberOfRememberedOwnLists = numberOfRememberedOwnLists;
        historicalOwnLists = new ArrayList<>();
    }

    /** Fügt die Daten einer neu besuchten eigenen Liste hinzu. */
    public void addOwnListData(OwnListPersistanceData ownListData) {
        /*
         * Falls schon enthalten, wird der alte Eintrag gelöscht, bevor man es am Ende einfügt,
         * damit er ans Ende gestellt wird.
         */
        historicalOwnLists.remove(ownListData);

        historicalOwnLists.add(ownListData);
        deleteOldEntriesIfListIsTooLong();
    }

    private void deleteOldEntriesIfListIsTooLong() {
        int numberOfEntriesToDelete = historicalOwnLists.size() - numberOfRememberedOwnLists;

        if (numberOfEntriesToDelete > 0) {
            for (int count = 0; count < numberOfEntriesToDelete; ++count) {
                historicalOwnLists.remove(0);
            }
        }
    }

    /**
     * Setter für die Anzahl der eigenen Listen, die in dieser Liste von besuchten eigenen Listen
     * gespeichert wird.
     */
    public void setNumberOfRememberedOwnLists(int numberOfRememberedOwnLists) {
        this.numberOfRememberedOwnLists = numberOfRememberedOwnLists;
        deleteOldEntriesIfListIsTooLong();
    }

    /**
     * Getter für die Liste mit den zuletzt angesehenen eigenen Listen.
     *
     * Das letzte Element ist die neuste (zuletzt besuchte) eigene Liste.
     */
    public List<OwnListPersistanceData> getHistoricalOwnLists() {
        return historicalOwnLists;
    }

    /** Gibt an, ob die Liste der gespeicherten eigenen Listen leer ist. */
    public boolean isEmpty() {
        return historicalOwnLists.isEmpty();
    }

    private static final String LIST_PERISTANCE_SEPARATOR = "###;###"; // anders als in der Klasse
                                                                       // OwListPersistanceData!

    /** Erzeugt aus der Liste mit den  Daten einen String zum persistenten Abspeichern. */
    public String toPersistanceString() {
        StringBuilder builder = new StringBuilder();
        boolean first = true;
        for (OwnListPersistanceData ownList : historicalOwnLists) {
            if (first) {
                first = false;
            }
            else {
                builder.append(LIST_PERISTANCE_SEPARATOR);
            }
            builder.append(ownList.toPersistanceString());
        }
        return builder.toString();
    }

    /**
     * Erzeugt aus einem persistent abgespeicherten String wieder ein Daten-Listen-Objekt.
     *
     * @param persistanceSring
     *            Der persistent gespeicherte String.
     * @param numberOfRememberedOwnLists
     *            Die Anzahl der eigenen Listen, die in dieser Liste von besuchten eigenen Listen
     *            gespeichert wird.
     * @return Das erzeugte Objekt.
     */
    public static HistoricalOwnListPersistanceDataList createFromPersistanceString(
            String persistanceSring, int numberOfRememberedOwnLists) {
        HistoricalOwnListPersistanceDataList historicalList =
                new HistoricalOwnListPersistanceDataList(numberOfRememberedOwnLists);

        List<String> parts = Text.splitBy(persistanceSring, LIST_PERISTANCE_SEPARATOR);
        for (String part : parts) {
            OwnListPersistanceData data = OwnListPersistanceData.createFromPersistanceString(part);
            historicalList.addOwnListData(data);
        }

        /* Falls sich zwischendurch in den Einstellungen dieser Wert verkleinert hat: */
        historicalList.deleteOldEntriesIfListIsTooLong();

        return historicalList;
    }

}
