package de.duehl.vocabulary.japanese.grammar;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;

/**
 * Diese Klasse sucht zu einer Vokabel, welche ein Verb ist, die zugehörige Wörterbuchform.
 *
 * @version 1.01     2025-06-11
 * @author Christian Dühl
 */

public class VerbWoerterbuchformFinder {

    public static final Vocable NO_WOERTERBUCHFORM_FOUND = new Vocable();

    /**
     * Sucht zur übergebenen Vokabel die Wörterbuchform.
     *
     * Hier wird davon ausgegangen, dass diese Vokabel ein Verb ist.
     *
     * Außerdem werden nur Vokabeln mit Kanji gefunden, denn ansonsten kann man sie nicht mit
     * der Wörterbuchform vergleichen.
     *
     * @param vocable
     *            Die Vokabel zu der die Wörterbuchform gesucht wird.
     * @param vocabularies
     *            Alle bekannten Vokabularien.
     * @return Die gefundene Wörterbuchform oder NO_WOERTERBUCHFORM_FOUND, wenn diese nicht
     *         gefunden wurde.
     */
    public static Vocable determineWoerterbuchform(Vocable vocable, List<Vocabulary> vocabularies) {
        List<String> quotedTexts = Text.findTextsBetweenJapaneseQuotes(vocable.getComment());
        if (quotedTexts.isEmpty()) {
            return NO_WOERTERBUCHFORM_FOUND;
        }

        String kanji = vocable.getKanji();
        if (kanji.isEmpty()) {
            return NO_WOERTERBUCHFORM_FOUND;
        }

        /*
         * TODO:
         *
         * Hier macht man beim Vergleich über das erste Kanji einen Fehler, wenn verschiedene
         * Verben mit dem gleichen Kanji anfangen. Eigentlich müsste man vermutlich alle Kanji am
         * Anfang vergleichen?
         *
         * Da muss ich mir nochmal Beispiele anschauen, neulich war mir so eine Liste bei der
         * Verwendung des Programms aufgefallen.
         *
         * Obwohl eigentlich sind das doch nur die Texte in der Bemerkung ... Hmmm.
         */
        String firstKanji = kanji.substring(0, 1);
        List<String> potentialWoerterbuchformInKanjiTexts = new ArrayList<>();
        for (String quotedText : quotedTexts) {
            if (quotedText.startsWith(firstKanji)) {
                potentialWoerterbuchformInKanjiTexts.add(quotedText);
            }
        }


        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable anOtherVocable : vocabulary.getVocables()) {
                if (!anOtherVocable.equals(vocable) // eigentlich überflüssig, da das nur
                                                    // aufgerufen wird, wenn vocable KEINE
                                                    // Wörterbuchform ist.
                        && anOtherVocable.isVerbInWoerterbuchform()) {
                    String anOtherKanji = anOtherVocable.getKanji();
                    if (potentialWoerterbuchformInKanjiTexts.contains(anOtherKanji)) {
                        return anOtherVocable;
                    }
                }
            }
        }

        return NO_WOERTERBUCHFORM_FOUND;
    }

}
