package de.duehl.vocabulary.japanese.logic.ownlists.groups;

/**
 * Diese Enum-Klasse stellt die Gruppe dar, zu der eine Liste gehören kann.
 *
 * Dies ist nicht zu verwechseln mit der Kategorie und Unterkategorie, in der die Listen im
 * Programm angeordnet sind, hierbei geht es um die Gruppen von Listen, welche ich zum Download
 * anbiete.
 *
 * Für diese Gruppen kann sich der Benutzer interessieren (oder sie abwählen) und wird dann nur
 * informiert, wenn es nur in den Bereichen, die ihn interessieren, neuere Listen auf dem Server
 * gibt.
 *
 * Auch wird diese Gruppenbezeichnung dazu verwendet, damit sich das Programm merken kann, zu
 * welcher Gruppe zuletzt welche Version an bereitgestellten eigenen Listen heruntergeladen wurde.
 *
 * Ich kann natürlich nicht wissen, ob der Benutzer die heruntergeladenen Gruppen auch importiert
 * hat, aber das ist dann auch eher das Problem des Benutzers.
 *
 * @version 1.01     2025-10-01
 * @author Christian Dühl
 */

public enum OwnListGroup {

    FROM_ZERO_1("Japanese From Zero - Buch 1", "B1 "),
    FROM_ZERO_2("Japanese From Zero - Buch 2", "B2 "),
    FROM_ZERO_3("Japanese From Zero - Buch 3", "B3 "),
    FROM_ZERO_4("Japanese From Zero - Buch 4", "B4 "),

    SCHRITT_FUER_SCHRITT_1("Japanisch Schritt für Schritt Band 1", "BD1 "),

    VHS_A1_1("VHS Bielefeld Kurs A1.1 bei Hiroko Watanabe-Schmidt ab März 2024", "VHS A1.1 "),
    VHS_A1_2("VHS Bielefeld Kurs A1.2 bei Hiroko Watanabe-Schmidt ab September 2024", "VHS A1.2 "),
    VHS_A1_3("VHS Bielefeld Kurs A1.3 bei Hiroko Watanabe-Schmidt ab April 2025", "VHS A1.3 "),
    VHS_A2_1("VHS Bielefeld Kurs A2.1 bei Hiroko Watanabe-Schmidt ab September 2025", "VHS A2.1 "),

    MARUGOTO_A1("Marugoto A1 Rikai und Katsudou", "Marugoto A1 "),
    MARUGOTO_A2E1("Marugoto A2 Elementary 1 Rikai und Katsudou", "Marugoto A2E1 "),
    KUMON("Kumon Bücher für japanische Grundschüler", "Kumon "),

    EIGENE_SAMMLUNGEN("Eigene Sammlungen", "X "),
    ;

    /*
     * - Kinderbücher hinzufügen?
     * - Musikstücke hinzufügen?
     */

    /** Die Beschreibung der Gruppe. */
    private final String description;

    /** Der Anfang der Namen der Listen dieser Gruppe. */
    private final String nameStart;

    /**
     * Konstruktor.
     *
     * @param description
     *            Die Beschreibung der Gruppe.
     * @param nameStart
     *            Der Anfang der Namen der Listen dieser Gruppe.
     */
    private OwnListGroup(String description, String nameStart) {
        this.description = description;
        this.nameStart = nameStart;
    }

    /** Getter für die Beschreibung der Gruppe. */
    public String getDescription() {
        return description;
    }

    /** Getter für den Anfang der Namen der Listen dieser Gruppe. */
    public String getNameStart() {
        return nameStart;
    }

    /** Ermittelt das passende Objekt zum Gruppennamen. */
    public static OwnListGroup createByGroupName(String groupName) {
        for (OwnListGroup group : values()) {
            String name = group.name();
            if (name.equals(groupName)) {
                return group;
            }
        }

        throw new RuntimeException(
                "Zum Gruppennamen '" + groupName + "' konnte keine Gruppe ermittelt werden.");
    }

    /**
     * Erzeugt zu der Gruppe und der übergebenen Version den Dateinamen ohne Pfad, wie er temporär
     * local und remote auf dem Webserver verwendet wird.
     */
    public String createGroupBareFilename(int version) {
        return createGroupBareFilename(name(), version);
    }

    /**
     * Erzeugt zu der Gruppe und der übergebenen Version den Dateinamen ohne Pfad und Extension,
     * wie er temporär local und remote auf dem Webserver verwendet wird.
     */
    public String createGroupBareFilenameWithoutExtension(int version) {
        return createGroupBareFilenameWithoutExtension(name(), version);
    }

    /**
     * Erzeugt zu einer Gruppe und der Version den Dateinamen ohne Pfad, wie er temporär local und
     * remote auf dem Webserver verwendet wird.
     */
    public static String createGroupBareFilename(String groupName, int version) {
        return createGroupBareFilenameWithoutExtension(groupName, version) + ".zip";
    }

    /**
     * Erzeugt zu einer Gruppe und der Version den Dateinamen ohne Pfad und Extension, wie er
     * temporär local und remote auf dem Webserver verwendet wird.
     */
    public static String createGroupBareFilenameWithoutExtension(String groupName, int version) {
        return "Gruppe_" + groupName + "_" + version;
    }

}
