package de.duehl.vocabulary.japanese.logic.symbol.kana.test;

import de.duehl.vocabulary.japanese.data.symbol.Hiragana;
import de.duehl.vocabulary.japanese.logic.symbol.kana.test.data.SingleUserInputHiraganaCheckResult;

/**
 * Diese Klasse bewertet die Benutzereingaben zu einem Hiragana.
 *
 * @version 1.01     2025-02-02
 * @author Christian Dühl
 */

public class HiraganaTestChecker {

    /** Das Hiragana welches abgefragt wurde. */
    private final Hiragana hiragana;

    /** Die vom Benutzer eingegebene Hepburn-Darstellung. */
    private final String hepburn;

    /** Die Beschreibung der Fehler, fall welche gemacht wurden. */
    private String errorDescription;

    /** Gibt an, ob alles richtig ist. */
    private boolean ok;

    /** Das Ergebnis der Prüfung. */
    private SingleUserInputHiraganaCheckResult result;

    /**
     * Konstruktor.
     *
     * @param hiragana
     *            Das Hiragana welches abgefragt wurde.
     * @param hepburn
     *            Die vom Benutzer eingegebene Hepburn-Darstellung.
     */
    public HiraganaTestChecker(Hiragana hiragana, String hepburn) {
        this.hiragana = hiragana;
        this.hepburn = hepburn;
    }

    /** Führt die Prüfung durch. */
    public void check() {
        init();
        checkHepburn();
        createCheckResult();
    }

    private void init() {
        ok = true;
        errorDescription = "";
    }

    /**
     * Überprüft die deutsche Bedeutung.
     *
     * Da es bei der deutschen Bedeutung durchaus Strings gibt, die Leerzeichen haben, kann ich
     * nicht einfach an Leerzeichen oder Kommata trennen. Beispiele:
     *     - "... Zeit"
     *     - "halb nach"
     * Daher trenne ich hier die korrekten deutschen Bedeutungen an Kommata und suche diese
     * Teile der Länge nach in der Eingabe des Benutzers.
     *
     * Am Ende werfe ich die Listen in die normale Überprüfung.
     */
    private void checkHepburn() {
        String correctHepburn = hiragana.getHepburn();

        if (hepburn.equals(correctHepburn)) {
            ok = true;
        }
        else {
            ok = false;
            addToErrorDescription("Hepburn: falsche Eingabe.\n");
        }
    }

    private void addToErrorDescription(String toAppend) {
        errorDescription = addToText(errorDescription, toAppend);
    }

    private String addToText(String text, String toAppend) {
        if (text.isBlank()) {
            return toAppend;
        }
        else {
            return text + toAppend;
        }
    }

    private void createCheckResult() {
        result = new SingleUserInputHiraganaCheckResult(hiragana);
        result.setOk(ok);
        result.setErrorDescription(errorDescription);
    }

    /** Getter für das Ergebnis der Prüfung. */
    public SingleUserInputHiraganaCheckResult getResult() {
        return result;
    }

}
