package de.duehl.vocabulary.japanese.ui.dialog.grammar.headers.io;

import java.util.List;
import java.util.Map;

import de.duehl.basics.io.Charset;
import de.duehl.basics.io.FileHelper;
import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.persistence.GrammarTableHeaderPersistenceHelper;

/**
 * Diese Klasse lädt aus dem Dialoge zur Bearbeitung der Überschriften für die Erstellung von
 * grammatikalischen Tabellen von Verben oder Adjektiven die vom Benutzer eingegebenen
 * Überschriften aus einer Datei, damit zwischen verschiedenen Versionen dieser Überschriften
 * gewechselt werden kann.
 *
 * @version 1.01     2025-11-29
 * @author Christian Dühl
 */

public class GrammarTableOwnHeadersLoader {

    /** Der Dateiname. */
    private final String filename;

    /**
     * Die Liste der eigenen Formulierungen, die anstelle der Begriffe aus
     * VerbSearchWords.SORTED_VERB_SEARCH_WORDS im Kopf der erzeugten Tabelle bei der Erstellung
     * von grammatikalischen Tabellen von Verben oder Adjektiven leer angezeigt werden sollen.
     *
     * Diese liegt vor in einer Form eines Verzeichnis, welches meiner Formulierung aus der Klasse
     * VerbSearchWords die Formulierung des Benutzers zuordnet. Aufgenommen werden nur solche, die
     * der Benutzer gegenüber meiner Formulierung geändert hat.
     */
    private Map<String, String> verbHeadersMap;

    /**
     * Die Liste der eigenen Formulierungen, die anstelle der Begriffe aus
     * AdjectiveSearchWords.SORTED_I_ADJEKTIVE_SEARCH_WORDS im Kopf der erzeugten Tabelle bei der
     * Erstellung von grammatikalischen Tabellen von Verben oder Adjektiven leer angezeigt werden
     * sollen.
     *
     * Diese liegt vor in einer Form eines Verzeichnis, welches meiner Formulierung aus der Klasse
     * AdjectiveSearchWords die Formulierung des Benutzers zuordnet. Aufgenommen werden nur solche,
     * die der Benutzer gegenüber meiner Formulierung geändert hat.
     */
    private Map<String, String> iAdjectiveHeadersMap;

    /**
     * Die Liste der eigenen Formulierungen, die anstelle der Begriffe aus
     * AdjectiveSearchWords.SORTED_NA_ADJEKTIVE_SEARCH_WORDS im Kopf der erzeugten Tabelle bei der
     * Erstellung von grammatikalischen Tabellen von Verben oder Adjektiven leer angezeigt werden
     * sollen.
     *
     * Diese liegt vor in einer Form eines Verzeichnis, welches meiner Formulierung aus der Klasse
     * AdjectiveSearchWords die Formulierung des Benutzers zuordnet. Aufgenommen werden nur solche,
     * die der Benutzer gegenüber meiner Formulierung geändert hat.
     */
    private Map<String, String> naAdjectiveHeadersMap;

    /**
     * Konstruktor.
     *
     * @param filename
     *            Der Dateiname.
     */
    public GrammarTableOwnHeadersLoader(String filename) {
        this.filename = filename;
    }

    /** Lädt die Datei. */
    public void load() {
        List<String> lines = FileHelper.readFileToList(filename, Charset.UTF_8);
        if (lines.size() != 3) {
            GuiTools.informUser("Die Datei kann nicht gelesen werden", ""
                    + "Die Datei " + filename + " kann nicht gelesen werden:\n\n"
                    + "Ewartet wurden 3 Zeilen, enthalten sind aber " + lines.size() + " Zeilen.");
            return;
        }

        int index = 0;
        String verbLine = lines.get(index++);
        String iAdjectiveLine = lines.get(index++);
        String naAdjectiveLine = lines.get(index++);
        verbHeadersMap = GrammarTableHeaderPersistenceHelper
                .createGrammarTableHeaderMapFormPersistenceString(verbLine);
        iAdjectiveHeadersMap = GrammarTableHeaderPersistenceHelper
                .createGrammarTableHeaderMapFormPersistenceString(iAdjectiveLine);
        naAdjectiveHeadersMap = GrammarTableHeaderPersistenceHelper
                .createGrammarTableHeaderMapFormPersistenceString(naAdjectiveLine);
    }

    /**
     * Getter für die Liste der eigenen Formulierungen, die anstelle der Begriffe aus
     * VerbSearchWords.SORTED_VERB_SEARCH_WORDS im Kopf der erzeugten Tabelle bei der Erstellung
     * von grammatikalischen Tabellen von Verben oder Adjektiven leer angezeigt werden sollen.
     *
     * Diese liegt vor in einer Form eines Verzeichnis, welches meiner Formulierung aus der Klasse
     * VerbSearchWords die Formulierung des Benutzers zuordnet. Aufgenommen werden nur solche, die
     * der Benutzer gegenüber meiner Formulierung geändert hat.
     */
    public Map<String, String> getVerbHeadersMap() {
        return verbHeadersMap;
    }

    /**
     * Getter für die Liste der eigenen Formulierungen, die anstelle der Begriffe aus
     * AdjectiveSearchWords.SORTED_I_ADJEKTIVE_SEARCH_WORDS im Kopf der erzeugten Tabelle bei der
     * Erstellung von grammatikalischen Tabellen von Verben oder Adjektiven leer angezeigt werden
     * sollen.
     *
     * Diese liegt vor in einer Form eines Verzeichnis, welches meiner Formulierung aus der Klasse
     * AdjectiveSearchWords die Formulierung des Benutzers zuordnet. Aufgenommen werden nur solche,
     * die der Benutzer gegenüber meiner Formulierung geändert hat.
     */
    public Map<String, String> getIAdjectiveHeadersMap() {
        return iAdjectiveHeadersMap;
    }

    /**
     * Getter für die Liste der eigenen Formulierungen, die anstelle der Begriffe aus
     * AdjectiveSearchWords.SORTED_NA_ADJEKTIVE_SEARCH_WORDS im Kopf der erzeugten Tabelle bei der
     * Erstellung von grammatikalischen Tabellen von Verben oder Adjektiven leer angezeigt werden
     * sollen.
     *
     * Diese liegt vor in einer Form eines Verzeichnis, welches meiner Formulierung aus der Klasse
     * AdjectiveSearchWords die Formulierung des Benutzers zuordnet. Aufgenommen werden nur solche,
     * die der Benutzer gegenüber meiner Formulierung geändert hat.
     */
    public Map<String, String> getNaAdjectiveHeadersMap() {
        return naAdjectiveHeadersMap;
    }

}
