package de.duehl.vocabulary.japanese.ui.dialog.kana.hiraganatest;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import de.duehl.basics.text.NumberString;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.ui.tools.VocabularyTrainerUiTools;

/**
 * Diese Klasse zeigt den Erfolg oder Misserfolg nach dem Übersetzen aller getesteten Hiragana
 * an.
 *
 * TODO man könnte fast EINE Klasse dafür benutzen, der man "Hiragana", "beantwortet" bzw.
 * "Vokabeln", "übersetzt" mitgibt. Beobachten, ob es ansonsten wirklich gleich bleibt.
 * Vgl. AllTranslationsEvaluationDialog
 *
 * @version 1.01     2025-02-02
 * @author Christian Dühl
 */

public class AllHiraganaTestEvaluationDialog extends ModalDialogBase {

    /** Die anzuzeigende Nachricht. */
    private final String message;

    /** Die Hintergrundfarbe. */
    private final Color backgroundColor;

    /** Der Button zum Beenden des Dialogs. */
    private final JButton quitButton;

    /** Der Titel mit Angabe der Prozentzahlen des Dialogs zur Abfrage der Hiragana. */
    private final String originalDialogTitle;

    /**
     * Die Prozentzahl des Erfolgs der letzten zehn Abfragen dieser Hiragana bevor der aktuelle
     * Test gelaufen war.
     */
    private final double lastTenTestsPercentBefore;

    /**
     * Die Prozentzahl des Erfolgs der letzten zehn Abfragen dieser Hiragana nachdem der aktuelle
     * Test gelaufen ist.
     */
    private final double lastTenTestsPercentAfter;

    /** Die Prozentzahl des Erfolgs der gerade durchgeführten Abfragen dieser Hiragana. */
    private final double actualPercent;

    /**
     * Die Prozentzahl mit zwei Nachkommastellen des Erfolgs der gerade durchgeführten Abfragen
     * dieser Hiragana.
     */
    private String actualPercentTextWithTwoDigitsAfterComma;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     * @param total
     *            Die Anzahl der getesteten Hiragana.
     * @param correct
     *            Die Anzahl der richtig getesteten Hiragana.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Anzuzeigendes ProgrammIcon.
     * @param lastTenTestsPercentAfter
     *            Die Prozentzahl des Erfolgs der letzten zehn Abfragen dieser Hiragana nachdem der
     *            aktuelle Test gelaufen ist.
     * @param lastTenTestsPercentBefore
     *            Die Prozentzahl des Erfolgs der letzten zehn Abfragen dieser Hiragana bevor der
     *            aktuelle Test gelaufen war.
     * @param originalDialogTitle
     *            Der Titel mit Angabe der Prozentzahlen des Dialogs zur Abfrage der Hiragana.
     */
    public AllHiraganaTestEvaluationDialog(Options options, int total, int correct,
            Point parentLocation, Image programImage, double lastTenTestsPercentBefore,
            double lastTenTestsPercentAfter, String originalDialogTitle) {
        super(parentLocation, programImage, "Auswertung der Übersetzungen");
        this.lastTenTestsPercentBefore = lastTenTestsPercentBefore;
        this.lastTenTestsPercentAfter = lastTenTestsPercentAfter;
        this.originalDialogTitle = originalDialogTitle;

        addEscapeBehaviour();

        message = VocabularyTrainerUiTools.createAllTestsMessage("Hiragana", "beantwortet", total,
                correct);

        VocableColors vocableColors = new VocableColors(options);
        backgroundColor = vocableColors.determineBackgroundColor(total, correct);
        actualPercent = NumberString.percentAsNumber(correct, total);
        actualPercentTextWithTwoDigitsAfterComma = NumberString.percent(correct, total);

        quitButton = new JButton("Beenden");

        fillDialog();
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        initElements();

        add(createCenterPart(), BorderLayout.CENTER);
        add(createButtonsPart(),  BorderLayout.SOUTH);

        pack();

        SwingUtilities.invokeLater(() -> quitButton.requestFocus());
    }

    private void initElements() {
        GuiTools.addReturnListener(quitButton, () -> quitButton.doClick());
    }

    private Component createCenterPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(createMessagePart());
        panel.add(createTestedVocablesInfoPart());

        return GuiTools.createScrollPane(panel);
    }

    private Component createMessagePart() {
        JLabel label = new JLabel("<html>" + message + "</html>");
        GuiTools.biggerFont(label, 5);
        label.setOpaque(true);
        label.setBackground(backgroundColor);
        return label;
    }

    private Component createTestedVocablesInfoPart() {
        String testedVocables = "Abgefragte Vokabeln: " + originalDialogTitle;
        String actualSuccessPercent = "Erfolg der aktuellen Abfrage in Prozent: "
                + actualPercentTextWithTwoDigitsAfterComma;
        String lastTenSuccessBefore = "Erfolg der letzten Zehn Abfragen VOR der aktuellen Abfrage "
                + "in Prozent: " + NumberString.twoDecimalPlaces(lastTenTestsPercentBefore);
        String lastTenSuccessAfter = "Erfolg der letzten Zehn Abfragen NACH der aktuellen Abfrage "
                + "in Prozent: " + NumberString.twoDecimalPlaces(lastTenTestsPercentAfter);

        String upOrDown;
        if (actualPercent > lastTenTestsPercentBefore) {
            upOrDown = "Sie haben sich verbessert.";
        }
        else if (actualPercent < lastTenTestsPercentBefore) {
            upOrDown = "Sie haben sich verschlechtert.";
        }
        else {
            upOrDown = "Sie waren genauso gut wie vorher.";
        }

        String labelText = "<html>"
                + "<br/>"
                + testedVocables
                + "<br/>"
                + actualSuccessPercent
                + "<br/>"
                + upOrDown
                + "<br/>"
                + "<br/>"
                + lastTenSuccessBefore
                + "<br/>"
                + lastTenSuccessAfter
                + "</html>";

        JLabel label = new JLabel(labelText);
        GuiTools.biggerFont(label, 5);
        label.setOpaque(true);
        label.setBackground(backgroundColor);
        return label;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createQuitButton() {
        quitButton.addActionListener(e -> closeDialog());
        quitButton.setBackground(backgroundColor);
        return quitButton;
    }

}
