package de.duehl.vocabulary.japanese.ui.dialog.update.bar;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.FlowLayout;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;

/**
 * Diese Klasse stellt ein Element für eine Gruppe des Dialogs, in dem der Benutzer die
 * Einstellung seiner Interessen vornehmen und eine oder mehrere Gruppen von Liste herunterladen
 * kann.
 *
 *   +----------------+------------------+----------+---------------+------------+
 *   | Listengruppe   |  Listenversion   |   neu?   |    Interesse  |  Download  |
 *   +----------------+------------------+----------+---------------+------------+
 *   | FROM_ZERO_1    |  35              |   nein   |    nein       |    [ ]     |
 *   | FROM_ZERO_2    |  17              |   nein   |    nein       |    [ ]     |
 *   | FROM_ZERO_3    |  3               |   nein   |    nein       |    [ ]     |
 *   | SCHRITT_FUER_1 |  4               |   nein   |    nein       |    [ ]     |
 *   | MARUGOTO_A1    |  7               |   ja     |    ja         |    [x]     |
 *   | VHS_A1_1       |  11              |   nein   |    ja         |    [ ]     |
 *   | VHS_A1_2       |  1               |   ja     |    ja         |    [x]     |
 *   +----------------+------------------+----------+---------------+------------+
 *
 * Angezeigt wird statt des Kürzels der Gruppe natürlich die im Enum hinterlegte
 * Beschreibung. Vielleicht hinterlege ich da auch noch eine noch kürzere Version.
 * Siehe de.duehl.vocabulary.japanese.logic.ownlists.OwnListGroup.
 * In den Entries ist das Kürzel gespeichert, also ("FROM_ZERO_1").
 *
 * Falls die Liste der neuen oder aktualisierten Gruppen leer ist, wird der Teil zur Anzeige der
 * Neuerungen nicht mit angezeigt.
 *
 * @version 1.01     2024-10-12
 * @author Christian Dühl
 */

public class GroupBar {

    /** Der Gruppenname ("FROM_ZERO_1"). */
    private final String groupName;

    /**
     * Die Beschreibung der Gruppe ("Japanese From Zero - Buch 1") wie sie in der Klasse
     * OwnListGroup hinterlegt ist.
     */
    private final String groupDescription;

    /** Die Nummer der Version ("17"). */
    private final int version;

    /** Gibt an, ob die Gruppe neu ist oder aktualisiert wurde. */
    private final boolean isNewOrActualised;

    /** Gibt an, ob angezeigt werden soll, ob Gruppen neu sind. */
    private final boolean showNewOrActualised;

    /** Gibt an, ob der Benutzer beim Start des Dialogs Interesse an dieser Gruppe hat. */
    private final boolean initialInterestedId;

    /** Der Panel auf dem die Gruppe dargestellt wird. */
    private final JPanel panel;

    private final JCheckBox newOrActualisedCheckbox;
    private final JCheckBox interestedCheckbox;
    private final JCheckBox downloadCheckbox;

    /**
     * Konstruktor.
     *
     * @param groupName
     *            Der Gruppenname ("FROM_ZERO_1").
     * @param groupDescription
     *            Die Beschreibung der Gruppe ("Japanese From Zero - Buch 1") wie sie in der Klasse
     *            OwnListGroup hinterlegt ist.
     * @param version
     *            Die Nummer der Version ("17").
     * @param isNewOrActualised
     *            Gibt an, ob die Gruppe neu ist oder aktualisiert wurde.
     * @param showNewOrActualised
     *            Gibt an, ob angezeigt werden soll, ob Gruppen neu sind.
     * @param initialInterestedId
     *            Gibt an, ob der Benutzer beim Start des Dialogs Interesse an dieser Gruppe hat.
     */
    public GroupBar(String groupName, String groupDescription, int version,
            boolean isNewOrActualised, boolean showNewOrActualised, boolean initialInterestedId) {
        this.groupName = groupName;
        this.groupDescription = groupDescription;
        this.version = version;
        this.isNewOrActualised = isNewOrActualised;
        this.showNewOrActualised = showNewOrActualised;
        this.initialInterestedId = initialInterestedId;

        panel = new JPanel();
        newOrActualisedCheckbox = new JCheckBox("Neu oder aktualisiert");
        interestedCheckbox = new JCheckBox("Interesse");
        downloadCheckbox = new JCheckBox("Download");

        initElements();
        createGui();
    }

    private void initElements() {
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        newOrActualisedCheckbox.setSelected(isNewOrActualised);
        newOrActualisedCheckbox.setEnabled(false);

        interestedCheckbox.setSelected(initialInterestedId);

        downloadCheckbox.setSelected(initialInterestedId && isNewOrActualised);
    }

    private void createGui() {
        panel.add(createNameAndVersionPart(), BorderLayout.CENTER);
        panel.add(createNewInterestedAnDownloadPart(), BorderLayout.EAST);
    }

    private Component createNameAndVersionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new FlowLayout(FlowLayout.LEFT, 15, 0));

        JLabel label1 = new JLabel(groupDescription);
        GuiTools.biggerFont(label1, 7);
        panel.add(label1);

        JLabel label2 = new JLabel("Version " + version);
        GuiTools.biggerFont(label1, -2);
        panel.add(label2);

        if (isNewOrActualised && showNewOrActualised) {
            label1.setForeground(Color.RED);
            label2.setForeground(Color.RED);
        }

        return panel;
    }

    private Component createNewInterestedAnDownloadPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new FlowLayout(FlowLayout.LEFT, 10, 0));

        if (showNewOrActualised) {
            //panel.add(newOrActualisedCheckbox); // Will ich in beiden Fällen nicht ...
        }
        panel.add(interestedCheckbox);
        panel.add(downloadCheckbox);

        return panel;
    }

    /** Getter für den Panel auf dem die Gruppe dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

    /** Getter für den Gruppennamen ("FROM_ZERO_1"). */
    public String getGroupName() {
        return groupName;
    }

    /** Gibt an, ob der Benutzer interessiert ist. */
    public boolean isInterested() {
        return interestedCheckbox.isSelected();
    }

    /** Gibt an, ob der Benutzer die Gruppe herunterladen und importieren möchte. */
    public boolean download() {
        return downloadCheckbox.isSelected();
    }

}
